/*
 * The MIT License

   BLDConograph (Bravais lattice determination module in Conograph)

Copyright (c) <2012> <Ryoko Oishi-Tomiyasu, KEK>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 *
 */
#ifdef _OPENMP
 # include <omp.h>
#endif
#include "utility_data_structure/index_set.hh"
#include "utility_func/transform_sym_matrix.hh"
#include "utility_lattice_reduction/super_basis3.hh"
#include "utility_lattice_reduction/put_Minkowski_reduced_lattice.hh"
#include "lattice_symmetry/LatticeFigureOfMeritToCheckSymmetry.hh"
#include "lattice_symmetry/ReducedLatticeToCheckBravais.hh"
#include "zerror_type/error_out.hh"
#include "ControlParam.hh"
#include "SortingLattice.hh"

const bool SortingLattice::m_DoesPrudentSymSearch = false;

SortingLattice::SortingLattice()
{
	for(ArrayIndex i=0; i<NUM_LS; i++)
	{
		OutputSymmetry[i] = false;
		JudgeSymmetry[i] = false;
	}
	
   	m_resol = 0.0;
}


SortingLattice::~SortingLattice()
{
}


// Set the member variables.
void SortingLattice::setParam(const ControlParam& cont) 
{
	OutputSymmetry[(Int4)Triclinic] = cont.putOutputSymmetry(Triclinic);
	JudgeSymmetry[(Int4)Triclinic] = false;
	for(ArrayIndex i=1; i<NUM_LS; i++)
	{
		OutputSymmetry[i] = cont.putOutputSymmetry(eCrystalSystem(i));
		JudgeSymmetry[i] = cont.putOutputSymmetry(eCrystalSystem(i));
	}

	if( JudgeSymmetry[(Int4)Cubic_P] )
	{
		JudgeSymmetry[(Int4)Tetragonal_P] = true;
	}
	if( JudgeSymmetry[(Int4)Hexagonal] )
	{
		JudgeSymmetry[(Int4)Monoclinic_P] = true;
	}
	if( JudgeSymmetry[(Int4)Tetragonal_P] )
	{
		JudgeSymmetry[(Int4)Orthorhombic_P] = true;
	}
	if( JudgeSymmetry[(Int4)Orthorhombic_P] )
	{
		JudgeSymmetry[(Int4)Monoclinic_P] = true;
	}
	
	if( JudgeSymmetry[(Int4)Orthorhombic_C] )
	{
		JudgeSymmetry[(Int4)Monoclinic_B] = true;
	}

	if( JudgeSymmetry[(Int4)Cubic_I] )
	{
		JudgeSymmetry[(Int4)Tetragonal_I] = true;
	}
	if( JudgeSymmetry[(Int4)Tetragonal_I] )
	{
		JudgeSymmetry[(Int4)Orthorhombic_I] = true;
	}

	if( JudgeSymmetry[(Int4)Cubic_F] )
	{
		JudgeSymmetry[(Int4)Orthorhombic_F] = true;
	}
	
	m_resol = cont.putResolution();
}


void SortingLattice::putLatticeFigureOfMerit(const LatticeFigureOfMeritToCheckSymmetry& lattice_original,
		const ePointGroup& epg, const Double& resol,
		vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result) const
{
	lattice_result.clear();
	map< SymMat<Double>, NRMat<Int4> > S_red_tray;
	if( !lattice_original.checkLatticeSymmetry(epg, resol, S_red_tray) ) return;

	const BravaisType& ebrat_original = lattice_original.putLatticeFigureOfMerit().putBravaisType();
	const eBravaisLattice eblat = (ebrat_original.enumCrystalSystem()==Monoclinic_B?
									(epg==D31d_rho?Prim:(epg==D3d_1_hex?Rhom_hex:BaseZ)):ebrat_original.enumBravaisLattice());

	const NRMat<Int4> matrix_min_to_sell = lattice_original.putInitialForm().second;

	SymMat<Double> S_super(4);
	Int4 itnum;
	NRMat<Int4> trans_mat(4,3);
	EquivLattice ans;
	
	for(map< SymMat<Double>, NRMat<Int4> >::const_iterator it=S_red_tray.begin(); it!=S_red_tray.end(); it++)
	{
		// S_super = it->second * it->first * Transpose(it->second) is close to
		// Delone-reduced form of the original lattice.
		S_super = transform_sym_matrix(it->second, it->first );

		trans_mat = identity_matrix<Int4>(4);

		// S_super = trans_mat * it->second * it->first * Transpose(trans_mat * it->second).
		put_super_Gram_matrix_obtuse_angle< Double, SymMat<Double> >(trans_mat, S_super, itnum);
		moveSmallerDiagonalLeftUpper< Double, SymMat<Double> >(S_super, trans_mat);
	
		lattice_result.push_back( LatticeFigureOfMeritToCheckSymmetry( BravaisType( pair<eBravaisLattice, ePointGroup>(eblat, epg) ),
										SymMat43_Double(it->first, mprod(trans_mat, it->second) ) ) );

		ans.label = lattice_original.putLatticeLabel();
		ans.transMat = Inverse3( put_transform_matrix_size4to3(trans_mat) );

		lattice_result.rbegin()->setEquivalentLatticeWithOtherCrystalSystem(ebrat_original.enumCrystalSystem(), ans);

		for(Int4 i=(Int4)ebrat_original.enumCrystalSystem()-1; i>=0; i--)
		{
			const vector<EquivLattice>& lattice_equiv0 = lattice_original.putEquivalentLatticeWithOtherCrystalSystem(eCrystalSystem(i));
			if( lattice_equiv0.empty() ) continue;
			assert( lattice_equiv0.size() == 1 );
			ans.label = lattice_equiv0.begin()->label;
			ans.transMat = mprod( lattice_equiv0.begin()->transMat, ans.transMat );

			lattice_result.rbegin()->setEquivalentLatticeWithOtherCrystalSystem(eCrystalSystem(i), ans);
		}
	}
}


void SortingLattice::putBravaisLatticeFigureOfMerit(const ReducedLatticeToCheckBravais& RLCB,
		const lattice_label& label_tri,
		const BravaisType& brat,
		vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result) const
{
	lattice_result.clear();
	
	const map< SymMat<Double>, NRMat<Int4> >& S_red_tray = RLCB.checkBravaisLatticeType(brat);
	if( S_red_tray.empty() ) return;

	// The lattice of RLCB has at least the symmetry given by eblat.
	SymMat<Double> S_super(4);
	Int4 itnum;
	NRMat<Int4> trans_mat(4,3);

	EquivLattice ans;
	ans.label = label_tri;

	for(map< SymMat<Double>, NRMat<Int4> >::const_iterator it=S_red_tray.begin(); it!=S_red_tray.end(); it++)
	{
		S_super = transform_sym_matrix(it->second, it->first);
		trans_mat = identity_matrix<Int4>(4);

		// S_super = trans_mat * it->second * it->first * Transpose(trans_mat * it->second) is Delone reduced.
		if( !put_super_Gram_matrix_obtuse_angle< Double, SymMat<Double> >(trans_mat, S_super, itnum) )
		{
			assert( false );
		}
		moveSmallerDiagonalLeftUpper< Double, SymMat<Double> >(S_super, trans_mat);

		lattice_result.push_back( LatticeFigureOfMeritToCheckSymmetry( brat, SymMat43_Double(it->first, mprod(trans_mat, it->second) ) ) );

		ans.transMat = Inverse3( put_transform_matrix_size4to3(trans_mat) );
		lattice_result.rbegin()->setEquivalentLatticeWithOtherCrystalSystem( Triclinic, ans );
	}
}



static SymMat43_Double putTransformMatrixFromSuperBasisObtuseToMinkowskiReduced(
		const SymMat<Double>& S_super_obtuse)
{
	NRMat<Int4> trans_mat(4,3);

	// S_red = trans_mat * S_super_obtuse * transpose(trans_mat).
	SymMat<Double> S_red(3);
	putMinkowskiReduced<Double, SymMat<Double> >(S_super_obtuse, false, S_red, trans_mat);

	return SymMat43_Double( S_red, put_transform_matrix_row3to4( Inverse3( trans_mat ) ) );
}



void SortingLattice::putLatticeCandidatesForTriclinic(const SymMat<Double>& S_super,
		vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_tri) const
{
	const SymMat43_Double S_red = putTransformMatrixFromSuperBasisObtuseToMinkowskiReduced(S_super);
	LatticeFigureOfMeritToCheckSymmetry latFOM(BravaisType( pair<eBravaisLattice, ePointGroup>(Prim, Ci) ), S_red);

	lattice_result_tri.clear();
	lattice_result_tri.push_back( latFOM );
}


void SortingLattice::putLatticeCandidatesForEachBravaisTypes(
					const eABCaxis& abc_axis,
					const eRHaxis& rh_axis,
					vector<LatticeFigureOfMeritToCheckSymmetry> lattice_result[NUM_LS]) const
{
	try{

		for(ArrayIndex i=1; i<NUM_LS; i++)
	{
		lattice_result[i].clear();
	}
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_tri = lattice_result[(ArrayIndex)Triclinic];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_mono_P = lattice_result[(ArrayIndex)Monoclinic_P];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_mono_B = lattice_result[(ArrayIndex)Monoclinic_B];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_ortho_P = lattice_result[(ArrayIndex)Orthorhombic_P];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_ortho_B = lattice_result[(ArrayIndex)Orthorhombic_C];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_ortho_I = lattice_result[(ArrayIndex)Orthorhombic_I];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_ortho_F = lattice_result[(ArrayIndex)Orthorhombic_F];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_tetra_P = lattice_result[(ArrayIndex)Tetragonal_P];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_tetra_I = lattice_result[(ArrayIndex)Tetragonal_I];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_rhom = lattice_result[(ArrayIndex)Rhombohedral];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_hex = lattice_result[(ArrayIndex)Hexagonal];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_cubic_P = lattice_result[(ArrayIndex)Cubic_P];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_cubic_I = lattice_result[(ArrayIndex)Cubic_I];
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_cubic_F = lattice_result[(ArrayIndex)Cubic_F];

	const Int4 num_tri = lattice_result_tri.size();

#ifdef _OPENMP
	#pragma omp parallel
#endif
	{
		vector<LatticeFigureOfMeritToCheckSymmetry> latFOM_tray;

#ifdef _OPENMP
		#pragma omp for
#endif
		for(Int4 n=0; n<num_tri; n++)
	   	{
			LatticeFigureOfMeritToCheckSymmetry& latFOM = lattice_result_tri[n];
			latFOM.setLabel(n+1);
			const SymMat43_Double& S_red = latFOM.putInitialForm();

			// Calculate figures of merit as triclinic
			const ReducedLatticeToCheckBravais RLCB(abc_axis, rh_axis, m_DoesPrudentSymSearch, m_resol, S_red);

			if( JudgeSymmetry[Monoclinic_B] )
			{
				putBravaisLatticeFigureOfMerit(RLCB, latFOM.putLatticeLabel(), BravaisType( put_monoclinic_b_type(abc_axis) ), latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical(monoB)
#endif
				{
					lattice_result_mono_B.insert(lattice_result_mono_B.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
			if( JudgeSymmetry[Orthorhombic_I] )
			{
				putBravaisLatticeFigureOfMerit(RLCB, latFOM.putLatticeLabel(), BravaisType( pair<eBravaisLattice, ePointGroup>(Inner, D2h) ), latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical(orthoI)
#endif
				{
					lattice_result_ortho_I.insert(lattice_result_ortho_I.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
			if( JudgeSymmetry[Orthorhombic_F] )
			{
				putBravaisLatticeFigureOfMerit(RLCB, latFOM.putLatticeLabel(), BravaisType( pair<eBravaisLattice, ePointGroup>(Face, D2h) ), latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical(orthoF)
#endif
				{
					lattice_result_ortho_F.insert(lattice_result_ortho_F.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
			if( JudgeSymmetry[Rhombohedral] )
			{
				putBravaisLatticeFigureOfMerit(RLCB, latFOM.putLatticeLabel(), BravaisType( put_rhombohedral_type(rh_axis) ), latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical(rhom)
#endif
				{
					lattice_result_rhom.insert(lattice_result_rhom.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
			if( JudgeSymmetry[Monoclinic_P] )
			{
				putLatticeFigureOfMerit(latFOM, put_monoclinic_p_type(abc_axis), m_resol, latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical(monoP)
#endif
				{
					lattice_result_mono_P.insert(lattice_result_mono_P.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
			if( JudgeSymmetry[Orthorhombic_P] )
			{
				putLatticeFigureOfMerit(latFOM, D2h, m_resol, latFOM_tray);
#ifdef _OPENMP
				#pragma omp critical (ortho_P)
#endif
				{
					lattice_result_ortho_P.insert(lattice_result_ortho_P.end(), latFOM_tray.begin(), latFOM_tray.end());
				}
			}
	   	}
	}

	for(ArrayIndex i=1; i<NUM_LS; i++)
	{
		if( !JudgeSymmetry[i] ) continue;
		sort( lattice_result[i].begin(), lattice_result[i].end() );

		const Int4 num_lattice = lattice_result[i].size();

#ifdef _OPENMP
		#pragma omp parallel
#endif
		{
			vector< VecDat3<Int4> > closest_hkl_tray;
			vector<bool> is_cal_Q_observed_tray;
//			vector<QData> qdata_modified;
			vector<LatticeFigureOfMeritToCheckSymmetry> latFOM_tray;

#ifdef _OPENMP
			#pragma omp for
#endif
			for(Int4 index=0; index<num_lattice; index++)
			{
				LatticeFigureOfMeritToCheckSymmetry& latFOM0 = lattice_result[i][index];
				latFOM0.setLabel(index+1);

				if( eCrystalSystem(i) == Monoclinic_P )
				{
					if( JudgeSymmetry[Hexagonal] )
					{
						putLatticeFigureOfMerit(latFOM0, D6h, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (hex)
#endif
						{
							lattice_result_hex.insert(lattice_result_hex.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
				}
				else if( eCrystalSystem(i) == Monoclinic_B )
				{
					if( JudgeSymmetry[Orthorhombic_C] )
					{
						putLatticeFigureOfMerit(latFOM0, D2h, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (ortho_B)
#endif
						{
							lattice_result_ortho_B.insert(lattice_result_ortho_B.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
				}
				else if( eCrystalSystem(i) == Orthorhombic_P )
				{
					if( JudgeSymmetry[Tetragonal_P] )
					{
						putLatticeFigureOfMerit(latFOM0, D4h_Z, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (tetra_P)
#endif
						{
							lattice_result_tetra_P.insert(lattice_result_tetra_P.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
					if( JudgeSymmetry[Cubic_P] )
					{
						putLatticeFigureOfMerit(latFOM0, Oh, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (cubic_P)
#endif
						{
							lattice_result_cubic_P.insert(lattice_result_cubic_P.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
				}
				else if( eCrystalSystem(i) == Orthorhombic_I )
				{
					if( JudgeSymmetry[Tetragonal_I] )
					{
						putLatticeFigureOfMerit(latFOM0, D4h_Z, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (tetra_I)
#endif
						{
							lattice_result_tetra_I.insert(lattice_result_tetra_I.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
					if( JudgeSymmetry[Cubic_I] )
					{
						putLatticeFigureOfMerit(latFOM0, Oh, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (cubic_I)
#endif
						{
							lattice_result_cubic_I.insert(lattice_result_cubic_I.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
				}
				else if( eCrystalSystem(i) == Orthorhombic_F )
				{
					if( JudgeSymmetry[Cubic_F] )
					{
						putLatticeFigureOfMerit(latFOM0, Oh, m_resol, latFOM_tray);
#ifdef _OPENMP
						#pragma omp critical (cubic_F)
#endif
						{
							lattice_result_cubic_F.insert(lattice_result_cubic_F.end(), latFOM_tray.begin(), latFOM_tray.end());
						}
					}
				}
			}
		}

ZLOG_INFO( "(" + num2str( i+1 ) + ") The number of candidates for " + put_cs_name(eCrystalSystem(i), abc_axis)
			+ " : " + num2str<Int4>( lattice_result[i].size() ) + "\n" );
	}
ZLOG_INFO( "\n" );
    }
    catch(bad_alloc& ball){
    	throw nerror(ball, __FILE__, __LINE__, __FUNCTION__);
    }
}


void SortingLattice::putLatticeCandidatesForEachBravaisTypes(const SymMat<Double>& S_super,
					const eABCaxis& abc_axis,
					const eRHaxis& rh_axis,
					vector<LatticeFigureOfMeritToCheckSymmetry> lattice_result[NUM_LS]) const
{
	vector<LatticeFigureOfMeritToCheckSymmetry>& lattice_result_tri = lattice_result[(Int4)Triclinic];
	putLatticeCandidatesForTriclinic(S_super, lattice_result_tri);

ZLOG_INFO( "Determination of the Bravais type is being carried out...\n" );
	putLatticeCandidatesForEachBravaisTypes(abc_axis, rh_axis, lattice_result);

	for(ArrayIndex i=1; i<NUM_LS; i++)
	{
		for(vector<LatticeFigureOfMeritToCheckSymmetry>::iterator it=lattice_result[i].begin();
				it<lattice_result[i].end(); it++)
		{
			it->setDistance(S_super);
		}
		sort( lattice_result[i].begin(), lattice_result[i].end() );
	}
}
