/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.notes.store;

import java.io.*;
import java.text.*;
import java.util.*;

import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.eai.notes.user.NotesAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;

import lotus.domino.*;

public class NotesMailStoreImpl extends EAINotesBase implements IMailStore
{
	public static final IStoreID storeID = new SimpleStoreID(NotesMailStoreImpl.class.getName());

	public static final String storeName = NotesConstants.MAILSTORE_NAME;

	public static final String DEFAULT_NOTES_INBOX_NAME = "($Inbox)";

	public static final String DEFAULT_NOTES_SENTBOX_NAME = "($Sent)";

	private IFolderIndex inboxElement = new NotesFolderIndex(DEFAULT_NOTES_INBOX_NAME);

	private IFolderIndex sentboxElement = new NotesFolderIndex(DEFAULT_NOTES_SENTBOX_NAME);

	public String getName() throws Exception
	{
		return storeName;
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		NotesServiceInfo info = new NotesServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		String serverAddress2 = (String)h.get("server2");
		if (serverAddress!=null) info.setServerAddress(serverAddress);
		if (serverAddress2!=null) info.setServerAddress2(serverAddress2);
		return info;
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new NotesServiceInfo(storeID, storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteMail(acc, oid);
	}

	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getMailsByDate(acc, dest, new Date());
	}

	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getMailFolders(acc, dest);
	}

	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getMail(acc, dest, oid);
	}

	public IRecordObject[] search(IAccountData acc, ISearchFormula form) throws Exception
	{
		throw new UnsupportedOperationException("IStore.search is not yet supported.");
	}

	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date startDate) throws Exception
	{
		return getMailsByDate(acc, dest, startDate);
	}

	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		IObjectIndex rc = sendMail(acc, (IMailDTO) data);
		return rc;
	}

	protected static NotesMailStoreImpl ms_objNotesImpl = null;

	/**
	 * Constructor
	 */
	public NotesMailStoreImpl()
	{

	}

	/**
	 * Returns an instance of EAINotesMailImpl
	 *
	 * @return Instance of EAINotesMailImpl
	 */
	public static NotesMailStoreImpl getInstance()
	{
		ms_objNotesImpl = new NotesMailStoreImpl();
		return ms_objNotesImpl;
	}

	private HeaderDTO populateHeaderDTO(Document docObjlcl) throws ServerDown, DataNotFound
	{
		HeaderDTO l_headerDTOObj = new HeaderDTO();
		Vector<?> l_toListObj = null;
		Vector<?> l_ccListObj = null;
		Vector<?> l_postedDateObj = null;
		String l_strPriority = null;
		try {
			l_headerDTOObj.setHeaderSize(docObjlcl.getSize());
			l_headerDTOObj.setHeaderUIDL(docObjlcl.getUniversalID());
			l_headerDTOObj.setOid(new NotesObjectIndex(l_headerDTOObj.getHeaderUIDL()));
			l_toListObj = docObjlcl.getItemValue(NotesConstants.SENDTO);
			l_headerDTOObj.setHeaderTo(this.convertVectorToStringArray(l_toListObj));
			l_ccListObj = docObjlcl.getItemValue(NotesConstants.COPYTO);
			l_headerDTOObj.setHeaderCC(this.convertVectorToStringArray(l_ccListObj));
			l_headerDTOObj.setHeaderFrom(docObjlcl.getItemValueString(NotesConstants.FROM));
			l_headerDTOObj.setHeaderReplyTo(docObjlcl.getItemValueString(NotesConstants.REPLYTO));
			l_headerDTOObj.setHeaderSubject(docObjlcl.getItemValueString(NotesConstants.SUBJECT));
			l_strPriority = docObjlcl.getItemValueString(NotesConstants.PRIORITY);
			l_headerDTOObj.setHeaderPriority(this.getMailPriority(l_strPriority));
			l_postedDateObj = docObjlcl.getItemValue(NotesConstants.POSTEDDATE);
			if(l_postedDateObj.isEmpty()) {
				//do nothing.
			} else {
				DateTime t = (DateTime) l_postedDateObj.firstElement();
				Date d = t.toJavaDate();
				t.recycle();
				l_headerDTOObj.setHeaderSendDate(d);
			}
		}
		catch (NullPointerException l_nullExObj) {
			throw new DataNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		return l_headerDTOObj;
	}

	private int getMailPriority(String strPriority)
	{
		int l_intPriority = 0;
		/* If mail does'nt have priority assume it to Normal priority */
		if (strPriority == null) {
			l_intPriority = 2;
		}
		else if (strPriority.equalsIgnoreCase(NotesConstants.PRIORITY_HIGH)) {
			l_intPriority = 1;
		}
		else if (strPriority.equalsIgnoreCase(NotesConstants.PRIORITY_NORMAL)) {
			l_intPriority = 2;
		}
		else if (strPriority.equalsIgnoreCase(NotesConstants.PRIORITY_LOW)) {
			l_intPriority = 3;
		}
		return l_intPriority;
	}

	private String[] convertVectorToStringArray(Vector<?> vectorObj)
	{
		String[] l_outPutArray;
		int l_intSize = vectorObj.size();
		l_outPutArray = new String[l_intSize];
		for (int l_intIndex = 0; l_intIndex < l_intSize; l_intIndex++) {
			l_outPutArray[l_intIndex] = (String) vectorObj.elementAt(l_intIndex).toString();
		}
		return l_outPutArray;
	}

	private Vector<String> convertArrayToStringVector(Object[] strArray)
	{
		int l_intSize = strArray.length;
		Vector<String> l_outPutVector = new Vector<String>(l_intSize);

		for (int l_intIndex = 0; l_intIndex < l_intSize; l_intIndex++) {
			l_outPutVector.addElement(strArray[l_intIndex].toString());
		}
		return l_outPutVector;
	}

	@SuppressWarnings("unchecked")
	private IMailDTO populateMailDTO(Document docObjlcl) throws ServerDown, StoreNotFound, DataNotFound, HandleException
	{
		IMailDTO l_IMailDTOObj = new BasicMailDTO();
		HeaderDTO l_headerDTOObj = new HeaderDTO();
		Vector<?> l_toListObj = null;
		Vector<?> l_ccListObj = null;
		Vector l_postedDateObj = null;
		String l_strPriority = null;
		DateTime t = null;
		Date d = null;
		try {
			l_postedDateObj = docObjlcl.getItemValue(NotesConstants.POSTEDDATE);
			if(l_postedDateObj.isEmpty()) {
				//do nothing.
			} else {
				t = (DateTime)l_postedDateObj.firstElement();
				d = t.toJavaDate();
				t.recycle();
				l_headerDTOObj.setHeaderSendDate(d);
			}

			// XXX: mod start ->
			Vector<DateTime> mailDate = docObjlcl.getItemValue(NotesConstants.DELIVEDDATE);
			if (mailDate.size() == 0) {
				mailDate = docObjlcl.getItemValue(NotesConstants.POSTEDDATE);
			}
			if(mailDate.isEmpty()) {
				//do nothing.
			} else {
				t = mailDate.firstElement();
			}

//			t = (DateTime) docObjlcl.getItemValue(NotesConstants.DELIVEDDATE).firstElement();
			// XXX: mod end <-
			if(t!=null) {
				d = t.toJavaDate();
				t.recycle();
			}
			l_headerDTOObj.setHeaderReceivedDate(d);
			l_headerDTOObj.setHeaderSize(docObjlcl.getSize());
			l_headerDTOObj.setHeaderUIDL(docObjlcl.getUniversalID());
			l_headerDTOObj.setOid(new NotesObjectIndex(l_headerDTOObj.getHeaderUIDL()));

			l_toListObj = docObjlcl.getItemValue(NotesConstants.SENDTO);
			l_headerDTOObj.setHeaderTo(this.convertVectorToStringArray(l_toListObj));
			l_ccListObj = docObjlcl.getItemValue(NotesConstants.COPYTO);
			l_headerDTOObj.setHeaderCC(this.convertVectorToStringArray(l_ccListObj));
			l_headerDTOObj.setHeaderFrom(docObjlcl.getItemValueString(NotesConstants.FROM));
			l_headerDTOObj.setHeaderReplyTo(docObjlcl.getItemValueString(NotesConstants.REPLYTO));
			l_headerDTOObj.setHeaderSubject(docObjlcl.getItemValueString(NotesConstants.SUBJECT));
			l_strPriority = docObjlcl.getItemValueString(NotesConstants.PRIORITY);
			l_headerDTOObj.setHeaderPriority(this.getMailPriority(l_strPriority));

			l_IMailDTOObj.setHeaderDTO(l_headerDTOObj);
			l_IMailDTOObj.setOid(new NotesObjectIndex(l_headerDTOObj.getHeaderUIDL()));
			l_IMailDTOObj.setBody(docObjlcl.getItemValueString(NotesConstants.BODY));
			HashMap<String,Object> h = new HashMap<String,Object>();
			h.put("CommonUserName", getCommonUserNameFromSession());
			try {
				l_IMailDTOObj.setFieldSet(h);
			}
			catch (IllegalAccessException iae) {
				Log.error("failed setting FieldSet",iae);
			}
			getAttachments(docObjlcl, l_IMailDTOObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new DataNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesException) {
			throw new ServerDown(l_notesException.id
					+ l_notesException.getMessage());
		}
		catch (NoSuchElementException nsee) {
			throw new DataNotFound(nsee.getMessage());
		}
		return l_IMailDTOObj;
	}

	@SuppressWarnings("unchecked")
	private void getAttachments(Document docObjlcl, IMailDTO l_IMailDTOObj) throws ServerDown, StoreNotFound, DataNotFound
	{
		//Vector<EmbeddedObject> l_attachmentObj;
		EmbeddedObject l_embedObj = null;
		AttachmentDTO[] l_byteObjArray = null;
		try {
			if (docObjlcl.hasEmbedded()) {
				Vector<Item> v = docObjlcl.getItems();
				Iterator<Item> ite = v.iterator();
				Vector<EmbeddedObject> attv = new Vector<EmbeddedObject>();
				while (ite.hasNext()) {
					Item item = ite.next();
					if (item.getType() == Item.ATTACHMENT) {
						l_embedObj = docObjlcl.getAttachment(item.getValueString());
					if (l_embedObj.getType() == EmbeddedObject.EMBED_ATTACHMENT) {
							attv.add(l_embedObj);
						}
					}
				}

				if (attv.size() > 0) {
					int l_intIndex = 0;
					l_byteObjArray = new AttachmentDTO[attv.size()];
					Iterator<EmbeddedObject> attite = attv.iterator();
					while (attite.hasNext()) {
						l_embedObj = (EmbeddedObject)attite.next();
						String attachName = l_embedObj.getName();
						ByteArrayOutputStream l_byteArr = new ByteArrayOutputStream();
						BufferedInputStream l_bufInStr = new BufferedInputStream(l_embedObj.getInputStream());
						BufferedOutputStream l_bufOutStr = new BufferedOutputStream(l_byteArr);
						int l_intBytExistence;
						while ((l_intBytExistence = l_bufInStr.read()) != -1) {
							l_bufOutStr.write(l_intBytExistence);
						}
						l_bufOutStr.flush();
						l_byteArr.flush();
						l_byteObjArray[l_intIndex] = new AttachmentDTO();
						l_byteObjArray[l_intIndex].setFileName(attachName);
						l_byteObjArray[l_intIndex].setFileBody(l_byteArr.toByteArray());
						l_byteArr.close();
						l_bufOutStr.close();
						l_bufInStr.close();
						l_intIndex++;
					}
					l_IMailDTOObj.setAttachments(l_byteObjArray);
				}
			}
		}
		catch (NullPointerException l_nullExObj) {
			Log.error("error on getting attachment file", l_nullExObj);
			AttachmentDTO nullAttach = new AttachmentDTO();
			nullAttach.setFileName("*cannot get attached file*");
			l_byteObjArray = new AttachmentDTO[1];
			l_byteObjArray[0] = nullAttach;
			l_IMailDTOObj.setAttachments(l_byteObjArray);
		}
		catch (NotesException l_exceptionObj) {
			Log.error("error on getting Notes attachment file", l_exceptionObj);
			AttachmentDTO nullAttach = new AttachmentDTO();
			nullAttach.setFileName("*cannot get attached file*");
			l_byteObjArray = new AttachmentDTO[1];
			l_byteObjArray[0] = nullAttach;
			l_IMailDTOObj.setAttachments(l_byteObjArray);
		}
		catch (IOException l_iOException) {
			Log.error("error on getting attachment file", l_iOException);
			AttachmentDTO nullAttach = new AttachmentDTO();
			nullAttach.setFileName("*cannot get attached file*");
			l_byteObjArray = new AttachmentDTO[1];
			l_byteObjArray[0] = nullAttach;
			l_IMailDTOObj.setAttachments(l_byteObjArray);
		}
		finally {
			try {
				if (l_embedObj != null) {
					l_embedObj.recycle();
				}
			}
			catch (NotesException l_exceptionObj) {
				throw new ServerDown(l_exceptionObj.getMessage());
			}
		}
	}

	private Document populateDocument(IMailDTO IMailDTOobj) throws NoSuchRights, HandleException, IncorrectData
	{
		HeaderDTO l_headerDTOobj = IMailDTOobj.getHeader();
		Document l_docObj = null;
		try {
			l_docObj = createDocument();
			l_docObj.appendItemValue(NotesConstants.FORM, NotesConstants.MEMO);
			l_docObj.appendItemValue(NotesConstants.LOGO, NotesConstants.STANDARD_LOGO_5);

			if (l_headerDTOobj.getHeaderSubject() != null) l_docObj.replaceItemValue(NotesConstants.SUBJECT, l_headerDTOobj.getHeaderSubject());
			if (l_headerDTOobj.getHeaderTo() != null) l_docObj.replaceItemValue(NotesConstants.SENDTO, convertArrayToStringVector(l_headerDTOobj.getHeaderTo()));
			if (l_headerDTOobj.getHeaderCC() != null) l_docObj.replaceItemValue(NotesConstants.COPYTO, convertArrayToStringVector(l_headerDTOobj.getHeaderCC()));
			if (l_headerDTOobj.getHeaderBCC() != null) l_docObj.replaceItemValue(NotesConstants.BLINDCOPYTO, convertArrayToStringVector(l_headerDTOobj.getHeaderBCC()));
			if (l_headerDTOobj.getHeaderFrom() != null) l_docObj.replaceItemValue(NotesConstants.FROM, l_headerDTOobj.getHeaderFrom().toString());
			if (l_headerDTOobj.getHeaderReplyTo() != null) l_docObj.replaceItemValue(NotesConstants.REPLYTO, l_headerDTOobj.getHeaderReplyTo().toString());

			RichTextItem l_body = l_docObj.createRichTextItem(NotesConstants.BODY);
			if (IMailDTOobj.getBody() != null) l_body.appendText(IMailDTOobj.getBody());

			int l_numberOfAttachments = IMailDTOobj.getNumberOfAttachments();
			AttachmentDTO[] att = IMailDTOobj.getAttachments();
			for (int i=0; i <l_numberOfAttachments; i++) {
				File f = null;
				FileOutputStream fos = null;
				try {
					f = File.createTempFile("attachment","."+att[i].getFileName());
					fos = new FileOutputStream(f);
					fos.write(att[i].getFileBody());
					l_body.embedObject(EmbeddedObject.EMBED_ATTACHMENT,"",f.getAbsolutePath(),att[i].getFileName());
				}
				catch (IOException ioe) {
					Log.error("error on create temp file for attach.",ioe);
				}
				catch (NotesException ne) {
					Log.error("error on create temp file for attach.", ne);
				}
				finally {
					try {
						fos.close();
						f.delete();
					} catch (Exception e) {
					}
				}
			}

			if (l_headerDTOobj.getHeaderSendDate() != null) {
				l_docObj.replaceItemValue(NotesConstants.POSTEDDATE, super.createDateTime(l_headerDTOobj.getHeaderSendDate()));
			}
			else {
				DateTime created = l_docObj.getCreated();
				try {
					l_docObj.replaceItemValue(NotesConstants.POSTEDDATE, super.createDateTime(created.getLocalTime()));
				}
				catch (ParseException pe) {
					Log.error("", pe);
					throw new IncorrectData(pe.getMessage());
				}
			}
		}
		catch (NotesException l_notesExpObj) {
			throw new IncorrectData(l_notesExpObj.id + l_notesExpObj.text);
		}
		return l_docObj;
	}

	public FolderMetadata[] getMailFolders(IAccountData acc, ISearchDestination dest) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		FolderMetadata[] l_strFolderlist = new FolderMetadata[0];
		ArrayList<FolderMetadata> dst = new ArrayList<FolderMetadata>();
		Vector<View> l_viewsObj = null;
		View l_viewObj = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_viewsObj = getViews();
			/* Searches for the Folders in list of View */
			for (int l_intIndex = 0; l_intIndex < l_viewsObj.size(); l_intIndex++) {
				l_viewObj = l_viewsObj.elementAt(l_intIndex);
				if (l_viewObj.isFolder()) {
					FolderMetadata folder = new FolderMetadata();
					folder.setFolderName(l_viewObj.getName());
					folder.setOid(new NotesFolderIndex(l_viewObj.getName()));
					dst.add(folder);
				}
			}
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		finally {
			try {
				if (l_viewsObj != null) {
					l_viewObj.recycle();
				}
			}
			catch (Exception l_exceptionObj) {
				Log.error("", l_exceptionObj);
			}
			finally {
				super.recycleObjects();
			}
		}
		l_strFolderlist = dst.toArray(l_strFolderlist);
		return l_strFolderlist;
	}

	public IMailDTO[] getHeaders(IAccountData acc, ISearchDestination dest) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		DocumentCollection l_docCollectionObj = null;
		View l_viewObj = null;
		DateTime l_dateTimeStartObj = null;
		DateRange l_dateRangeObj = null;
		IMailDTO[] headers = null;
		IFolderIndex folder = dest.getFolder();

		initSession((NotesAccountData) acc);
		try {
			String l_offsetDays = prop.getProperty(NotesConstants.OFFSET_DAYS, "30");
			int l_intoffsetDays = Integer.parseInt(l_offsetDays);

			l_viewObj = getView(folder);
			l_dateRangeObj = createDateRange(NotesConstants.DATE_TODAY, l_intoffsetDays, false);
			l_docCollectionObj = l_viewObj.getAllDocumentsByKey((Object) l_dateRangeObj);
			headers = generateMailArray(l_docCollectionObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound((String) folder.getIndex()
					+ l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExObj) {
			throw new DataNotFound(l_notesExObj.id + l_notesExObj.text);
		}
		catch (Exception e) {/* must be solved */
		}
		finally {
			try {
				if (l_viewObj != null) {
					l_viewObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
				if (l_dateTimeStartObj != null) {
					l_dateTimeStartObj.recycle();
				}
				if (l_dateRangeObj != null) {
					l_dateRangeObj.recycle();
				}
			}
			catch (NotesException l_exceptionObj) {
				throw new ServerDown(l_exceptionObj.id + l_exceptionObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return headers;
	}

	protected HeaderDTO[] generateHeaderArray(DocumentCollection l_docCollectionObj) throws HandleException
	{
		if (l_docCollectionObj == null) {
			throw new HandleException("l_docCollectionObj nust not be null");
		}
		HeaderDTO l_headerDTOObj = new HeaderDTO();
		ArrayList<HeaderDTO> l_listHeaderObj = new ArrayList<HeaderDTO>();
		Document l_docObj = null;
		try {
			l_docObj = l_docCollectionObj.getFirstDocument();
			while (l_docObj != null) {
				Document rec_dummy = null;
				try {
					l_headerDTOObj = populateHeaderDTO(l_docObj);
					l_listHeaderObj.add(l_headerDTOObj);
				}
				catch (ClassCastException l_classCastExObj) {
					Log.info("Notes mail failed to cast. (UniversalID:" + l_docObj.getUniversalID()+")");
				}
				rec_dummy = l_docObj;
				l_docObj = l_docCollectionObj.getNextDocument(l_docObj);
				rec_dummy.recycle();
			}
		}
		catch (NotesException l_notesExObj) {
			throw new DataNotFound(l_notesExObj.id + l_notesExObj.text);
		}
		finally {
			if (l_docObj != null) {
				try {
					l_docObj.recycle();
					l_docObj = null;
				}
				catch (NotesException e) {
					Log.warn("notes object recycle failed.");
				}
			}
		}

		HeaderDTO l_obj[] = new HeaderDTO[1];
		l_obj = l_listHeaderObj.toArray(l_obj);

		return l_obj;
	}

	public IMailDTO[] getHeadersByDate(IAccountData acc, ISearchDestination dest, java.util.Date lastDate) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		IFolderIndex folder = dest.getFolder();
		View l_viewObj = null;
		ViewEntryCollection l_viewEntryCollection = null;
		IMailDTO[] headers = null;
		DateTime l_dateTimeObj = null;

		super.initSession((NotesAccountData) acc);
		try {
			l_viewObj = getView(folder);
			SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");
			l_dateTimeObj = super.createDateTime(df.format(lastDate));
			Log.debug("l_viewObj=" + l_viewObj + ";l_dateTimeObj = " + l_dateTimeObj);
			l_viewEntryCollection = l_viewObj.getAllEntriesByKey((Object) l_dateTimeObj);
			headers = exGenerateMailArray(l_viewEntryCollection);
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound((String) folder.getIndex() + l_nullExObj.getMessage());
		}
		catch (NotesException notesExObj) {
			throw new DataNotFound(notesExObj.id + notesExObj.text);
		}

		catch (ParseException pe) {
			Log.error("", pe);
			throw new IncorrectData(pe.getMessage());
		}
		finally {
			super.recycleObjects();
		}
		return headers;
	}

	public IMailDTO getMail(IAccountData acc, ISearchDestination dest, IObjectIndex strReferenceUID) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		IMailDTO l_IMailDTOObj = null;
		Document l_docObj = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_docObj = getDocumentByUNID(strReferenceUID);
			l_IMailDTOObj = populateMailDTO(l_docObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new DataNotFound(l_nullExObj.getMessage());
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException l_notesExcepObj) {
				throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return l_IMailDTOObj;
	}

	public IObjectIndex[] getUIDLsByDate(IAccountData acc, ISearchDestination dest, Date dateObj) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		IFolderIndex folder = dest.getFolder();
		;
		View l_viewObj = null;
		DateRange l_dateRangeObj = null;
		DocumentCollection l_docCollectionObj = null;
		IObjectIndex l_strUidl[] = null;
		Date tmpDate = dateObj;
		SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");
		SimpleDateFormat df2 = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
		String tmpStr = df.format(dateObj) + " 00:00:00";
		try {
			tmpDate = df2.parse(tmpStr);
		}
		catch (ParseException e) {
		}
		;
		Date startDate = tmpDate;
		tmpStr = df.format(dateObj) + " 23:59:59";
		try {
			tmpDate = df2.parse(tmpStr);
		}
		catch (ParseException e) {
		}
		;
		Date endDate = tmpDate;
		super.initSession((NotesAccountData) acc);
		try {
			l_viewObj = getView(folder);
			l_dateRangeObj = super.createDateRange(startDate, endDate);
			l_docCollectionObj = l_viewObj.getAllDocumentsByKey((Object) l_dateRangeObj);
			l_strUidl = generateUIDLArray(l_docCollectionObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound((String) folder.getIndex()
					+ l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		finally {
			try {
				if (l_viewObj != null) {
					l_viewObj.recycle();
				}
				if (l_dateRangeObj != null) {
					l_dateRangeObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
			}
			catch (NotesException l_excepObj) {
				throw new ServerDown(l_excepObj.id + l_excepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return l_strUidl;
	}

	protected IObjectIndex[] generateUIDLArray(DocumentCollection l_docCollectionObj) throws HandleException
	{
		if (l_docCollectionObj == null) {
			throw new HandleException("l_docCollectionObj nust not be null");
		}
		BasicObjectIndex l_strUidl[] = null;
		Document l_docObj = null;
		try {
			int l_intMailCount = l_docCollectionObj.getCount();
			l_strUidl = new BasicObjectIndex[l_intMailCount];
			l_docObj = l_docCollectionObj.getFirstDocument();
			int l_intIndex = 0;
			while (l_docObj != null) {
				Document rec_dummy = null;
				String unid = l_docObj.getUniversalID();
				l_strUidl[l_intIndex] = new BasicObjectIndex(unid);
				HashMap<String,Object> h = new HashMap<String,Object>();
				h.put("UNID", unid);
				try {
					l_strUidl[l_intIndex].setFieldSet(h);
				}
				catch (Exception e) {
					Log.error("Exception on BasicRecordObject.setFieldSet.", e);
				}
				l_intIndex += 1;
				rec_dummy = l_docObj;
				l_docObj = l_docCollectionObj.getNextDocument(l_docObj);
				rec_dummy.recycle();
			}
		}
		catch (NotesException l_notesExObj) {
			throw new DataNotFound(l_notesExObj.id + l_notesExObj.text);
		}
		finally {
			if (l_docObj != null) {
				try {
					l_docObj.recycle();
					l_docObj = null;
				}
				catch (NotesException e) {
					// WORN level
				}
			}
		}
		return l_strUidl;
	}

	public IObjectIndex[] getUIDLs(IAccountData acc, ISearchDestination dest) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		IFolderIndex folder = null;
		if (dest != null) {
			folder = dest.getFolder();
		}
		View l_viewObj = null;
		DateTime l_dateTimeStartObj = null;
		DateRange l_dateRangeObj = null;
		DocumentCollection l_docCollectionObj = null;
		IObjectIndex[] l_strUIDL = null;

		super.initSession((NotesAccountData) acc);
		try {
			String l_offsetDays = prop.getProperty(NotesConstants.OFFSET_DAYS, "30");
			int l_intoffsetDays = Integer.parseInt(l_offsetDays);
			l_dateRangeObj = createDateRange(NotesConstants.DATE_TODAY, l_intoffsetDays, false);
			l_viewObj = getView(folder);
			l_docCollectionObj = l_viewObj.getAllDocumentsByKey((Object) l_dateRangeObj);

			l_strUIDL = generateUIDLArray(l_docCollectionObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound((String) folder.getIndex()
					+ l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			Log.error("", l_notesExcepObj);
		}
		catch (Exception e) {/* must be solved */
		}
		finally {
			try {
				if (l_viewObj != null) {
					l_viewObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
				if (l_dateTimeStartObj != null) {
					l_dateTimeStartObj.recycle();
				}
				if (l_dateRangeObj != null) {
					l_dateRangeObj.recycle();
				}
			}
			catch (NotesException l_excepObj) {
				throw new ServerDown(l_excepObj.id + l_excepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return l_strUIDL;
	}

	protected IMailDTO[] generateMailArray(DocumentCollection l_docCollectionObj) throws HandleException, DataNotFound
	{
		if (l_docCollectionObj == null) {
			throw new HandleException("l_docCollectionObj must not be null");
		}

		IMailDTO l_mailArray[] = new IMailDTO[0];
		IMailDTO l_IMailDTOObj = null;
		ArrayList<IMailDTO> l_listMailObj = new ArrayList<IMailDTO>();
		Document l_docObj = null;

		try {
			l_docObj = l_docCollectionObj.getFirstDocument();
			while (l_docObj != null) {
				Document rec_dummy = null;
				try {
					l_IMailDTOObj = populateMailDTO(l_docObj);
					l_listMailObj.add(l_IMailDTOObj);
				}
				catch (DataNotFound dnf) {
					// in case form not equals mail-form
					// do nothing for this element
				}
				catch (ClassCastException l_classCastExObj) {
					Log.info("Notes mail failed to cast. (UniversalID:" + l_docObj.getUniversalID()+")");
				}
				rec_dummy = l_docObj;
				l_docObj = l_docCollectionObj.getNextDocument(l_docObj);
				rec_dummy.recycle();
			}
		}
		catch (NotesException l_notesExObj) {
			throw new DataNotFound(l_notesExObj.id + l_notesExObj.text);
		}
		finally {
			if (l_docObj != null) {
				try {
					l_docObj.recycle();
					l_docObj = null;
				}
				catch (NotesException e) {
					// WARN level
				}
			}
		}

		if (l_listMailObj.size()>0) l_mailArray = l_listMailObj.toArray(l_mailArray);

		return l_mailArray;
	}

	public IMailDTO[] getMailsByDate(IAccountData acc, ISearchDestination dest, Date dateObj) throws StoreNotFound, NoSuchRights, ServerDown, DataNotFound, IncorrectStore, IncorrectData, HandleException
	{
		Log.debug("EAINotesMailImpl.getMailsByDate");
		IFolderIndex folder = dest.getFolder();
		View l_viewObj = null;
		DateTime l_dateTimeObj = null;
		DocumentCollection l_docCollectionObj = null;
		IMailDTO l_mailArray[] = null;
		super.initSession((NotesAccountData) acc);
		try {
			l_viewObj = getView(folder);
			Log.debug("l_viewObj=" + l_viewObj);
			l_dateTimeObj = super.createDateTime(dateObj);
			// To get a mails on a particular day, Date should be in
			// "YYYY/MM/DD"
			// format
			try {
				l_dateTimeObj = super.createDateTime(l_dateTimeObj.getDateOnly());
			}
			catch (ParseException pe) {
				Log.error("", pe);
				throw new IncorrectData(pe.getMessage());
			}
			Log.debug("l_viewObj=" + l_viewObj + ";l_dateTimeObj = "
					+ l_dateTimeObj);
			l_docCollectionObj = l_viewObj.getAllDocumentsByKey((Object) l_dateTimeObj);
			l_mailArray = generateMailArray(l_docCollectionObj);
		}
		catch (NullPointerException l_nullExObj) {
			throw new StoreNotFound((String) folder.getIndex()
					+ l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		finally {
			try {
				if (l_viewObj != null) {
					l_viewObj.recycle();
				}
				if (l_dateTimeObj != null) {
					l_dateTimeObj.recycle();
				}
				if (l_docCollectionObj != null) {
					l_docCollectionObj.recycle();
				}
			}
			catch (NotesException l_excepObj) {
				throw new ServerDown(l_excepObj.id + l_excepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return l_mailArray;
	}

	public IObjectIndex sendMail(IAccountData acc, IMailDTO IMailDTOobj) throws NoSuchRights, ServerDown, IncorrectData, HandleException
	{
		Document l_docObj = null;
		IObjectIndex rc = null;
		super.initSession((NotesAccountData) acc);

		try {
			l_docObj = populateDocument(IMailDTOobj);
			l_docObj.save();
			rc = new NotesObjectIndex(l_docObj.getUniversalID());
			l_docObj.send(l_docObj.getItemValue(NotesConstants.SENDTO));
		}
		catch (NotesException notesEx) {
			throw new HandleException(notesEx.id + notesEx.text);
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException l_notesExp) {
				new ServerDown(l_notesExp.id + l_notesExp.text);
			}
			finally {
				super.recycleObjects();
			}
		}
		return rc;
	}

	public IMailDTO[] getSentMails(IAccountData acc) throws ServerDown, HandleException
	{
		return getMailsByDate(acc, getPresetDestination(acc, DEST_TYPE_SENT_FOLDER), new Date());
	}

	public IMailDTO[] getSentMailsByDate(IAccountData acc, Date dateObj) throws ServerDown, HandleException
	{
		return getMailsByDate(acc, getPresetDestination(acc, DEST_TYPE_SENT_FOLDER), dateObj);
	}

	public void deleteMail(IAccountData acc, IObjectIndex strUIDL) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		Document l_docObj = null;
		Log.debug("delete mail " + strUIDL);
		super.initSession((NotesAccountData) acc);
		try {
			l_docObj = getDocumentByUNID(strUIDL);
			l_docObj.remove(true);
			Log.debug("deleted");
		}
		catch (NullPointerException l_nullExObj) {
			Log.debug("mail not found");
			throw new DataNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExObj) {
			throw new DeleteFailed();
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException l_notesExcepObj) {
				throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
			}
			finally {
				super.recycleObjects();
			}
		}
	}

	/*
	 * ( Javadoc)
	 *
	 * @see jp.co.connectone.store.IStore#getPresetDestination(jp.co.connectone.user.IAccountData,
	 *      int)
	 */
	public ISearchDestination getPresetDestination(IAccountData acc, int type) throws IncorrectData, HandleException
	{
		IDatabaseIndex db = null;
		IFolderIndex folder = null;
		switch (type) {
		case IMailStore.DEST_TYPE_DEFAULT_MAIL_FOLDER:
			folder = inboxElement;
			break;
		case IMailStore.DEST_TYPE_SENT_FOLDER:
			folder = sentboxElement;
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}

		return (ISearchDestination) new BasicSearchDestination(db, folder);
	}

	/*
	 * ( Javadoc)
	 *
	 * @see jp.co.connectone.eai.notes.store.EAINotesBase#getElmentName()
	 */
	protected String getElmentName()
	{
		return "($Inbox)";
	}

	@SuppressWarnings("unchecked")
	protected HashMap<String, Object> populateHashMap(Document doc) throws HandleException {

		HashMap<String, Object> h = new HashMap<String, Object>();
		try {
			Vector<Item> items = doc.getItems();
			Item item = null;
			String name = null;
			Vector<Item> values = null;
			Object value = null;
			for (int i = 0; i < items.size(); i++) {
				try {
					item = items.get(i);
					name = item.getName();
					values = item.getValues();
					if (values == null || values.size() <= 1) {
						value = item.getValueString();
					} else {
						value = convertVectorToStringArray(values);
					}
					h.put(name, value);
				} catch (Exception e2) {
					if (name == null) {
						name = "[couldn't get name]";
					}
					Log.trace("skip convert field " + name + " by error. message is " + e2.getMessage());
					name = null;
				}
			}
		} catch (NotesException e) {
			throw new HandleException(e.id + e.text);
		}

		return h;
	}

	protected IMailDTO[] exGenerateMailArray(ViewEntryCollection l_viewEntryCollection) throws HandleException, DataNotFound {

		if (l_viewEntryCollection == null) {
			throw new HandleException("l_viewEntryCollection must not be null");
		}
		IMailDTO l_mailArray[] = null;
		IMailDTO l_IMailDTOObj = null;
		HeaderDTO l_headerDTOObj = null;
		ArrayList<IMailDTO> l_listMailObj = new ArrayList<IMailDTO>();
		Document l_docObj = null;
		ViewEntry l_viewEntry = null;

		try {
			int collectionCount = l_viewEntryCollection.getCount();
			Log.debug("collectionCount=" + collectionCount);
			l_viewEntry = l_viewEntryCollection.getLastEntry();
			while (l_viewEntry != null) {
				try {
					l_IMailDTOObj = new BasicMailDTO();
					l_docObj = l_viewEntry.getDocument();
					try {
						l_headerDTOObj = exPopulateHeaderDTO(l_docObj);

						if (l_docObj.hasEmbedded()) {
							AttachmentDTO[] l_byteObjArray = new AttachmentDTO[1];
							l_byteObjArray[0] = new AttachmentDTO(); // dummy attach
							l_IMailDTOObj.setAttachments(l_byteObjArray);
						}
						l_IMailDTOObj.setHeaderDTO(l_headerDTOObj);
						l_IMailDTOObj.setOid(new NotesObjectIndex(l_headerDTOObj.getHeaderUIDL()));

						HashMap<String, Object> h = new HashMap<String, Object>();
						h.put("CommonUserName", getCommonUserNameFromSession());
						String form = l_docObj.getItemValueString("Form");
						if (form == null || form.length() == 0) {
							form = l_docObj.getItemValueString("FORM");
						}
						h.put("Form", form);
						try {
							l_IMailDTOObj.setFieldSet(h);
						}
						catch (IllegalAccessException iae) {
							Log.error("failed setting FieldSet", iae);
						}
						l_listMailObj.add(l_IMailDTOObj);
					} catch (ClassCastException l_classCastExObj) {
						Log.info("Notes mail failed to cast. (UniversalID:" + l_docObj.getUniversalID() + ")");
					}
				} catch (DataNotFound dnf) {
					// in case form not equals mail-form
					// do nothing for this element
					Log.debug("DataNotFound via exGenerateMailArray :\n" + dnf.getStackTraceString());
				}
				l_viewEntry = l_viewEntryCollection.getPrevEntry(l_viewEntry);
			}
		} catch (NotesException l_notesExObj) {
			throw new DataNotFound(l_notesExObj.id + l_notesExObj.text);
		}

		IMailDTO[] a = new IMailDTO[0];
		l_mailArray = l_listMailObj.toArray(a);
		return l_mailArray;
	}

	@SuppressWarnings("unchecked")
	private HeaderDTO exPopulateHeaderDTO(Document docObjlcl) throws ServerDown, DataNotFound {

		HeaderDTO l_headerDTOObj = new HeaderDTO();

		try {

			Vector<?> l_toListObj = docObjlcl.getItemValue(NotesConstants.SENDTO);
			l_headerDTOObj.setHeaderTo(this.convertVectorToStringArray(l_toListObj));
			l_headerDTOObj.setHeaderUIDL(docObjlcl.getUniversalID());
			l_headerDTOObj.setHeaderFrom(docObjlcl.getItemValueString(NotesConstants.FROM));
			l_headerDTOObj.setHeaderSubject(docObjlcl.getItemValueString(NotesConstants.SUBJECT));
			Vector l_postedDateObj = docObjlcl.getItemValue(NotesConstants.POSTEDDATE);
			DateTime t = null;
			Date d = null;
			if(l_postedDateObj.isEmpty()) {
				//do nothing.
			} else {
				t = (DateTime) l_postedDateObj.firstElement();
				d = t.toJavaDate();
				l_headerDTOObj.setHeaderSendDate(d);
			}

			// XXX: mod start ->
			Vector<DateTime> mailDate = docObjlcl.getItemValue(NotesConstants.DELIVEDDATE);
			if (mailDate.size() != 0) {
				t = mailDate.firstElement();
			}

			// XXX: mod end <-
			if(t!=null) {
				d = t.toJavaDate();
				l_headerDTOObj.setHeaderReceivedDate(d);
			}
		}
		catch (NullPointerException l_nullExObj) {
			throw new DataNotFound(l_nullExObj.getMessage());
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.id + l_notesExcepObj.text);
		}
		return l_headerDTOObj;
	}
}
