package jp.crestmuse.cmx.inference.game;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;

import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiSystem;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import jp.crestmuse.cmx.amusaj.filewrappers.BayesNetWrapper;
import jp.crestmuse.cmx.amusaj.sp.MidiEventWithTicktime;
import jp.crestmuse.cmx.amusaj.sp.MidiInputModule;
import jp.crestmuse.cmx.amusaj.sp.MidiOutputModule;
import jp.crestmuse.cmx.amusaj.sp.SPExecutor;
import jp.crestmuse.cmx.amusaj.sp.SPSpreadModule;
import jp.crestmuse.cmx.inference.AccompanimentGenerator;
import jp.crestmuse.cmx.inference.BayesianCalculator;
import jp.crestmuse.cmx.inference.BayesianMapping;
import jp.crestmuse.cmx.inference.Calculator;
import jp.crestmuse.cmx.inference.MelodyWriter;
import jp.crestmuse.cmx.inference.MusicRepresentation;
import jp.crestmuse.cmx.inference.NaiveVoicingCalculator;
import jp.crestmuse.cmx.inference.MusicRepresentation.MusicElement;
import jp.crestmuse.cmx.sound.SequencerManager;
import jp.crestmuse.cmx.sound.TickTimer;

public class GUI2 extends JPanel implements Runnable, Calculator {

  public final int KEYBOARD_WIDTH = 100;
  public final int MOVE_LIMIT = 10;
  private final boolean[] isWhiteKey = { true, false, true, false, true, true,
      false, true, false, true, false, true };
  private final int[] noteIndex2whiteKey = { 0, -1, 1, -1, 2, 3, -1, 4, -1, 5,
      -1, 6 };
  private MusicRepresentation mr;
  private TickTimer timer;
  private ColoredKey currentNote;
  private ColoredKey nextNote;
  private int currentX, currentY = 0;
  private int targetY;
  private Ripple[] ripples;

  public GUI2(MusicRepresentation musicRepresentation, TickTimer tickTimer) {
    mr = musicRepresentation;
    timer = tickTimer;
    mr.addCalculator("melody", this);
    currentNote = new ColoredKey(0, 255, 0);
    nextNote = new ColoredKey(255, 0, 0);
    setPreferredSize(new Dimension(1024, 420));
    setBackground(Color.BLACK);
    ripples = new Ripple[10];
    for (int i = 0; i < ripples.length; i++)
      ripples[i] = new Ripple();
    Thread t = new Thread(this);
    t.start();
  }

  public void run() {
    while (!Thread.interrupted()) {
      // update target
      int measureTick = timer.getTicksPerBeat() * 4;
      long tick = timer.getTickPosition();
      try {
        MusicElement nextChord = mr.getMusicElement("chord", mr.getIndex(tick
            + measureTick));
        if (nextChord.set())
          targetY = getHeight() * (6 - nextChord.getHighestProbIndex()) / 7;
      } catch (ArrayIndexOutOfBoundsException e) {
      }

      // aim target
      currentY += Math.max(Math.min((targetY - currentY), MOVE_LIMIT),
          -MOVE_LIMIT);

      // get x from mr and current tick
      currentX = (int) ((getWidth() - KEYBOARD_WIDTH)
          * (measureTick - tick % measureTick) / measureTick + KEYBOARD_WIDTH);

      // update ripples
      for (Ripple r : ripples)
        r.update();

      // keys
      currentNote.update();
      nextNote.update();

      repaint();
      try {
        Thread.sleep(10);
      } catch (InterruptedException e) {
        break;
      }
    }
  }

  public void paint(Graphics g) {
    super.paint(g);

    // keyboard
    g.setColor(Color.WHITE);
    g.fillRect(0, 0, KEYBOARD_WIDTH, getHeight());
    int whiteKeyHeight = (int) (getHeight() / 7.0);
    boolean nextInWhite = isWhiteKey[nextNote.index];
    boolean currentInWhite = isWhiteKey[currentNote.index];
    if(currentInWhite && currentNote.alive) {
      g.setColor(currentNote.color());
      g.fillRect(0, whiteKeyHeight * (6 - noteIndex2whiteKey[currentNote.index]),
          KEYBOARD_WIDTH, whiteKeyHeight);
    }
    if (nextInWhite && nextNote.alive) {
      g.setColor(nextNote.color());
      g.fillRect(0, whiteKeyHeight * (6 - noteIndex2whiteKey[nextNote.index]),
          KEYBOARD_WIDTH, whiteKeyHeight);
    }
    g.setColor(Color.BLACK);
    g.drawLine(KEYBOARD_WIDTH, 0, KEYBOARD_WIDTH, getHeight());
    for (int i = 0; i < 7; i++)
      g.drawLine(0, whiteKeyHeight * i, KEYBOARD_WIDTH, whiteKeyHeight * i);
    int blackKeyHeight = (int) (getHeight() / 12.0);
    for (int i = 0; i < 12; i++)
      if (!isWhiteKey[11 - i])
        g.fillRect(0, blackKeyHeight * i, KEYBOARD_WIDTH * 2 / 3,
            blackKeyHeight);
    if (!nextInWhite) {
      g.setColor(nextNote.color());
      g.fillRect(0, blackKeyHeight * (11 - nextNote.index),
          KEYBOARD_WIDTH * 2 / 3, blackKeyHeight);
    }

    // chord ball
    g.setColor(Color.RED);
    g.fillOval(currentX, currentY, whiteKeyHeight, whiteKeyHeight);

    // ripples
    for (Ripple r : ripples)
      r.draw(g);
  }

  public void update(MusicRepresentation musRep, MusicElement me, int index) {
//    int currentNoteIndex = me.getHighestProbIndex();
//    for (int i = 0; i < ripples.length; i++) {
//      if (ripples[i].alive)
//        continue;
//      int y;
//      if (isWhiteKey[currentNoteIndex]) {
//        y = getHeight() * (6 - noteIndex2whiteKey[currentNoteIndex]) / 7;
//      } else {
//        y = getHeight() * (11 - currentNoteIndex) / 12;
//      }
//      ripples[i].reset(KEYBOARD_WIDTH, y,
//          currentNoteIndex == nextNoteIndex);
//      break;
//    }
    currentNote.reset(0);
    currentNote.index = me.getHighestProbIndex();
    nextNote.reset(0);
    nextNote.index = musRep.getMusicElement("melody", index + 1)
        .getHighestProbIndex();
  }

  private class Ripple {
    int x, y, r;
    boolean circle;
    boolean alive;
    Color color;

    Ripple() {
      reset(0, 0, true);
      alive = false;
    }

    void reset(int x, int y, boolean isCircle) {
      this.x = x;
      this.y = y;
      r = 0;
      circle = isCircle;
      color = circle ? Color.GREEN : Color.RED;
      alive = true;
    }

    void update() {
      if (!alive)
        return;
      r += 5;
      if (r > getHeight())
        alive = false;
    }

    void draw(Graphics g) {
      if (!alive)
        return;
      g.setColor(color);
      if (circle)
        g.drawOval(x - r, y - r, r * 2, r * 2);
      else
        g.drawRect(x - r, y - r, r * 2, r * 2);
    }
  }

  private class ColoredKey {
    int index;
    int delay;
    int alpha;
    int red, green, blue;
    boolean alive = false;
    ColoredKey(int r, int g, int b) {
      red = r;
      green = g;
      blue = b;
      alpha = 255;
    }
    void reset(int delay) {
      this.delay = delay;
      alpha = 255;
      alive = false;
    }
    void update() {
//      if(!alive) return;
      if(delay > 0) {
        delay--;
        return;
      }
      alive = true;
      if(alpha <= 0) return;
      alpha -= 5;
    }
    Color color() {
      return new Color(red, green, blue, Math.max(alpha, 0));
    }
  }

  public static void main(String[] args) {
    try {
      int measureLength = 8;
      String modelFile = "contents/model.bif";
      String firstChord = "C";
      int inputDeviceIndex = 0;
      int outputDeviceIndex = 3;
      String chordMIDIFile = "contents/C.mid";

      final MusicRepresentation mr = new MusicRepresentation(measureLength, 8);
      mr.addMusicLayer("melody", 12);
      mr.addMusicLayer("chord", new String[] { "C", "Dm", "Em", "F", "G", "Am",
          "Bm(b5)" }, 8);
      mr.addMusicLayer("bass", 128, 2);
      mr.addMusicLayer("voicingHigh", 128, 8);
      mr.addMusicLayer("voicingMidHigh", 128, 8);
      mr.addMusicLayer("voicingMidLow", 128, 8);
      mr.addMusicLayer("voicingLow", 128, 8);

      BayesianCalculator bc = new BayesianCalculator(new BayesNetWrapper(
          modelFile));
      bc.addReadMapping(new BayesianMapping("melody", -1,
          BayesianMapping.SET_ONLY, 0));
      bc.addReadMapping(new BayesianMapping("chord", -1,
          BayesianMapping.BY_TIED_LENGTH, 1));
      bc.addReadMapping(new BayesianMapping("melody", 0,
          BayesianMapping.NORMAL, 2));
      bc.addReadMapping(new BayesianMapping("chord", 0, BayesianMapping.NORMAL,
          3));
      bc.addWriteMapping(new BayesianMapping("melody", 1,
          BayesianMapping.NORMAL, 4));
      bc.addWriteMapping(new BayesianMapping("chord", 1,
          BayesianMapping.BY_TIED_LENGTH, 5));

      mr.addCalculator("melody", bc);
      mr.addCalculator("chord", new NaiveVoicingCalculator());
      MusicElement chord = mr.getMusicElement("chord", 0);
      chord.setProb(chord.indexOf(firstChord), 1.0);
      mr.update("chord", 0);

      SPExecutor sp = new SPExecutor();
      MidiInputModule mi = new MidiInputModule(MidiSystem.getMidiDevice(MidiSystem.getMidiDeviceInfo()[inputDeviceIndex]));
//      MidiInputModule mi = new MidiInputModule(new VirtualKeyboard());
      MelodyWriter mw = new MelodyWriter(mr, false);
      MidiDevice device = MidiSystem.getMidiDevice(MidiSystem
          .getMidiDeviceInfo()[outputDeviceIndex]);
      device.open();
      MidiOutputModule mo = new MidiOutputModule(device.getReceiver());

      final SequencerManager sm = new SequencerManager(device.getReceiver());
      AccompanimentGenerator mrg = new AccompanimentGenerator(mr, chordMIDIFile);
      sm.addGeneratable(mrg);

      mi.setTickTimer(sm);
      SPSpreadModule towway = new SPSpreadModule(MidiEventWithTicktime.class, 2);
      sp.addSPModule(mi);
      sp.addSPModule(towway);
      sp.addSPModule(mo);
      sp.addSPModule(mw);
      sp.connect(mi, 0, towway, 0);
      sp.connect(towway, 0, mo, 0);
      sp.connect(towway, 1, mw, 0);

      SwingUtilities.invokeAndWait(new Runnable() {
        public void run() {
          JFrame f = new JFrame();
          f.add(new GUI2(mr, sm));
          f.pack();
          f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
          f.setVisible(true);
        }
      });

      sp.start();
      sm.start();

      // 終了
      System.in.read();
      sm.stop();
      sp.stop();
      device.close();
    } catch (Exception e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

}
