//
//  chnlib05.c
//  AI003
//
//  Created by 西田　耀 on 13/06/30.
//  Copyright (c) 2013年 Hikaru Nishida. All rights reserved.
//  A C-program for MT19937, modified by hikarupsp for CHNLIB.

// MT19937を利用した乱数生成器

/*
   A C-program for MT19937, with initialization improved 2002/1/26.
   Coded by Takuji Nishimura and Makoto Matsumoto.

   Before using, initialize the state by using init_genrand(seed)  
   or init_by_array(init_key, key_length).

   Copyright (C) 1997 - 2002, Makoto Matsumoto and Takuji Nishimura,
   All rights reserved.                          

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote 
        products derived from this software without specific prior written 
        permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


   Any feedback is very welcome.
   http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
   email: m-mat @ math.sci.hiroshima-u.ac.jp (remove space)
*/
/*
    Modified by hikarupsp on 2013/06/30 for CHNLIB.
*/

//
//Include headers
//
#include <stdio.h>
#include "chnlib.h"

//
//Define values
//
/* Period parameters */
#define N 624
#define M 397
#define MATRIX_A 0x9908b0dfUL   /* constant vector a */
#define UPPER_MASK 0x80000000UL /* most significant w-r bits */
#define LOWER_MASK 0x7fffffffUL /* least significant r bits */

//
//Declare internal functions
//
CHNLIB_RandomGeneratorMT *CHNLIB_RandomGeneratorMT_Internal_Allocate(void);
void CHNLIB_RandomGeneratorMT_Internal_Destruct(void **structure);
uint CHNLIB_RandomGeneratorMT_Internal_GetHash(const void *structure);

//
//Define types
//
struct CHNLIB_RANDOM_GENERATOR_MT {
    //MT19937を利用した乱数生成器
    CHNLIB_StructureHeader header;
    unsigned long mt[N]; /* the array for the state vector  */
    int mti; /* mti==N+1 means mt[N] is not initialized */
};

//
//Functions
//

CHNLIB_RandomGeneratorMT *CHNLIB_RandomGeneratorMT_Initialise(void)
{
    CHNLIB_RandomGeneratorMT *env;
    
    env = CHNLIB_RandomGeneratorMT_Internal_Allocate();
    env->header.destructor = &CHNLIB_RandomGeneratorMT_Internal_Destruct;
    env->header.getHash = &CHNLIB_RandomGeneratorMT_Internal_GetHash;
    env->mti = N + 1; /* mti==N+1 means mt[N] is not initialized */
    return env;
}

void CHNLIB_RandomGeneratorMT_Free(CHNLIB_RandomGeneratorMT *env)
{
    //envを解放する。
    if(CHNLIB_StructureHeader_GetTypeID(env) != CHNLIB_STRUCT_ID_RandomGeneratorMT){
        return;
    }
    
    env->header.typeid = CHNLIB_STRUCT_ID_Null;
    env->header.signature = 0;
    CHNLIB_System_FreeMemory(env, CHNLIB_DEBUG_ARGUMENTS);
    
    return;
}


/* initializes mt[N] with a seed */
void CHNLIB_RandomGeneratorMT_SetSeed(CHNLIB_RandomGeneratorMT *env, unsigned long s)
{
    if(CHNLIB_StructureHeader_GetTypeID(env) != CHNLIB_STRUCT_ID_RandomGeneratorMT){
        return;
    }
    
    env->mt[0]= s & 0xffffffffUL;
    for (env->mti=1; env->mti<N; env->mti++) {
        env->mt[env->mti] = 
	    (1812433253UL * (env->mt[env->mti-1] ^ (env->mt[env->mti-1] >> 30)) + env->mti); 
        /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
        /* In the previous versions, MSBs of the seed affect   */
        /* only MSBs of the array mt[].                        */
        /* 2002/01/09 modified by Makoto Matsumoto             */
        env->mt[env->mti] &= 0xffffffffUL;
        /* for >32 bit machines */
    }
}

/* initialize by an array with array-length */
/* init_key is the array for initializing keys */
/* key_length is its length */
/* slight change for C++, 2004/2/26 */
void CHNLIB_RandomGeneratorMT_SetSeedFromArray(CHNLIB_RandomGeneratorMT *env, unsigned long init_key[], int key_length)
{
    int i, j, k;
    
    if(CHNLIB_StructureHeader_GetTypeID(env) != CHNLIB_STRUCT_ID_RandomGeneratorMT){
        return;
    }
    
    CHNLIB_RandomGeneratorMT_SetSeed(env, 19650218UL);
    i=1; j=0;
    k = (N>key_length ? N : key_length);
    for (; k; k--) {
        env->mt[i] = (env->mt[i] ^ ((env->mt[i-1] ^ (env->mt[i-1] >> 30)) * 1664525UL))
          + init_key[j] + j; /* non linear */
        env->mt[i] &= 0xffffffffUL; /* for WORDSIZE > 32 machines */
        i++; j++;
        if (i>=N) { env->mt[0] = env->mt[N-1]; i=1; }
        if (j>=key_length) j=0;
    }
    for (k=N-1; k; k--) {
        env->mt[i] = (env->mt[i] ^ ((env->mt[i-1] ^ (env->mt[i-1] >> 30)) * 1566083941UL))
          - i; /* non linear */
        env->mt[i] &= 0xffffffffUL; /* for WORDSIZE > 32 machines */
        i++;
        if (i>=N) { env->mt[0] = env->mt[N-1]; i=1; }
    }

    env->mt[0] = 0x80000000UL; /* MSB is 1; assuring non-zero initial array */ 
}

/* generates a random number on [0,0xffffffff]-interval */
uint CHNLIB_RandomGeneratorMT_GetRandomUnsignedInteger32(CHNLIB_RandomGeneratorMT *env)
{
    unsigned long y;
    static unsigned long mag01[2]={0x0UL, MATRIX_A};
    /* mag01[x] = x * MATRIX_A  for x=0,1 */
    
    if(CHNLIB_StructureHeader_GetTypeID(env) != CHNLIB_STRUCT_ID_RandomGeneratorMT){
        return 0;
    }

    if (env->mti >= N) { /* generate N words at one time */
        int kk;

        if (env->mti == N+1)   /* if init_genrand() has not been called, */
            CHNLIB_RandomGeneratorMT_SetSeed(env, 5489UL); /* a default initial seed is used */

        for (kk=0;kk<N-M;kk++) {
            y = (env->mt[kk]&UPPER_MASK)|(env->mt[kk+1]&LOWER_MASK);
            env->mt[kk] = env->mt[kk+M] ^ (y >> 1) ^ mag01[y & 0x1UL];
        }
        for (;kk<N-1;kk++) {
            y = (env->mt[kk]&UPPER_MASK)|(env->mt[kk+1]&LOWER_MASK);
            env->mt[kk] = env->mt[kk+(M-N)] ^ (y >> 1) ^ mag01[y & 0x1UL];
        }
        y = (env->mt[N-1]&UPPER_MASK)|(env->mt[0]&LOWER_MASK);
        env->mt[N-1] = env->mt[M-1] ^ (y >> 1) ^ mag01[y & 0x1UL];

        env->mti = 0;
    }
  
    y = env->mt[env->mti++];

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return (uint)y;
}

/* generates a random number on [0,1]-real-interval */
//double genrand_real1(void)
//{
//    return genrand_int32()*(1.0/4294967295.0);
//    /* divided by 2^32-1 */
//}

/* generates a random number on [0,1)-real-interval */
//double genrand_real2(void)
//{
//    return genrand_int32()*(1.0/4294967296.0);
//    /* divided by 2^32 */
//}

/* generates a random number on (0,1)-real-interval */
//double genrand_real3(void)
//{
//    return (((double)genrand_int32()) + 0.5)*(1.0/4294967296.0);
//    /* divided by 2^32 */
//}

/* generates a random number on [0,1) with 53-bit resolution*/
//double genrand_res53(void)
//{
//    unsigned long a=genrand_int32()>>5, b=genrand_int32()>>6;
//    return(a*67108864.0+b)*(1.0/9007199254740992.0);
//}

//
//Internal functions
//

CHNLIB_RandomGeneratorMT *CHNLIB_RandomGeneratorMT_Internal_Allocate(void)
{
    CHNLIB_RandomGeneratorMT *tag;
    
    tag = (CHNLIB_RandomGeneratorMT *)CHNLIB_System_AllocateMemory_Strict(sizeof(CHNLIB_RandomGeneratorMT), CHNLIB_DEBUG_ARGUMENTS);
    
    CHNLIB_StructureHeader_Initialize(&tag->header, CHNLIB_STRUCT_ID_RandomGeneratorMT);
    
    return tag;
}

void CHNLIB_RandomGeneratorMT_Internal_Destruct(void **structure)
{
    //デストラクタ（実際にRelease->freeされる時に呼ばれる）
    if(structure == NULL){
        return;
    }
    
#ifdef DEBUG_MEMORY_REFERENCE_COUNT
    CHNLIB_Debug("Release(with free)[%p].", CHNLIB_DEBUG_ARGUMENTS, *structure);
#endif
    
    CHNLIB_RandomGeneratorMT_Free(*structure);
    
    *structure = NULL;
    
    return;
}

uint CHNLIB_RandomGeneratorMT_Internal_GetHash(const void *structure)
{
    //構造体の値が静的な意味を持たない場合は、構造体のアドレスの下位32bit分を返し、Hashの代わりとする。
    uint hash;
    
    hash = 0;
    
    if(CHNLIB_StructureHeader_GetTypeID(structure) == CHNLIB_STRUCT_ID_RandomGeneratorMT){
        hash = (uint)&structure;
    }
    
    return hash;
}
