// Chattr - Twitter client on .NET
//
// Copyright (c) 2007 Katsuhiko Ichinose <ichi@users.sourceforge.jp>
//
// Chattr is Free Software released under GNU General Public License.
//
// $Id: ChattrFormMain.cs 10 2007-09-30 14:34:09Z ichi $

using System;
using System.Collections.Generic;
using System.Text;

using System.ComponentModel;
using System.IO;
using System.Data;
using System.Drawing;
using System.Drawing.Imaging;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Net;
using System.Net.NetworkInformation;
using System.Xml;
using System.Globalization;
using System.Collections;
using System.Runtime.InteropServices;
using System.Threading;
using System.Data.SQLite;
using System.Reflection;
using agsXMPP;

using System.Diagnostics;

namespace Chattr
{
    public partial class FormMain : Form
    {
        private static readonly string EXEC_PATH = System.IO.Path.GetDirectoryName(Application.ExecutablePath);
        private static readonly string SQL_PATH = EXEC_PATH + @"\..\";

        public const int NOTIFY_ALL = 0;
        public const int NOTIFY_INME = 1;
        public const int NOTIFY_KEYWORDS = 2;

        private const int FLAG_INME = 0x01;
        private const int FLAG_KEYWORDS = 0x02;
        private const int FLAG_NEW = 0x04;

        private const int ICON_SIZE = 48;

        private enum ProgressOp
        {
            BG_FLAG = 0,
            STATUS_BAR,
            LABEL_STATUS,
            NUM_STATUS,
            PROGRESS_BAR,
            LIST_ENABLE,
            LIST_BEGIN_UPDATE,
            LIST_END_UPDATE,
            LIST_ADD,
            LIST_INSERT,
            LIST_REPLACE,
            LIST_DELETE,
            LIST_REMOVE,
            LIST_REMOVES,
            LIST_SELECT,
            LIST_CLEAR,
            LIST_SEARCH_BEGIN_UPDATE,
            LIST_SEARCH_END_UPDATE,
            LIST_SEARCH_ADD,
            LIST_SEARCH_INSERT,
            LIST_SEARCH_REPLACE,
            LIST_SEARCH_DELETE,
            LIST_SEARCH_REMOVE,
            LIST_SEARCH_REMOVES,
            LIST_SEARCH_SELECT,
            LIST_SEARCH_CLEAR,
        }

        private enum LogLimit
        {
            NONE = 0,
            DAY1,
            DAY3,
            WEEK1,
            WEEK2,
            MONTH1,
            MONTH2,
            MONTH3,
            MONTH6,
            YEAR1,
        }

        private const int VALID_SPAN = 3;

        private const string LABEL_AVAILABLE = "Characters available: ";
        private const string REGEX_TINYURL = @"^((?:http|https)://tinyurl\.com/[0-9a-zA-Z]+)";

        private ArrayList m_BgWork = new ArrayList();
        private bool m_fBgBusy = false;
        private bool m_fBgFlag;

        private FormPreferences m_Prefs;

        private bool m_fEnableFriends = false;
        private DateTime m_TimeFriends = DateTime.Now;
        private TimeSpan m_SpanFriends = TimeSpan.Zero;
        private bool m_fEnableReplies = false;
        private DateTime m_TimeReplies = DateTime.Now;
        private TimeSpan m_SpanReplies = TimeSpan.Zero;
        private DateTime m_TimeOfflineIn = DateTime.Now;
        private TimeSpan m_SpanOfflineFriends = TimeSpan.Zero;
        private TimeSpan m_SpanOfflineReplies = TimeSpan.Zero;

        private int m_Loading;
        private int m_Load;
        private int m_LoadP;
        private long m_LastId;

        private XmppClientConnection m_Xmpp = null;

        private DataSetChattr.TableStatusDataTable m_TblSearch = null;
        private string m_Search;

        private int m_ListWidth;

        private FormBrowser m_Browser;
        private FormPopup m_Popup;

        private ToolTip m_ToolTip = new ToolTip();

        private Mutex m_MutexStatus;
        private Mutex m_MutexSearch;

        [DllImport("user32.dll")]
        static extern bool FlashWindow(IntPtr hwnd, bool bInvert);

        public void Navigate(string url)
        {
            try
            {
                if (Properties.Settings.Default.UseBuiltinBrowser)
                {
                    m_Browser.Navigate(url, m_Loading >= m_Load);
                }
                else
                {
                    System.Diagnostics.Process.Start(url);
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("Navigate() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        public void DeleteLog()
        {
            RequestDeleteAll();
        }

        private bool InitializeXmpp()
        {
            if (Properties.Settings.Default.UseXmpp && (Properties.Settings.Default.XmppID != "") && (m_Prefs.XmppPass != ""))
            {
                try
                {
                    m_Xmpp = new XmppClientConnection();
                    Jid jid = new Jid(Properties.Settings.Default.XmppID);
                    m_Xmpp.OnSocketError += new ErrorHandler(xmpp_OnSocketError);
                    m_Xmpp.OnError += new ErrorHandler(xmpp_OnError);
                    m_Xmpp.OnAuthError += new XmppElementHandler(xmpp_OnAuthError);
                    m_Xmpp.OnLogin += new ObjectHandler(xmpp_OnLogin);
                    m_Xmpp.OnIq += new agsXMPP.protocol.client.IqHandler(xmpp_OnIq);
                    m_Xmpp.OnMessage += new agsXMPP.protocol.client.MessageHandler(xmpp_OnMessage);
                    m_Xmpp.ConnectServer = Properties.Settings.Default.XmppServer;
                    m_Xmpp.Username = jid.User;
                    m_Xmpp.Server = jid.Server;
                    m_Xmpp.Password = m_Prefs.XmppPass;
                    m_Xmpp.Resource = "Chattr";
                    m_Xmpp.Port = Properties.Settings.Default.XmppPort;
                    //xmpp.AutoResolveConnectServer = true;
                    //xmpp.Priority = 10;
                    m_Xmpp.UseSSL = Properties.Settings.Default.XmppUseSSL;
                    m_Xmpp.SocketConnectionType = agsXMPP.net.SocketConnectionType.Direct;
                    m_Xmpp.UseStartTLS = true;
                    //xmpp.RegisterAccount = true;
                    return true;
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("InitializeXmpp() Error\n" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
            }
            return false;
        }

        delegate void DelegateXmppError(string msg, string caption);
        private void XmppErrorDelegate(string msg, string caption)
        {
            MessageBox.Show(this, msg, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private void XmppError(string msg, string caption)
        {
            Invoke(new DelegateXmppError(XmppErrorDelegate), msg, caption);
        }

        private void xmpp_OnSocketError(object sender, Exception ex)
        {
            XmppError("XMPP \PbgG[ " + ex.Message, "XMPP G[");
            Debug.WriteLine("xmpp_OnSocketError: " + sender + ex);
        }

        private void xmpp_OnError(object sender, Exception ex)
        {
            XmppError("XMPP G[ " + ex.Message, "XMPP G[");
            Debug.WriteLine("xmpp_OnError: " + sender + ex);
        }

        private void xmpp_OnAuthError(object sender, agsXMPP.Xml.Dom.Element e)
        {
            XmppError("XMPP F؃G[", "XMPP G[");
            Debug.WriteLine("xmpp_OnAuthError: " + sender + e);
        }

        private void xmpp_OnLogin(object sender)
        {
            Debug.WriteLine("xmpp_OnLogin: " + sender);
            m_Xmpp.SendMyPresence();
            m_Xmpp.Status = "OnLine";
            m_Xmpp.Show = agsXMPP.protocol.client.ShowType.chat;
        }

        private void xmpp_OnIq(object sender, agsXMPP.protocol.client.IQ iq)
        {
            Debug.WriteLine("xmpp_OnIq: " + sender + iq);
        }

        private void xmpp_OnMessage(object sender, agsXMPP.protocol.client.Message msg)
        {
            Debug.WriteLine("xmpp_OnMessage: " + sender + msg);
            RequestBackGround(msg);
        }

        private void OpenXmpp()
        {
            Debug.WriteLine("OpenXmpp()");
            try
            {
                if (NetworkInterface.GetIsNetworkAvailable())
                {
                    if (m_Xmpp == null)
                    {
                        if (InitializeXmpp())
                        {
                            m_Xmpp.Open();
                        }
                    }
                }
                else
                {
                    throw new WebException("Is not network available.", WebExceptionStatus.ConnectFailure);
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("OpenXmpp() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        private void CloseXmpp()
        {
            Debug.WriteLine("CloseXmpp()");
            try
            {
                if (m_Xmpp != null)
                {
                    //m_Xmpp.Status = "OffLine";
                    //m_Xmpp.Show = agsXMPP.protocol.client.ShowType.NONE;
                    m_Xmpp.Close();
                    m_Xmpp = null;
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("CloseXmpp() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        private void SetWindowTitle()
        {
            if (Properties.Settings.Default.UserName != "")
            {
                this.Text = Properties.Settings.Default.UserName + " - " + "Chattr";
            }
        }

        private void SetWindowSize()
        {
            if (Screen.AllScreens.Length > 1)
            {
                this.Location = Properties.Settings.Default.FormMainLocation2;
                this.Size = Properties.Settings.Default.FormMainSize2;
            }
            else
            {
                this.Location = Properties.Settings.Default.FormMainLocation;
                this.Size = Properties.Settings.Default.FormMainSize;
            }
        }

        private void SaveWindowSize()
        {
            if (this.Visible && (this.WindowState == FormWindowState.Normal))
            {
                if (Screen.AllScreens.Length > 1)
                {
                    Properties.Settings.Default.FormMainSize2 = this.Size;
                }
                else
                {
                    Properties.Settings.Default.FormMainSize = this.Size;
                }
            }
        }

        private void SaveWindowPosition()
        {
            if (this.Visible && (this.WindowState == FormWindowState.Normal))
            {
                if (Screen.AllScreens.Length > 1)
                {
                    Properties.Settings.Default.FormMainLocation2 = this.Location;
                }
                else
                {
                    Properties.Settings.Default.FormMainLocation = this.Location;
                }
            }
        }

        private void SetLocation()
        {
            foreach (string location in Regex.Split(Properties.Settings.Default.Locations, "\n"))
            {
                if ((location != "") && (location != "\r"))
                {
                    comboBoxLocation.Items.Add(location);
                }
            }
            if (Properties.Settings.Default.Location != "")
            {
                comboBoxLocation.ForeColor = Color.FromName(KnownColor.WindowText.ToString());
                comboBoxLocation.Text = Properties.Settings.Default.Location;
            }
        }

        private void UpdateLocation()
        {
            string locations = "";
            foreach (string location in comboBoxLocation.Items)
            {
                if (locations != "")
                {
                    locations += "\n";
                }
                locations += location;
            }
            Properties.Settings.Default.Locations = locations;
            if (comboBoxLocation.ForeColor != Color.FromName(KnownColor.GrayText.ToString()))
            {
                Properties.Settings.Default.Location = comboBoxLocation.Text;
            }
            else
            {
                Properties.Settings.Default.Location = "";
            }
        }

        private String MeasureDate(ListBox listBox, Rectangle bounds, Graphics graph, int idx, DataSetChattr.TableStatusRow row, ref SizeF sizeF, ref RectangleF rectF)
        {
            DateTime at = row.created_at.ToLocalTime();
            String str = at.ToString("HH:mm\nMM/dd");
            sizeF = graph.MeasureString(str, Properties.Settings.Default.FontTime);
            rectF = (RectangleF)bounds;
            rectF.X = rectF.Right - sizeF.Width - 2;
            rectF.Width = sizeF.Width;
            rectF.Y += 3;
            rectF.Height = sizeF.Height;
            return str;
        }

        static private String MeasureText(ListBox listBox, Rectangle bounds, Graphics graph, int idx, DataSetChattr.TableStatusRow row, bool selected, int width, SizeF sizeAt, ref SizeF sizeF, ref RectangleF rectF)
        {
            String str = row.user_screen_name;
            rectF = (RectangleF)bounds;
            if (Properties.Settings.Default.SmallIcon)
            {
                str += "  " + Regex.Replace(row.text, @"\n", " ");
                if (Properties.Settings.Default.IndicateFrom)
                {
                    str += "  from " + row.source;
                }
                if (Properties.Settings.Default.NotEllipsis)
                {
                    SizeF z = graph.MeasureString("@\n@", Properties.Settings.Default.FontStatuses);
                    if (z.Height > width)
                    {
                        rectF.X = z.Height + 4;
                    }
                    else
                    {
                        rectF.X = width + 4;
                    }
                    z = new SizeF(rectF.Width - (sizeAt.Width + rectF.X + 2), 255);
                    sizeF = graph.MeasureString(str, Properties.Settings.Default.FontStatuses, z);
                }
                else
                {
                    sizeF = graph.MeasureString("@\n@", Properties.Settings.Default.FontStatuses);
                    if (sizeF.Height > width)
                    {
                        rectF.X = sizeF.Height + 4;
                    }
                    else
                    {
                        rectF.X = width + 4;
                    }
                }
            }
            else
            {
                if ((row.user_name != "") && (row.user_name != row.user_screen_name))
                {
                    str += " / " + row.user_name;
                }
                str += "\n" + Regex.Replace(row.text, @"\n", " "); ;
                if (Properties.Settings.Default.IndicateFrom)
                {
                    str += "  from " + row.source;
                }
                if (Properties.Settings.Default.NotEllipsis)
                {
                    SizeF z = new SizeF(rectF.Width - (sizeAt.Width + ICON_SIZE + 6), 255);
                    sizeF = graph.MeasureString(str, Properties.Settings.Default.FontStatuses, z);
                }
                else
                {
                    sizeF = graph.MeasureString("@\n@\n@", Properties.Settings.Default.FontStatuses);
                }
                if (sizeF.Height < ICON_SIZE)
                {
                    sizeF.Height = ICON_SIZE;
                }
                rectF.X = ICON_SIZE + 4;
            }
            rectF.Width -= sizeAt.Width + rectF.X + 2;
            rectF.Y += 3;
            rectF.Height = sizeF.Height;
            return str;
        }

        private int listBox_MeasureItem(ListBox listBox, MeasureItemEventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            int height = ICON_SIZE;
            if (table != null)
            {
                if (mutex.WaitOne(0, false))
                {
                    try
                    {
                        Rectangle bounds = new Rectangle(0, 0, m_ListWidth - 21, e.ItemHeight);
                        bool selected = (e.Index == listBox.SelectedIndex);
                        SizeF sizeAt = SizeF.Empty;
                        SizeF sizeF = SizeF.Empty;
                        RectangleF rectF = RectangleF.Empty;
                        int idx = e.Index;
                        if ((idx < 0) || (idx >= table.Rows.Count))
                        {
                            idx = 0;
                        }
                        long id = ((IListItem)listBox.Items[idx]).Id;
                        int i = idx - 50;
                        if (i < 0)
                        {
                            i = 0;
                        }
                        int cnt = table.Rows.Count;
                        DataSetChattr.TableStatusRow row = table.Rows[i] as DataSetChattr.TableStatusRow;
                        while (i < cnt)
                        {
                            if (row.id == id)
                            {
                                break;
                            }
                            i++;
                            row = table.Rows[i] as DataSetChattr.TableStatusRow;
                        }
                        if (i >= cnt)
                        {
                            row = table.Rows[0] as DataSetChattr.TableStatusRow;
                            Debug.Assert(false, "listBox_MeasureItem: not found");
                        }
                        MeasureDate(listBox, bounds, e.Graphics, e.Index, row, ref sizeAt, ref rectF);
                        int width = (int)(sizeAt.Height + 0.5);
                        if (width > ICON_SIZE)
                        {
                            width = ICON_SIZE;
                        }
                        MeasureText(listBox, bounds, e.Graphics, e.Index, row, selected, width, sizeAt, ref sizeF, ref rectF);
                        height = (int)(sizeF.Height + 0.5);
                        if (height < width)
                        {
                            height = width;
                        }
                        height += 5;
                        if (height > 255)
                        {
                            height = 255;
                        }
                        Debug.WriteLineIf(m_Loading >= m_Load, "listBox_MeasureItem: " + height + " / " + row.id + ": " + row.text);
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("listBox_MeasureItem() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
            return height;
        }

        private void listBox_DrawItem(ListBox listBox, DrawItemEventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                if (mutex.WaitOne(0, false))
                {
                    try
                    {
                        if ((e.Index >= 0) && (e.Index < listBox.Items.Count))
                        {
                            bool selected = ((e.State & DrawItemState.Selected) != 0);
                            int idx = e.Index;
                            DataSetChattr.TableStatusRow row = table.Rows[idx] as DataSetChattr.TableStatusRow;
                            // Brush
                            Brush brushBg;
                            if (selected)
                            {
                                brushBg = new SolidBrush(Properties.Settings.Default.BgColorHighlight);
                            }
                            else if ((row.flags & FLAG_INME) != 0)
                            {
                                brushBg = new SolidBrush(Properties.Settings.Default.BgColorInMe);
                            }
                            else if ((row.flags & FLAG_KEYWORDS) != 0)
                            {
                                brushBg = new SolidBrush(Properties.Settings.Default.BgColorKeywords);
                            }
                            else if ((row.flags & FLAG_NEW) != 0)
                            {
                                brushBg = new SolidBrush(Properties.Settings.Default.BgColorNew);
                            }
                            else
                            {
                                brushBg = new SolidBrush(Properties.Settings.Default.BgColor);
                            }
                            Brush brushText;
                            Brush brushTextAt;
                            if (selected)
                            {
                                brushText = new SolidBrush(Properties.Settings.Default.ColorHighlight);
                                brushTextAt = new SolidBrush(Properties.Settings.Default.ColorHighlight);
                            }
                            else
                            {
                                brushText = new SolidBrush(Properties.Settings.Default.ColorText);
                                brushTextAt = new SolidBrush(Properties.Settings.Default.ColorText);
                            }
                            // Date
                            SizeF sizeAt = SizeF.Empty;
                            RectangleF rectAt = RectangleF.Empty;
                            String strAt = MeasureDate(listBox, e.Bounds, e.Graphics, e.Index, row, ref sizeAt, ref rectAt);
                            // Status
                            int height = (int)(sizeAt.Height + 0.5);
                            SizeF sizeF = SizeF.Empty;
                            RectangleF rectF = RectangleF.Empty;
                            String text = MeasureText(listBox, e.Bounds, e.Graphics, e.Index, row, selected, height, sizeAt, ref sizeF, ref rectF);
                            StringFormat format = StringFormat.GenericDefault;
                            format.Alignment = StringAlignment.Near;
                            format.Trimming = StringTrimming.EllipsisCharacter;
                            // ICON
                            DataSetChattr.TableProfileImageRow pro = dataSetChattr.TableProfileImage.Rows.Find(row.user_profile_image) as DataSetChattr.TableProfileImageRow;
                            Bitmap bmp = null;
                            if (pro != null)
                            {
                                int iconHeight = height;
                                if (iconHeight < (int)rectF.X - 4)
                                {
                                    iconHeight = (int)rectF.X - 4;
                                }
                                if ((!Properties.Settings.Default.SmallIcon) || (iconHeight > ICON_SIZE))
                                {
                                    iconHeight = ICON_SIZE;
                                }
                                bmp = new Bitmap(ByteArray2Bitmap(pro.image, pro.width, pro.height), iconHeight, iconHeight);
                            }
                            // Draw
                            e.Graphics.FillRectangle(brushBg, e.Bounds);
                            if (bmp != null)
                            {
                                e.Graphics.DrawImage(bmp, e.Bounds.X + 2, e.Bounds.Y + 2);
                            }
                            e.Graphics.DrawString(text, Properties.Settings.Default.FontStatuses, brushText, rectF, format);
                            format = StringFormat.GenericDefault;
                            format.Alignment = StringAlignment.Center;
                            e.Graphics.DrawString(strAt, Properties.Settings.Default.FontTime, brushTextAt, rectAt, format);
                            e.Graphics.DrawLine(new Pen(Color.FromName(KnownColor.ControlDark.ToString())), e.Bounds.X, e.Bounds.Bottom - 1, e.Bounds.Right, e.Bounds.Bottom - 1);
                            e.DrawFocusRectangle();
                        }
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("listBox_DrawItem() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
        }

        private void listBox_KeyDown(ListBox listBox, KeyEventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            switch (e.KeyData)
            {
                case Keys.Enter:
                    listBox_DoubleClick(listBox, e, table, mutex);
                    e.Handled = true;
                    break;
            }
        }

        private void listBox_DoubleClick(ListBox listBox, EventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                EntryReply(listBox, e, table, mutex);
            }
        }

        private void listBox_SelectedIndexChanged(ListBox listBox, EventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                bool enter = mutex.WaitOne(0, false);
                int idx = listBox.SelectedIndex;
                if (enter)
                {
                    try
                    {
                        if ((idx >= 0) && (idx < table.Rows.Count))
                        {
                            DataSetChattr.TableStatusRow row = table.Rows[idx] as DataSetChattr.TableStatusRow;
                            if (row.RowState != DataRowState.Deleted)
                            {
                                Debug.WriteLine("SelectChanged: " + idx + " / " + row.id);
                                int flags = row.flags;
                                flags &= (~FLAG_NEW);
                                StatusParser parser = new StatusParser(row.text);
                                string[] urls = parser.URLs;
                                if (urls != null)
                                {
                                    foreach (string url in urls)
                                    {
                                        RequestTinyUrl(url);
                                    }
                                }
                                if (!ParseKeyword(parser, row))
                                {
                                    flags &= (~FLAG_KEYWORDS);
                                }
                                if (row.flags != flags)
                                {
                                    Debug.WriteLine("flags changed: " + row.flags + " --> " + flags);
                                    row.flags = flags;
                                }
                                RequestProfileImage(row.user_profile_image);
                                RequestWork();
                            }
                        }
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("listBox_SelectedIndexChanged() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
        }

        private void EntryReply(ListBox listBox, EventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                if (mutex.WaitOne(0, false))
                {
                    try
                    {
                        if (textBoxStatus.Enabled)
                        {
                            int i = listBox.SelectedIndex;
                            if (i >= 0)
                            {
                                DataSetChattr.TableStatusRow row = table.Rows[i] as DataSetChattr.TableStatusRow;
                                textBoxStatus_Enter(textBoxStatus, e);
                                textBoxStatus.Text = "@" + row.user_screen_name + " " + textBoxStatus.Text;
                                textBoxStatus.Focus();
                                textBoxStatus.SelectionStart = textBoxStatus.Text.Length;
                                textBoxStatus.SelectionLength = 0;
                            }
                        }
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("ReplyTextBox() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
        }

        private void EntryDirectMessage(ListBox listBox, EventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                if (mutex.WaitOne(0, false))
                {
                    try
                    {
                        if (textBoxStatus.Enabled)
                        {
                            int i = listBox.SelectedIndex;
                            if (i >= 0)
                            {
                                DataSetChattr.TableStatusRow row = table.Rows[i] as DataSetChattr.TableStatusRow;
                                textBoxStatus_Enter(textBoxStatus, e);
                                textBoxStatus.Text = "d " + row.user_screen_name + " " + textBoxStatus.Text;
                                textBoxStatus.Focus();
                                textBoxStatus.SelectionStart = textBoxStatus.Text.Length;
                                textBoxStatus.SelectionLength = 0;
                            }
                        }
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("DirectMessageTextBox() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
        }

        private void Favorite(ListBox listBox, EventArgs e, DataSetChattr.TableStatusDataTable table, Mutex mutex)
        {
            if (table != null)
            {
                if (mutex.WaitOne(0, false))
                {
                    try
                    {
                        int idx = listBox.SelectedIndex;
                        if ((idx >= 0) && (idx < table.Rows.Count))
                        {
                            long id = ((DataSetChattr.TableStatusRow)table.Rows[idx]).id;
                            if (id >= 0)
                            {
                                SetToolStripStatusLabelText("Cɓo^...");
                                HttpWebResponse res = TwitterAPI.Favorite(Properties.Settings.Default.UserName, m_Prefs.Password, id);
                                if (res != null)
                                {
                                    if (res.StatusCode == HttpStatusCode.OK)
                                    {
                                        SetToolStripStatusLabelText("Cɓo^");
                                    }
                                    else
                                    {
                                        Debug.WriteLine("Favoring: " + res.StatusCode);
                                        SetToolStripStatusLabelText("Cɓo^G[");
                                    }
                                }
                            }
                        }
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("Favorite() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        mutex.ReleaseMutex();
                    }
                }
            }
        }

        private Bitmap ByteArray2Bitmap(byte[] img, int width, int height)
        {
            Bitmap bmp = new Bitmap(width, height);
            BitmapData bmpData = bmp.LockBits(
                new Rectangle(0, 0, bmp.Width, bmp.Height),
                ImageLockMode.WriteOnly,
                PixelFormat.Format32bppArgb);
            Marshal.Copy(img, 0, bmpData.Scan0, img.Length);
            bmp.UnlockBits(bmpData);
            bmp.RotateFlip(RotateFlipType.RotateNoneFlipY);
            return bmp;
        }

        private byte[] Bitmap2ByteArray(Bitmap bmp)
        {
            bmp.RotateFlip(RotateFlipType.RotateNoneFlipY);
            BitmapData bmpData = bmp.LockBits(
                new Rectangle(0, 0, bmp.Width, bmp.Height),
                ImageLockMode.ReadOnly,
                PixelFormat.Format32bppArgb);
            byte[] img = new byte[bmp.Width * bmp.Height * 4];
            Marshal.Copy(bmpData.Scan0, img, 0, img.Length);
            bmp.UnlockBits(bmpData);
            return img;
        }

        private bool ExistRequest(string arg)
        {
            int cnt = ArrayList.Synchronized(m_BgWork).Count;
            for (int i = 0; i < cnt; i++)
            {
                if ((!m_fBgBusy) || (i > 0))
                {
                    object obj = ArrayList.Synchronized(m_BgWork)[i];
                    if ((obj.GetType() == typeof(string)) && ((string)obj == arg))
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        private void RequestBackGround(object arg)
        {
            if (arg != null)
            {
                ArrayList.Synchronized(m_BgWork).Add(arg);
            }
            if (m_BgWork.Count > 0)
            {
                if (!backgroundWorker.IsBusy && !m_fBgBusy)
                {
                    m_fBgBusy = true;
                    backgroundWorker.RunWorkerAsync(m_BgWork[0]);
                }
            }
        }

        private void RequestWork()
        {
            RequestBackGround(null);
        }

        private void RequestLoad()
        {
            if (!ExistRequest("load"))
            {
                RequestBackGround("load");
            }
        }

        private void RequestDeleteAll()
        {
            if (!ExistRequest("delete_all"))
            {
                RequestBackGround("delete_all");
            }
        }

        private void RequestProfileImage(String profile_image)
        {
            if (dataSetChattr.TableProfileImage.Rows.Find(profile_image) == null)
            {
                Debug.WriteLine("RequestProfileImage: " + profile_image);
                RequestBackGround("profile_image " + profile_image);
            }
        }

        private void RequestTinyUrl(String url)
        {
            Regex r = new Regex(REGEX_TINYURL);
            Match m = r.Match(url);
            if (m.Success && (dataSetChattr.TableTinyUrl.Rows.Find(m.Groups[1].Value) == null))
            {
                Debug.WriteLine("RequestTinyUrl: " + m.Groups[1].Value);
                RequestBackGround("tinyurl " + m.Groups[1].Value);
            }
        }

        private void UpdateStatus(String status)
        {
            RequestBackGround("update " + status);
        }

        private void GetFriendsTimeline()
        {
            String arg = "friends_timeline";
            if (!ExistRequest(arg))
            {
                Debug.WriteLine("GetFriends(" + DateTime.Now + ")");
                m_fEnableFriends = false;
                RequestBackGround(arg);
            }
        }

        private void GetReplies()
        {
            Debug.WriteLine("GetReplies(" + DateTime.Now + ")");
            m_fEnableReplies = false;
            String arg = "replies";
            RequestBackGround(arg);
        }

        private void StartTimer()
        {
            DateTime now = DateTime.Now;
            StartFriendsTimer(now);
            StartRepliesTimer(now);
        }

        private void StartFriendsTimer(DateTime now)
        {
            if (Properties.Settings.Default.IntervalFriends != 0)
            {
                m_SpanFriends = new TimeSpan(Properties.Settings.Default.IntervalFriends * 10000000 * 60);
                m_TimeFriends = now;
                m_SpanOfflineFriends = TimeSpan.Zero;
                timer.Enabled = true;
                m_fEnableFriends = true;
                Debug.WriteLine("StartFriendsTimer(" + now + ")");
            }
            else
            {
                m_fEnableFriends = false;
            }
        }

        private void StartRepliesTimer(DateTime now)
        {
            if (Properties.Settings.Default.IntervalReplies != 0)
            {
                m_SpanReplies = new TimeSpan(Properties.Settings.Default.IntervalReplies * 10000000 * 60);
                m_TimeReplies = now;
                m_SpanOfflineReplies = TimeSpan.Zero;
                timer.Enabled = true;
                m_fEnableReplies = true;
                Debug.WriteLine("StartRepliesTimer(" + now + ")");
            }
            else
            {
                m_fEnableReplies = false;
            }
        }

        private object DoBackGround_String(String arg, BackgroundWorker worker)
        {
            object result = null;
            Regex r = new Regex(@"^(.*?) (.*)$");
            Match m = r.Match(arg);
            String a = new String(arg.ToCharArray());
            String v = "";
            if (m.Success)
            {
                a = m.Groups[1].Value;
                v = m.Groups[2].Value;
            }
            switch (a)
            {
                case "fill":
                    result = DoBackGround_Fill(v, worker);
                    break;
                case "load":
                    result = DoBackGround_Load(v, worker);
                    break;
                case "friends_timeline":
                    result = DoBackGround_GetFriendsTimeline(v, worker);
                    break;
                case "replies":
                    result = DoBackGround_GetReplies(v, worker);
                    break;
                case "profile_image":
                    result = DoBackGround_GetProfileImage(v, worker);
                    break;
                case "update":
                    result = DoBackGround_Update(v, worker);
                    break;
                case "tinyurl":
                    result = DoBackGround_GetTinyUrl(v, worker);
                    break;
                case "delete_all":
                    result = DoBackGround_DeleteAll(v, worker);
                    break;
                //case "delete":
                //    result = DoBackGround_Delete(v, worker);
                //    break;
            }
            return result;
        }

        private static void ThreadFillStatus(object param)
        {
            if (param.GetType() == typeof(IFillTableStatus))
            {
                IFillTableStatus ift = param as IFillTableStatus;
                ift.Adapter.Fill(ift.Table);
            }
        }

        private static void ThreadFillProfileImage(object param)
        {
            if (param.GetType() == typeof(IFillTableProfileImage))
            {
                IFillTableProfileImage ift = param as IFillTableProfileImage;
                ift.Adapter.Fill(ift.Table);
            }
        }

        private static void ThreadFillTinyUrl(object param)
        {
            if (param.GetType() == typeof(IFillTableTinyUrl))
            {
                IFillTableTinyUrl ift = param as IFillTableTinyUrl;
                ift.Adapter.Fill(ift.Table);
            }
        }

        private static bool RunThread(Thread thread, BackgroundWorker worker, object param)
        {
            thread.Start(param);
            while (thread.IsAlive)
            {
                if (worker.CancellationPending)
                {
                    thread.Abort();
                    return false;
                }
                Thread.Sleep(0);
            }
            return true;
        }

        private object DoBackGround_Fill(String arg, BackgroundWorker worker)
        {
            Debug.WriteLine("DoBackGround_Fill(" + arg + ")");
            if (worker.CancellationPending)
            {
                return null;
            }
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "[h...");

            SQLite.CreateDB(SQL_PATH);
            DateTime today = DateTime.Today;
            //if ((today > Properties.Settings.Default.VacuumDay) || Properties.Settings.Default.VacuumReq)
            //{
            //    Properties.Settings.Default.VacuumDay = today;
            //    Properties.Settings.Default.VacuumReq = false;
            //    SQLite.Vacuum(SQL_PATH);
            //}
            Directory.SetCurrentDirectory(EXEC_PATH + @"\..");

            worker.ReportProgress((int)ProgressOp.LIST_BEGIN_UPDATE);
            Thread t = new Thread(ThreadFillStatus);
            if (!RunThread(t, worker, new IFillTableStatus(tableStatusTableAdapter, dataSetChattr.TableStatus)))
            {
                return null;
            }
            t = new Thread(ThreadFillProfileImage);
            if (!RunThread(t, worker, new IFillTableProfileImage(tableProfileImageTableAdapter, dataSetChattr.TableProfileImage)))
            {
                return null;
            }
            t = new Thread(ThreadFillTinyUrl);
            if (!RunThread(t, worker, new IFillTableTinyUrl(tableTinyUrlTableAdapter, dataSetChattr.TableTinyUrl)))
            {
                return null;
            }
            int cnt = dataSetChattr.TableStatus.Rows.Count;
            Debug.WriteLine("DoBackGround_Fill: Loading...");
            m_Load = dataSetChattr.TableStatus.Rows.Count;
            m_Loading = 0;
            foreach (DataSetChattr.TableStatusRow row in dataSetChattr.TableStatus.Rows)
            {
                if (worker.CancellationPending)
                {
                    return null;
                }
                else if (m_Loading < 500)
                {
                    m_Loading++;
                    worker.ReportProgress((int)ProgressOp.LIST_ADD, new IListItem(row.id, 0));
                    m_LastId = row.id;
                }
                else
                {
                    break;
                }
            }
            Debug.WriteLine("DoBackGround_Fill: Loading done.");
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "[hI");
            if (dataSetChattr.TableStatus.Rows.Count > 0)
            {
                SetSelectedID(Properties.Settings.Default.SelectedID, worker);
                m_LoadP = m_Loading / m_Load;
            }
            else
            {
                m_LoadP = 0;
            }
            worker.ReportProgress((int)ProgressOp.LIST_END_UPDATE);
            worker.ReportProgress((int)ProgressOp.NUM_STATUS);
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "");
            return new IResultInit();
        }

        private object DoBackGround_Load(String arg, BackgroundWorker worker)
        {
            if (m_Loading < m_Load)
            {
                int cnt = dataSetChattr.TableStatus.Rows.Count;
                int idx = m_Loading - 50;
                while (((DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[idx]).id != m_LastId)
                {
                    idx++;
                    if (idx >= cnt)
                    {
                        idx = 0;// cnt - 2;
                        //break;
                        Debug.WriteLine("DoBackGround_Load() Error: overfllow");
                    }
                }
                idx++;
                if (idx >= cnt)
                {
                    idx = 0;
                }
                while (m_Loading < m_Load)
                {
                    m_Loading++;
                    DataSetChattr.TableStatusRow row = (DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[idx];
                    worker.ReportProgress((int)ProgressOp.LIST_ADD, new IListItem(row.id, idx));
                    m_LastId = row.id;
                    idx++;
                    int loadP = m_Loading * 100 / m_Load;
                    if (m_LoadP != loadP)
                    {
                        m_LoadP = loadP;
                        worker.ReportProgress((int)ProgressOp.NUM_STATUS);
                        worker.ReportProgress((int)ProgressOp.PROGRESS_BAR, m_LoadP);
                        if (m_BgWork.Count >= 2)
                        {
                            break;
                        }
                        m_fBgFlag = true;
                        worker.ReportProgress((int)ProgressOp.BG_FLAG, false);
                        do
                        {
                            if (worker.CancellationPending)
                            {
                                return null;
                            }
                            Thread.Sleep(67);
                        }
                        while (m_fBgFlag);
                    }
                }
                worker.ReportProgress((int)ProgressOp.NUM_STATUS);
                if (m_Loading >= m_Load)
                {
                    worker.ReportProgress((int)ProgressOp.STATUS_BAR, "");
                    Debug.WriteLine("DoBackGround_Load: done.");
                }
            }
            return new IResultLoadFin();
        }

        private static void ThreadGetFriendsTimeline(object param)
        {
            if (param.GetType() == typeof(IGetTimeline))
            {
                IGetTimeline arg = param as IGetTimeline;
                HttpWebResponse res = TwitterAPI.GetFriendsTimeline(arg.User, arg.Pass, arg.Option);
                arg.Response = res;
            }
        }

        private object DoBackGround_GetFriendsTimeline(String arg, BackgroundWorker worker)
        {
            IResultGetFriends result = new IResultGetFriends();
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "Following ^CC擾...");
            IGetTimeline igt = new IGetTimeline(Properties.Settings.Default.UserName, m_Prefs.Password, arg);
            Thread t = new Thread(ThreadGetFriendsTimeline);
            if (!RunThread(t, worker, igt))
            {
                return null;
            }
            HttpWebResponse res = igt.Response;
            if ((res != null) && (res.StatusCode == HttpStatusCode.OK))
            {
                XmlDocument doc = new XmlDocument();
                try
                {
                    doc.Load(res.GetResponseStream());
                    IResultGetFriends ret = RetrieveStatuses(doc.DocumentElement, worker, "Xe[^X");
                    if (ret != null)
                    {
                        result.Flags = ret.Flags;
                        result.List = ret.List;
                    }
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("doc.Load(res.GetResponseStream) Error\n" +
                        "in DoBackGround_GetFriendsTimeline()" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
                finally
                {
                    res.Close();
                }
            }
            else
            {
                worker.ReportProgress((int)ProgressOp.STATUS_BAR, "Following ^CC擾G[" + GetError(res));
                if (res != null)
                {
                    Debug.WriteLine("friends timeline: " + res.StatusCode + " - " + res.StatusDescription);
                }
            }
            return (object)result;
        }

        private static void ThreadGetReplies(object param)
        {
            if (param.GetType() == typeof(IGetTimeline))
            {
                IGetTimeline arg = param as IGetTimeline;
                HttpWebResponse res = TwitterAPI.GetReplies(arg.User, arg.Pass, arg.Option);
                arg.Response = res;
            }
        }

        private object DoBackGround_GetReplies(String arg, BackgroundWorker worker)
        {
            IResultGetReplies result = new IResultGetReplies();
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "Replies 擾...");
            IGetTimeline igt = new IGetTimeline(Properties.Settings.Default.UserName, m_Prefs.Password, arg);
            Thread t = new Thread(ThreadGetReplies);
            if (!RunThread(t, worker, igt))
            {
                return null;
            }
            HttpWebResponse res = igt.Response;
            if ((res != null) && (res.StatusCode == HttpStatusCode.OK))
            {
                XmlDocument doc = new XmlDocument();
                try
                {
                    doc.Load(res.GetResponseStream());
                    IResultGetFriends ret = RetrieveStatuses(doc.DocumentElement, worker, "vC");
                    if (ret != null)
                    {
                        result.Flags = ret.Flags;
                        result.List = ret.List;
                    }
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("doc.Load(res.GetResponseStream) Error\n" +
                        "in DoBackGround_GetRepliessTimeline()" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
                finally
                {
                    res.Close();
                }
            }
            else
            {
                worker.ReportProgress((int)ProgressOp.STATUS_BAR, "Reples 擾G[" + GetError(res));
                if (res != null)
                {
                    Debug.WriteLine("replies: " + res.StatusCode + " - " + res.StatusDescription);
                }
            }
            return (object)result;
        }

        private object DoBackGround_XmppMessage(agsXMPP.protocol.client.Message msg, BackgroundWorker worker)
        {
            IResultXmppMessage result = new IResultXmppMessage();
            DataSetChattr.TableStatusRow row_new = ParseXmpp(msg);
            if (row_new != null)
            {
                bool valid = false;
                try
                {
                    long selected = 0;
                    long selectedSearch = 0;
                    StatusParser parser = new StatusParser(row_new.text);
                    ParseUserName(parser, row_new);
                    ParseTinyUrl(parser);
                    ParseKeyword(parser, row_new);
                    DataSetChattr.TableStatusDataTable q = tableStatusTableAdapter.GetDataByUserText(row_new.user_screen_name, row_new.text);
                    if (q.Rows.Count == 0)
                    {
                        valid = true;
                    }
                    else
                    {
                        for (int i = q.Rows.Count - 1; i >= 0; i--)
                        {
                            if (worker.CancellationPending)
                            {
                                return null;
                            }
                            DataSetChattr.TableStatusRow rowq = (DataSetChattr.TableStatusRow)q.Rows[i];
                            if ((rowq.RowState != DataRowState.Deleted) &&
                                (Math.Abs((row_new.created_at - rowq.created_at).Hours) >= VALID_SPAN))
                            {
                                valid = true;
                                break;
                            }
                        }
                    }
                    if (worker.CancellationPending)
                    {
                        return null;
                    }
                    else if (valid)
                    {
                        selected = GetSelectedID();
                        selectedSearch = GetSelectedSearchID();
                        int cnt = dataSetChattr.TableStatus.Rows.Count;
                        if (cnt > 50)
                        {
                            cnt = 50;
                        }
                        for (int i = 0; i < cnt; i++)
                        {
                            DataSetChattr.TableStatusRow row = (DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[i];
                            if ((row.RowState != DataRowState.Deleted) && (row.id >= 0) &&
                                (row.user_screen_name == row_new.user_screen_name) && (row.text == row_new.text) &&
                                (Math.Abs((row_new.created_at - row.created_at).Hours) < VALID_SPAN))
                            {
                                valid = false;
                                break;
                            }
                        }
                    }
                    if (worker.CancellationPending)
                    {
                        return null;
                    }
                    else if (valid)
                    {
                        DataSetChattr.TableStatusDataTable changes = null;
                        m_MutexStatus.WaitOne();
                        try
                        {
                            InsertStatus(row_new, new ArrayList());
                            RequestProfileImage(row_new.user_profile_image);
                            if (worker.CancellationPending)
                            {
                                return null;
                            }
                            DateTime now = DateTime.Now;
                            Debug.WriteLine(now + " GetChanges start");
                            changes = dataSetChattr.TableStatus.GetChanges() as DataSetChattr.TableStatusDataTable;
                            dataSetChattr.TableStatus.AcceptChanges();
                            DateTime end = DateTime.Now;
                            TimeSpan span = end - now;
                            Debug.WriteLine(end + " GetChanges end: " + span);
                        }
                        catch (SystemException ex)
                        {
                            Debug.WriteLine("m_MutexStatus.WaitOne() Error\n" +
                                "in DoBackGround_XmppMessage()" +
                                ex.Message + "\n" + "Source: " + ex.Source);
                        }
                        finally
                        {
                            m_MutexStatus.ReleaseMutex();
                            if (valid)
                            {
                                worker.ReportProgress((int)ProgressOp.LIST_BEGIN_UPDATE);
                                worker.ReportProgress((int)ProgressOp.LIST_INSERT, new IListItem(row_new.id, 0));
                                SetSelectedID(selected, worker);
                                worker.ReportProgress((int)ProgressOp.LIST_END_UPDATE);
                                worker.ReportProgress((int)ProgressOp.NUM_STATUS);
                                result.Flags = row_new.flags;
                                result.List.Add(MakePopup(row_new));
                                UpdateSearchTable(worker, 0, selectedSearch);
                            }
                            if (changes != null)
                            {
                                DateTime now = DateTime.Now;
                                Debug.WriteLine(now + " Update start: " + changes.Rows.Count);
                                try
                                {
                                    tableStatusTableAdapter.Update(changes);
                                }
                                catch (SystemException ex)
                                {
                                    Debug.WriteLine("tableStatusTableAdapter.Update(changes) Error\n" +
                                        "in DoBackGround_XmppMessage()" +
                                        ex.Message + "\n" + "Source: " + ex.Source);
                                }
                                DateTime end = DateTime.Now;
                                TimeSpan span = end - now;
                                Debug.WriteLine(end + " Update end: " + span);
                            }
                        }
                    }
                    else
                    {
                        Debug.WriteLine("Cancel");
                    }
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("DoBackGround_XmppMessage() Error\n" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
            }
            return result;
        }

        private string GetError(HttpWebResponse res)
        {
            if (res != null)
            {
                return " m" + res.StatusDescription + "n";
            }
            SystemException ex = TwitterAPI.GetLastError();
            string error = ex.Message;
            Match m;
            if ((m = new Regex(@"^.*?\: (.*)$").Match(error)).Success)
            {
                error = m.Groups[1].Value;
            }
            return " m" + error + "n";
        }

        private IResultGetFriends RetrieveStatuses(XmlElement root, BackgroundWorker worker, string unit)
        {
            IResultGetFriends result = new IResultGetFriends();
            if ((root.Name == "statuses") && (root.Attributes["type"].Value == "array"))
            {
                DataSetChattr.TableStatusDataTable changes = null;
                int retrieve = 0;
                int replace = 0;
                string timeline = "friends timeline";
                if (unit != "status")
                {
                    timeline = "replies";
                }
                Debug.WriteLine(timeline +": " + root.ChildNodes.Count + " retrieve");
                m_MutexStatus.WaitOne();
                long selected = GetSelectedID();
                long selectedSearch = GetSelectedSearchID();
                ArrayList list = new ArrayList();
                try
                {
                    for (int i = root.ChildNodes.Count - 1; i >= 0; i--)
                    {
                        XmlNode node = root.ChildNodes[i];
                        if (worker.CancellationPending)
                        {
                            return null;
                        }
                        DataSetChattr.TableStatusRow row_new = ParseStatus(node);
                        if (row_new != null)
                        {
                            StatusParser parser = new StatusParser(row_new.text);
                            ParseUserName(parser, row_new);
                            ParseTinyUrl(parser);
                            ParseKeyword(parser, row_new);
                            int cnt = dataSetChattr.TableStatus.Rows.Count;
                            if (cnt == 0)
                            {
                                AddStatus(row_new, list);
                            }
                            else
                            {
                                int x = cnt;
                                if (row_new.id >= 0)
                                {
                                    for (x = 0; x < cnt; x++)
                                    {
                                        DataSetChattr.TableStatusRow row = (DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[x];
                                        if ((row.RowState != DataRowState.Deleted) &&
                                            (row.id < 0) &&
                                            (row.user_screen_name == row_new.user_screen_name) &&
                                            (row.text == row_new.text))
                                        {
                                            if (worker.CancellationPending)
                                            {
                                                return null;
                                            }
                                            selected = ReplaceStatus(row_new, x, selected, list);
                                            replace++;
                                            break;
                                        }
                                        else if ((row.RowState != DataRowState.Deleted) &&
                                            (row_new.created_at - row.created_at).Hours >= 1)
                                        {
                                            x = cnt;
                                        }
                                    }
                                }
                                if (worker.CancellationPending)
                                {
                                    return null;
                                }
                                else if (x >= cnt)
                                {
                                    result.List.Add(MakePopup(row_new));
                                    result.Flags |= row_new.flags;
                                    InsertStatus(row_new, list);
                                    RequestProfileImage(row_new.user_profile_image);
                                }
                            }
                            retrieve++;
                        }
                    }
                    Debug.WriteLine(timeline + ": " + retrieve + " / " + replace);
                    if (retrieve > 0)
                    {
                        if (worker.CancellationPending)
                        {
                            return null;
                        }
                        DateTime now = DateTime.Now;
                        Debug.WriteLine(now + " GetChanges start");
                        try
                        {
                            changes = dataSetChattr.TableStatus.GetChanges() as DataSetChattr.TableStatusDataTable;
                            dataSetChattr.TableStatus.AcceptChanges();
                        }
                        catch (SystemException ex)
                        {
                            Debug.WriteLine("dataSetChattr.TableStatus.GetChanges() Error\n" +
                                "in RetrieveStatuses()" +
                                ex.Message + "\n" + "Source: " + ex.Source);
                        }
                        DateTime end = DateTime.Now;
                        TimeSpan span = end - now;
                        Debug.WriteLine(end + " GetChanges end: " + span);
                    }
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("m_MutexStatus.WaitOne() Error\n" +
                        "in RetrieveStatuses()" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
                finally
                {
                    m_MutexStatus.ReleaseMutex();
                }
                Debug.WriteLine("list: " + list.Count);
                worker.ReportProgress((int)ProgressOp.LIST_BEGIN_UPDATE);
                int delete = 0;
                foreach (IListItem item in list)
                {
                    if (item.GetType() == typeof(IListItemAdd))
                    {
                        worker.ReportProgress((int)ProgressOp.LIST_ADD, new IListItem((IListItem)item));
                    }
                    else if (item.GetType() == typeof(IListItemInsert))
                    {
                        worker.ReportProgress((int)ProgressOp.LIST_INSERT, new IListItem((IListItem)item));
                    }
                    else if (item.GetType() == typeof(IListItemDelete))
                    {
                        worker.ReportProgress((int)ProgressOp.LIST_DELETE, item.Index);
                        delete++;
                    }
                }
                if (delete > 0)
                {
                    worker.ReportProgress((int)ProgressOp.LIST_REMOVES, delete);
                }
                SetSelectedID(selected, worker);
                worker.ReportProgress((int)ProgressOp.LIST_END_UPDATE);
                worker.ReportProgress((int)ProgressOp.NUM_STATUS);
                if (worker.CancellationPending)
                {
                    return null;
                }
                if (changes != null)
                {
                    DateTime now = DateTime.Now;
                    Debug.WriteLine(now + " Update start: " + changes.Rows.Count);
                    try
                    {
                        tableStatusTableAdapter.Update(changes);
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("tableStatusTableAdapter.Update(table) Error\n" +
                            "in RetrieveStatuses()" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    DateTime end = DateTime.Now;
                    TimeSpan span = end - now;
                    Debug.WriteLine(end + " Update end: " + span);
                }
                if (worker.CancellationPending)
                {
                    return null;
                }
                if (worker.CancellationPending)
                {
                    return null;
                }
                if (retrieve > 0)
                {
                    if (!UpdateSearchTable(worker, replace, selectedSearch))
                    {
                        return null;
                    }
                }
                if ((retrieve - replace) == 0)
                {
                    worker.ReportProgress((int)ProgressOp.STATUS_BAR, "");
                }
                else
                {
                    worker.ReportProgress((int)ProgressOp.STATUS_BAR, (retrieve - replace) + " " + unit + "擾܂");
                }
            }
            return result;
        }

        private IPopup MakePopup(DataSetChattr.TableStatusRow row)
        {
            return new IPopup(row.flags, row.user_screen_name, row.text);
        }

        private bool UpdateSearchTable(BackgroundWorker worker, int replace, long selected)
        {
            if (m_TblSearch != null)
            {
                m_MutexSearch.WaitOne();
                try
                {
                    m_TblSearch = SearchStatus(m_Search);
                    if (m_TblSearch == null)
                    {
                        listBoxSearch.Items.Clear();
                    }
                    else
                    {
                        if (listBoxSearch.Items.Count != m_TblSearch.Rows.Count)
                        {
                            worker.ReportProgress((int)ProgressOp.LIST_SEARCH_BEGIN_UPDATE);
                            while (listBoxSearch.Items.Count > m_TblSearch.Rows.Count)
                            {
                                if (worker.CancellationPending)
                                {
                                    return false;
                                }
                                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_REMOVE, 0);
                            }
                            int i = 0;
                            while (listBoxSearch.Items.Count < m_TblSearch.Rows.Count)
                            {
                                if (worker.CancellationPending)
                                {
                                    return false;
                                }
                                DataSetChattr.TableStatusRow row = m_TblSearch.Rows[i] as DataSetChattr.TableStatusRow;
                                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_INSERT, new IListItem(row.id, 0));
                            }
                            if (worker.CancellationPending)
                            {
                                return false;
                            }
                            SetSelectedSearchID(selected, worker);
                            worker.ReportProgress((int)ProgressOp.LIST_SEARCH_END_UPDATE, 0);
                        }
                        if (m_TblSearch.Rows.Count == 0)
                        {
                            m_TblSearch = null;
                        }
                    }
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("UpdateSearchTable() Error\n" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                }
                finally
                {
                    m_MutexSearch.ReleaseMutex();
                }
            }
            return true;
        }

        private DataSetChattr.TableStatusDataTable SearchStatus(string pattern)
        {
            if (toolStripMenuItemUser.Checked)
            {
                return tableStatusTableAdapter.GetDataBySearchUser(m_Search);
            }
            else if (toolStripMenuItemStatus.Checked)
            {
                return tableStatusTableAdapter.GetDataBySearchStatus(m_Search);
            }
            else if (toolStripMenuItemLocation.Checked)
            {
                return tableStatusTableAdapter.GetDataBySearchStatus(m_Search);
            }
            else if (toolStripMenuItemSource.Checked)
            {
                return tableStatusTableAdapter.GetDataBySearchSource(m_Search);
            }
            return tableStatusTableAdapter.GetDataBySearchAll(m_Search);
        }

        private DataSetChattr.TableStatusRow ParseStatus(XmlNode node)
        {
            String str = node["id"].InnerText;
            if ((str == null) || (str == ""))
            {
                return null;
            }
            Int64 id = Int64.Parse(node["id"].InnerText);
            if (dataSetChattr.TableStatus.Rows.Find(id) != null)
            {
                return null;
            }
            String created_at = node["created_at"].InnerText;
            Regex r = new Regex(@"^(.*) (\+0000) (.*)$");
            Match m = r.Match(created_at);
            if (m.Success)
            {
                created_at = m.Groups[1].Value + " " + m.Groups[3].Value + " +0";
            }
            DateTime at = DateTime.ParseExact(created_at, "ddd MMM dd HH:mm:ss yyyy z", new CultureInfo("en-US")).ToUniversalTime();
            String text = Regex.Replace(Regex.Replace(node["text"].InnerText, @"&gt;", ">"), "&lt;", "<");
            String source_url = "";
            String source = node["source"].InnerText;
            r = new Regex(@"<.*?=""(.*?)"">(.*)<");
            m = r.Match(source);
            if (m.Success)
            {
                source_url = m.Groups[1].Value;
                source = m.Groups[2].Value;
            }
            Int64 user_id = Int64.Parse(node["user"]["id"].InnerText);
            String user_name = node["user"]["name"].InnerText;
            String screen_name = node["user"]["screen_name"].InnerText;
            String location = node["user"]["location"].InnerText;
            String profile_image = node["user"]["profile_image_url"].InnerText;
            r = new Regex(@"^(.*)\?");
            m = r.Match(profile_image);
            if (m.Success)
            {
                profile_image = m.Groups[1].Value;
            }
            DataSetChattr.TableStatusRow row_new = dataSetChattr.TableStatus.NewTableStatusRow();
            row_new.ItemArray = new Object[] {
                    id,
                    at,
                    text,
                    source,
                    source_url,
                    user_id,
                    user_name,
                    screen_name,
                    location,
                    profile_image,
                    FLAG_NEW
                };
            return row_new;
        }

        private DataSetChattr.TableStatusRow ParseXmpp(agsXMPP.protocol.client.Message msg)
        {
            if ((msg.From.Server != "twitter.com") || (msg.From.User != "twitter"))
            {
                return null;
            }
            DateTime at = DateTime.Now.ToUniversalTime();
            Int64 id = Int64.Parse(at.ToString("-yyyyMMddHHmmssfff"));
            String text = "";
            String source_url = "";
            String source = "Jabber/XMPP";
            Int64 user_id = -1;
            String user_name = "";
            String screen_name = "";
            String location = "";
            String profile_image = "";
            Regex r;
            Match m;
            bool isMsg = false;
            for (int i = 1; i < msg.ChildNodes.Count; i++)
            {
                agsXMPP.Xml.Dom.Element node = (agsXMPP.Xml.Dom.Element)msg.ChildNodes.Item(i);
                Debug.WriteLine(node.TagName + " :\'" + node.Value + "'");
                switch (node.TagName)
                {
                    case "body":
                        r = new Regex(@"^\S+: (.*)$");
                        m = r.Match(node.Value);
                        if (m.Success)
                        {
                            text = m.Groups[1].Value;
                            isMsg = true;
                        }
                        break;
                    case "screen_name":
                        screen_name = node.Value;
                        break;
                    case "profile_image":
                        profile_image = node.Value;
                        r = new Regex(@"^(.*)\?");
                        m = r.Match(profile_image);
                        if (m.Success)
                        {
                            profile_image = m.Groups[1].Value;
                        }
                        break;
                    default:
                        Debug.WriteLine("ParseXmpp: " + node.TagName + "[" + node.Value + "]");
                        break;
                }
            }
            if (!isMsg)
            {
                return null;
            }
            DataSetChattr.TableStatusRow row_new = dataSetChattr.TableStatus.NewTableStatusRow();
            row_new.ItemArray = new Object[] {
                id,
                at,
                text,
                source,
                source_url,
                user_id,
                user_name,
                screen_name,
                location,
                profile_image,
                FLAG_NEW
            };
            return row_new;
        }

        private void ParseUserName(StatusParser parser, DataSetChattr.TableStatusRow row)
        {
            string[] users = parser.Users;
            if (users != null)
            {
                foreach (string user in users)
                {
                    if (user == Properties.Settings.Default.UserName)
                    {
                        row.flags |= FLAG_INME;
                    }
                }
            }
        }

        private void ParseTinyUrl(StatusParser parser)
        {
            string[] urls = parser.URLs;
            if (urls != null)
            {
                foreach (string url in urls)
                {
                    RequestTinyUrl(url);
                }
            }
        }

        private bool ParseKeyword(StatusParser parser, DataSetChattr.TableStatusRow row)
        {
            foreach (string keyword in Regex.Split(Properties.Settings.Default.Keywords, "\n"))
            {
                if ((keyword != "") && (keyword != "\r"))
                {
                    Regex r = new Regex(Regex.Replace(keyword, "\r", ""));
                    if ((r.Match(row.user_screen_name)).Success)
                    {
                        row.flags |= FLAG_KEYWORDS;
                        return true;
                    }
                    if ((r.Match(row.user_name)).Success)
                    {
                        row.flags |= FLAG_KEYWORDS;
                        return true;
                    }
                    if ((r.Match(row.text)).Success)
                    {
                        row.flags |= FLAG_KEYWORDS;
                        return true;
                    }
                }
            }
            return false;
        }

        private long GetSelectedID()
        {
            long select = 0;
            int idx = (int)Invoke(new DelegateGetSelectedIdx(GetSelectedIdx));
            if (idx != -1)
            {
                select = ((DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[idx]).id;
            }
            return select;
        }

        private long GetSelectedSearchID()
        {
            long select = 0;
            if (m_TblSearch != null)
            {
                int idx = (int)Invoke(new DelegateGetSelectedIdx(GetSelectedSearchIdx));
                if (idx != -1)
                {
                    select = ((DataSetChattr.TableStatusRow)m_TblSearch.Rows[idx]).id;
                }
            }
            return select;
        }

        delegate int DelegateGetSelectedIdx();
        private int GetSelectedIdx()
        {
            return listBoxStatuses.SelectedIndex;
        }

        private int GetSelectedSearchIdx()
        {
            return listBoxSearch.SelectedIndex;
        }

        private void SetSelectedID(long select, BackgroundWorker worker)
        {
            if (select == 0)
            {
                worker.ReportProgress((int)ProgressOp.LIST_SELECT, -1);
                worker.ReportProgress((int)ProgressOp.LIST_SELECT, 0);
            }
            else
            {
                int cnt = dataSetChattr.TableStatus.Rows.Count;
                for (int i = 0; i < cnt; i++)
                {
                    if (((DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[i]).id == select)
                    {
                        worker.ReportProgress((int)ProgressOp.LIST_SELECT, -1);
                        worker.ReportProgress((int)ProgressOp.LIST_SELECT, i);
                        break;
                    }
                }
            }
        }

        private void SetSelectedSearchID(long select, BackgroundWorker worker)
        {
            if (select == 0)
            {
                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_SELECT, -1);
                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_SELECT, 0);
            }
            else
            {
                int cnt = m_TblSearch.Rows.Count;
                for (int i = 0; i < cnt; i++)
                {
                    if (((DataSetChattr.TableStatusRow)m_TblSearch.Rows[i]).id == select)
                    {
                        worker.ReportProgress((int)ProgressOp.LIST_SEARCH_SELECT, -1);
                        worker.ReportProgress((int)ProgressOp.LIST_SEARCH_SELECT, i);
                        break;
                    }
                }
            }
        }

        private void AddStatus(DataSetChattr.TableStatusRow row_new, ArrayList list)
        {
            AddRow(row_new);
            list.Add(new IListItemAdd(row_new.id, 0));
        }

        private void InsertStatus(DataSetChattr.TableStatusRow row_new, ArrayList list)
        {
            int cnt = dataSetChattr.TableStatus.Rows.Count;
            int x;
            for (x = 0; x < cnt; x++)
            {
                DataSetChattr.TableStatusRow row = (DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[x];
                if (row.RowState == DataRowState.Deleted)
                {
                }
                else if (row_new.created_at > row.created_at)
                {
                    break;
                }
                else if ((row_new.id >= 0) && (row.id >= 0) && (row_new.id > row.id))
                {
                    break;
                }
                else if ((row_new.id < 0) && (row.id < 0) && (row_new.id < row.id))
                {
                    break;
                }
                else if ((row_new.id < 0) && (row.id >= 0))
                {
                    break;
                }
            }
            InsertRow(row_new, x);
            list.Add(new IListItemInsert(row_new.id, x));
            Debug.WriteLine(row_new.user_screen_name + ": \'" + row_new.text + "\'");
        }

        private long ReplaceStatus(DataSetChattr.TableStatusRow row_new, int pos, long select, ArrayList list)
        {
            if (((DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[pos]).id == select)
            {
                select = row_new.id;
            }
            row_new.flags = ((DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[pos]).flags;
            dataSetChattr.TableStatus.Rows[pos].Delete();
            list.Add(new IListItemDelete(0, pos));
            InsertStatus(row_new, list);
            return select;
        }

        private void AddRow(DataSetChattr.TableStatusRow row)
        {
            try
            {
                dataSetChattr.TableStatus.Rows.Add(row);
            }
            catch (SystemException ex)
            {
                dataSetChattr.TableStatus.RejectChanges();
                Debug.WriteLine("dataSetChattr.TableStatus.Rows.Add() Error\n" +
                    "in AddRow()\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        private void InsertRow(DataSetChattr.TableStatusRow row, int pos)
        {
            try
            {
                dataSetChattr.TableStatus.Rows.InsertAt(row, pos);
            }
            catch (SystemException ex)
            {
                dataSetChattr.TableStatus.RejectChanges();
                Debug.WriteLine("dataSetChattr.TableStatus.Rows.Add() Error\n" +
                    "in InsertRow()\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        private static void ThreadGetProfileImage(object param)
        {
            if (param.GetType() == typeof(IGetHttp))
            {
                IGetHttp arg = param as IGetHttp;
                HttpWebResponse res = TwitterAPI.Get(arg.Url);
                arg.Response = res;
            }
        }

        private object DoBackGround_GetProfileImage(String arg, BackgroundWorker worker)
        {
            object result = null;
            if (dataSetChattr.TableProfileImage.Rows.Find(arg) == null)
            {
                IGetHttp igh = new IGetHttp(arg);
                Thread t = new Thread(ThreadGetProfileImage);
                if (!RunThread(t, worker, igh))
                {
                    return null;
                }
                HttpWebResponse res = igh.Response;
                if ((res != null) && (res.StatusCode == HttpStatusCode.OK))
                {
                    try
                    {
                        Bitmap bmp = (Bitmap)Bitmap.FromStream(res.GetResponseStream());
                        DataSetChattr.TableProfileImageRow row_new = dataSetChattr.TableProfileImage.NewTableProfileImageRow();
                        row_new.ItemArray = new object[] { arg, Bitmap2ByteArray(bmp), bmp.Width, bmp.Height };
                        dataSetChattr.TableProfileImage.Rows.Add(row_new);
                        Debug.WriteLine("ProfileImag :" + arg);
                        if (worker.CancellationPending)
                        {
                            return null;
                        }
                        CommitTableProfileImage();
                    }
                    catch (SystemException ex)
                    {
                        dataSetChattr.TableProfileImage.RejectChanges();
                        Debug.WriteLine("DoBackGround_GetProfileImage() Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                    finally
                    {
                        res.Close();
                    }
                    int cnt = dataSetChattr.TableStatus.Rows.Count;
                    int profile = 0;
                    DataSetChattr.TableStatusDataTable tbl = tableStatusTableAdapter.GetDataByProfileImage(arg);
                    if (tbl.Rows.Count > 0)
                    {
                        bool begin = false;
                        foreach (DataSetChattr.TableStatusRow row in tbl.Rows)
                        {
                            if (worker.CancellationPending)
                            {
                                return null;
                            }
                            for (int i = 0; i < cnt; i++)
                            {
                                if (worker.CancellationPending)
                                {
                                    return null;
                                }
                                DataSetChattr.TableStatusRow rowx = (DataSetChattr.TableStatusRow)dataSetChattr.TableStatus.Rows[i];
                                if (row.id == rowx.id)
                                {
                                    Debug.WriteLine("ProfileImag :" + row.user_screen_name + ": " + row.text);
                                    profile++;
                                    if (!begin)
                                    {
                                        begin = true;
                                        worker.ReportProgress((int)ProgressOp.LIST_BEGIN_UPDATE);
                                    }
                                    worker.ReportProgress((int)ProgressOp.LIST_REPLACE, i);
                                }
                            }
                        }
                        if (begin)
                        {
                            worker.ReportProgress((int)ProgressOp.LIST_END_UPDATE);
                        }
                    }
                    Debug.WriteLine("ProfileImag :" + profile + " / " + tbl.Rows.Count);
                }
            }
            return result;
        }

        private static void ThreadGetTinyUrl(object param)
        {
            if (param.GetType() == typeof(IGetHttp))
            {
                IGetHttp arg = param as IGetHttp;
                HttpWebRequest req = (HttpWebRequest)WebRequest.Create(arg.Url);
                req.Method = "HEAD";
                req.KeepAlive = false;
                req.CookieContainer = null;
                req.Timeout = 5000;
                try
                {
                    arg.Response = (HttpWebResponse)req.GetResponse();
                }
                catch (SystemException ex)
                {
                    Debug.WriteLine("ThreadGetTinyUrl(\'" + arg + "\') Error\n" +
                        ex.Message + "\n" + "Source: " + ex.Source);
                    if (ex.GetType() == typeof(WebException))
                    {
                        arg.Exception = ex;
                    }
                }
            }
        }

        private object DoBackGround_GetTinyUrl(String arg, BackgroundWorker worker)
        {
            object result = null;
            if (dataSetChattr.TableTinyUrl.Rows.Find(arg) == null)
            {
                Debug.WriteLine("TinyUrl: Get(" + arg + ")");
                IGetHttp igh = new IGetHttp(arg);
                Thread t = new Thread(ThreadGetTinyUrl);
                if (!RunThread(t, worker, igh))
                {
                    return null;
                }
                HttpWebResponse res = igh.Response;
                SystemException exc = igh.Exception;
                if (exc != null)
                {
                    if (exc.GetType() == typeof(WebException))
                    {
                        WebException wex = (WebException)exc;
                        if (wex.Status == WebExceptionStatus.Timeout)
                        {
                            RequestTinyUrl(arg);
                        }
                    }
                }
                else if ((res != null) && (res.StatusCode == HttpStatusCode.OK))
                {
                    try
                    {
                        res.Close();
                        dataSetChattr.TableTinyUrl.Rows.Add(arg, res.ResponseUri.AbsoluteUri);
                        if (worker.CancellationPending)
                        {
                            return null;
                        }
                        tableTinyUrlTableAdapter.Update(dataSetChattr.TableTinyUrl);
                        dataSetChattr.TableTinyUrl.AcceptChanges();
                        Debug.WriteLine("TinyUrl: " + arg + " = " + res.ResponseUri.AbsoluteUri);
                    }
                    catch (SystemException ex)
                    {
                        Debug.WriteLine("DoBackGround_GetTinyUrl(\'" + arg + "\') Error\n" +
                            ex.Message + "\n" + "Source: " + ex.Source);
                    }
                }
            }
            return result;
        }

        private object DoBackGround_DeleteAll(String arg, BackgroundWorker worker)
        {
            Debug.WriteLine("DoBackGround_DeleteAll");
            // Statuses
            m_MutexStatus.WaitOne();
            try
            {
                worker.ReportProgress((int)ProgressOp.LIST_BEGIN_UPDATE);
                worker.ReportProgress((int)ProgressOp.LIST_CLEAR);
                worker.ReportProgress((int)ProgressOp.LIST_END_UPDATE);
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("DoBackGround_DeleteAll() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
            finally
            {
                m_MutexStatus.ReleaseMutex();
            }
            // Search
            m_MutexSearch.WaitOne();
            try
            {
                m_TblSearch = null;
                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_BEGIN_UPDATE);
                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_CLEAR);
                worker.ReportProgress((int)ProgressOp.LIST_SEARCH_END_UPDATE);
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("DoBackGround_Delete() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
            finally
            {
                m_MutexSearch.ReleaseMutex();
            }
            worker.ReportProgress((int)ProgressOp.LIST_ENABLE);
            worker.ReportProgress((int)ProgressOp.NUM_STATUS);
            // Drop
            SQLite.DropTable(SQL_PATH);
            SQLite.CreateDB(SQL_PATH);
            SQLite.Vacuum(SQL_PATH);
            // Fill
            tableStatusTableAdapter.Fill(dataSetChattr.TableStatus);
            tableProfileImageTableAdapter.Fill(dataSetChattr.TableProfileImage);
            tableTinyUrlTableAdapter.Fill(dataSetChattr.TableTinyUrl);
            m_Load = 0;
            m_Loading = 0;
            return null;
        }

        private void CommitTableProfileImage()
        {
            try
            {
                tableProfileImageTableAdapter.Update(dataSetChattr.TableProfileImage);
                dataSetChattr.TableProfileImage.AcceptChanges();
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("CommitTableProfileImage() Error\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
        }

        private static void ThreadUpdate(object param)
        {
            if (param.GetType() == typeof(IGetTimeline))
            {
                IGetTimeline arg = param as IGetTimeline;
                HttpWebResponse res = TwitterAPI.Update(arg.User, arg.Pass, arg.Option);
                arg.Response = res;
            }
        }

        private object DoBackGround_Update(String arg, BackgroundWorker worker)
        {
            IResultUpdate result = null;
            worker.ReportProgress((int)ProgressOp.STATUS_BAR, "Xe[^XM...");
            IGetTimeline igt = new IGetTimeline(Properties.Settings.Default.UserName, m_Prefs.Password, arg);
            Thread t = new Thread(ThreadUpdate);
            if (!RunThread(t, worker, igt))
            {
                return null;
            }
            HttpWebResponse res = igt.Response;
            if ((res != null) && (res.StatusCode == HttpStatusCode.OK))
            {
                worker.ReportProgress((int)ProgressOp.STATUS_BAR, "M");
                result = new IResultUpdate(true);
            }
            else
            {
                worker.ReportProgress((int)ProgressOp.STATUS_BAR, "MG[" + GetError(res));
                result = new IResultUpdate(false);
            }
            return (object)result;
        }

        private String NumStatus()
        {
            String label = listBoxStatuses.Items.Count + " ";
            if (m_TblSearch != null)
            {
                label = listBoxSearch.Items.Count + " / " + label;
            }
            return label;
        }

        private void SetToolStripStatusLabelText(string text)
        {
            toolStripStatusLabel.Text = text;
            if (text == "")
            {
                notifyIcon.Text = "Chattr";
            }
            else
            {
                notifyIcon.Text = "Chattr\n" + text;
            }
        }

        private void NotifyFlash(IResultGetFriends result)
        {
            if (notifyIcon.Visible)
            {
                bool flash = false;
                foreach (IPopup popup in result.List)
                {
                    switch (Properties.Settings.Default.PopupKind)
                    {
                        case NOTIFY_INME:
                            flash = ((popup.Flags & (FLAG_INME | FLAG_NEW)) == (FLAG_INME | FLAG_NEW));
                            break;
                        case NOTIFY_KEYWORDS:
                            flash = ((popup.Flags & (FLAG_KEYWORDS | FLAG_NEW)) == (FLAG_KEYWORDS | FLAG_NEW));
                            break;
                        default:
                            flash = ((popup.Flags & FLAG_NEW) != 0);
                            break;
                    }
                    if (flash)
                    {
                        m_Popup.Add(popup.User, popup.Text);
                    }
                }
            }
            else
            {
                bool flash = false;
                if (Properties.Settings.Default.NotifyTaskbar)
                {
                    switch (Properties.Settings.Default.NotifyTaskbarKind)
                    {
                        case NOTIFY_INME:
                            flash = ((result.Flags & (FLAG_INME | FLAG_NEW)) == (FLAG_INME | FLAG_NEW));
                            break;
                        case NOTIFY_KEYWORDS:
                            flash = ((result.Flags & (FLAG_KEYWORDS | FLAG_NEW)) == (FLAG_KEYWORDS | FLAG_NEW));
                            break;
                        default:
                            flash = ((result.Flags & FLAG_NEW) != 0);
                            break;
                    }
                }
                if (flash)
                {
                    FlashWindow(this.Handle, true);
                }
            }
        }

        private void Quit()
        {
            CloseXmpp();
            timer.Enabled = false;
            DataTable change = null;
            try
            {
                change = dataSetChattr.TableStatus.GetChanges();
                if ((change != null) && (change.Rows.Count != 0))
                {
                    tableStatusTableAdapter.Update(dataSetChattr.TableStatus);
                    dataSetChattr.TableStatus.AcceptChanges();
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("Commit Error (dataSetChattr.TableStatus)\n" +
                    "in Quit()\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
            try
            {
                change = dataSetChattr.TableProfileImage.GetChanges();
                if ((change != null) && (change.Rows.Count != 0))
                {
                    tableProfileImageTableAdapter.Update(dataSetChattr.TableProfileImage);
                    dataSetChattr.TableProfileImage.AcceptChanges();
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("Commit Error (dataSetChattr.TableProfileImage)\n" +
                    "in Quit()\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
            try
            {
                change = dataSetChattr.TableTinyUrl.GetChanges();
                if ((change != null) && (change.Rows.Count != 0))
                {
                    tableTinyUrlTableAdapter.Update(dataSetChattr.TableTinyUrl);
                    dataSetChattr.TableTinyUrl.AcceptChanges();
                }
            }
            catch (SystemException ex)
            {
                Debug.WriteLine("Commit Error (dataSetChattr.TableTinyUrl)\n" +
                    "in Quit()\n" +
                    ex.Message + "\n" + "Source: " + ex.Source);
            }
            Properties.Settings.Default.SelectedID = GetSelectedID();
            DeleteOldStatuses();
            UpdateLocation();
            m_Prefs.Save();
            m_fBgBusy = false;
        }

        private void DeleteOldStatuses()
        {
            DateTime now = DateTime.Today.ToUniversalTime();
            DateTime date = GetOldTime(now);
            if (date != DateTime.MinValue)
            {
                SQLite.DeleteOldStatuses(SQL_PATH, date);
            }
        }

        private DateTime GetOldTime(DateTime date)
        {
            switch ((LogLimit)Properties.Settings.Default.LogLimit)
            {
                case LogLimit.DAY1:
                    return date.AddDays(-1);
                case LogLimit.DAY3:
                    return date.AddDays(-3);
                case LogLimit.WEEK1:
                    return date.AddDays(-7);
                case LogLimit.WEEK2:
                    return date.AddDays(-14);
                case LogLimit.MONTH1:
                    return date.AddMonths(-1);
                case LogLimit.MONTH2:
                    return date.AddMonths(-2);
                case LogLimit.MONTH3:
                    return date.AddMonths(-3);
                case LogLimit.MONTH6:
                    return date.AddMonths(-6);
                case LogLimit.YEAR1:
                    return date.AddYears(-1);
            }
            return DateTime.MinValue;
        }
    }
}
