<?php
//////////////////////////////////////////////
// 指定したzipに対するパーツ一覧(JSON)を取得するCGI
//////////////////////////////////////////////
require_once __DIR__ . '/common.php';
require_once __DIR__ . '/calc_trip.php';

header("Content-type: application/json; charset=utf-8");

// DataGridのページャのページサイズとページ範囲、ソートカラム、順序等の情報を取得する
$page = intval(isset($_POST['page']) ? $_POST['page'] : 1);
$pageSize = intval(isset($_POST['rows']) ? $_POST['rows'] : 10);
$sortColumn = isset($_POST['sort']) ? $_POST['sort'] : 'id';
$sortOrder = isset($_POST['order']) ? ($_POST['order'] == 'asc' ? 'asc' : 'desc') : 'asc';
$tzoffset = intval(isset($_POST['tzoffset']) ? $_POST['tzoffset'] : 0);
$tagids = isset($_POST['tagids']) ? $_POST['tagids'] : [];

// パラメータの妥当性チェック
$valid_columns = [
    'id', 'partsid', 'partsname', 'title', 'author', 'fname', 'fsize', 'regdate', 'downloaded', 'license'
];
if (!in_array($sortColumn, $valid_columns)) {
    // 妥当なカラム名でなければサーバーエラーを返す
    header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error', true, 500);
    exit;
}

$strtagids = ',' . implode(',', $tagids) . ',';

$pdo = create_pdo();

// パーツ別タグ関係、ZIP別タグ関係から指定したtagidを含む全てのパーツリストを、そのパーツのZIPの情報とともに列挙する。
// (同一のpartsidがtagidにマッチした回数だけ繰り返し出現する。)
$fetch = $pdo->prepare("select tagid, zip_entries.id id, partsid, partsname, title,
    author, fname, fsize, regdate, downloaded, license from parts_entries
    inner join zip_entries on parts_entries.zipid = zip_entries.id inner join (
    select tagid, partsid from tag_parts_rel where instr(:strtagids, ',' || tagid || ',') <> 0
    union all
    select tagid, parts_entries.id partsid from tag_zip_rel
    inner join parts_entries on tag_zip_rel.zipid = parts_entries.zipid where instr(:strtagids, ',' || tagid || ',') <> 0
    ) C on parts_entries.id = c.partsid
    where not exists (select * from report_entries r where r.zipid = zip_entries.id and r.remote_addr = :remote_addr)");
$fetch->execute([
    'strtagids' => $strtagids,
    'remote_addr' => get_hostaddr()
]);

$rows = $fetch->fetchAll();
if ($rows === false) {
    // 読み取りに失敗した場合はサーバーエラーとする
    header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error', true, 500);
    exit;
}

$fetch = null;
$pdo = null;

// tagidが異なる同一のpartsidが繰り返されるレコードリストから
// partsidごとのtagidのリストを構築して、要求時のtagidsをすべて満たすパーツのレコードだけを返す
function filterMatchAll($rows, $tagids) {
    $partsMap = [];
    $partsTagMap = [];
    foreach ($rows as $row) {
        $partsid = $row['partsid'];
        $tagid = $row['tagid'];
    
        if (array_key_exists($partsid, $partsTagMap)) {
            $partsTagMap[$partsid][] = $tagid;
        } else {
            $partsMap[$partsid] = $row;
            $partsTagMap[$partsid] = [$tagid];
        }
    }
    
    $filteredRows = [];
    foreach ($partsTagMap as $partsId => $tags) {
        if (count(array_intersect($tags, $tagids)) == count($tagids)) { // 配列の中身が一致
            $row = $partsMap[$partsId];
            unset($row['tagid']); // tagidは削除しておく
            $filteredRows[] = $row;
        }
    }
    return $filteredRows;
}
$filteredRows = filterMatchAll($rows, $tagids);

// 指定したカラムでソートする
$sortkeys = [];
foreach ($filteredRows as $row) {
    $sortkeys[] = array_key_exists($sortColumn, $row) ? $row[$sortColumn] : null;
}
array_multisort($sortkeys, $sortOrder == 'asc' ? SORT_ASC : SORT_DESC, $filteredRows);

$startPos = ($page - 1) * $pageSize;
$result = [
    'total' => count($filteredRows),
    'rows' => array_slice($filteredRows, $startPos, $pageSize) // ページャの範囲で返す
];
echo json_encode($result);
?>