﻿using System;
using System.Collections.Generic;
using System.Text;
using ChaKi.Entity.Search;
using ChaKi.Entity.Corpora;
using NHibernate;
using System.Diagnostics;
using System.Collections;
using ChaKi.Service.Database;
using ChaKi.Entity.Kwic;

namespace ChaKi.Service.Search
{
    /// <summary>
    /// 複数のコーパスを縦断的に検索するサービスの基底となる抽象クラス。
    /// コーパスのオープンとセッション管理、および進捗表示の初期化を実装している。
    /// 
    /// 派生クラスでは、出力となるEntityを定義し、ExecuteSearchSessionが呼ばれたら
    /// そのコーパスを検索・Entityに結果を格納するコードを実装する。
    /// </summary>
    public abstract class SearchServiceBase : IServiceCommand
    {
        protected CommandProgress m_Progress;
        protected SearchConditionsSequence m_CondSeq;
        protected ISession m_Session;

        public SearchServiceBase(SearchConditionsSequence condSeq, CommandProgress progress)
        {
            m_CondSeq = condSeq;
            m_Progress = progress;
            m_Session = null;
        }

        protected abstract void ExecuteSearchSession(Corpus c);

        /// <summary>
        /// 複数コーパス検索の入口関数（派生クラスに共通の入口を与える）
        /// </summary>
        public void Begin()
        {
            if (m_Progress != null)
            {
                m_Progress.Reset();
            }
            foreach (Corpus c in m_CondSeq.Last.CorpusCond.Corpora)    //@todo: 絞り込みの場合、条件間でのCorpusの不一致はどうするか?
            {
                lock (c)
                {
                    // Corpus(DB)の種類に合わせてConfigurationをセットアップする
                    DBService dbs = DBService.Create(c.DBParam);
                    NHibernate.Cfg.Configuration cfg = dbs.GetConnection();
                    using (ISessionFactory factory = cfg.BuildSessionFactory())
                    {
                        try
                        {
                            m_Session = factory.OpenSession();
                            if (m_Progress != null)
                            {
                                // Nc, Ndを取得
                                IQuery query = m_Session.CreateQuery("select count(*) from Word");
                                long nc = (long)query.UniqueResult();
                                query = m_Session.CreateQuery("select count(*) from Lexeme");
                                long nd = (long)query.UniqueResult();
                                m_Progress.StartOnItem(c.Name, (int)nc, (int)nd, 0);
                            }
                            ExecuteSearchSession(c);
                        }
                        finally
                        {
                            m_Session.Close();
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 検索条件Boxで指定された各Lexemeの属性からLexicon検索を行い、
        /// マッチするLexemeをすべて得る。
        /// 結果を int(Box番号) -> IList(Lexemeリスト) のマップに格納して返す。
        /// </summary>
        protected LexemeResultSet QueryLexemeResultSet(IList<LexemeCondition> lexConds)
        {
            Debug.Assert(m_Session != null);

            LexemeResultSet resultset = new LexemeResultSet();
            //   条件内のLexeme属性をすべて制約として追加する。
            for (int i = 0; i < lexConds.Count; i++)
            {
                LexemeCondition lexcond = lexConds[i];
                string qstr = QueryBuilder.BuildLexemeQuery(lexcond);
                IQuery query = m_Session.CreateQuery(qstr);
                IList lexemeResult = query.List();

                if (lexemeResult.Count == 0)
                {
                    // 条件Box中にhitしないものがひとつでもあれば、結果は0
                    resultset.Clear();
                    return resultset;
                }
                resultset.Add(i, lexcond, lexemeResult);
            }
            return resultset;
        }
    }
}
