﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using ChaKi.Service.Search;
using ChaKi.Entity.Settings;
using System.IO;
using DependencyEdit;
using System.Threading;
using System.Reflection;
using System.Diagnostics;

namespace ChaKi
{
    internal sealed class Program
    {
        static public MainForm MainForm;
        static public string ProgramDir { get; private set; }
        static public string SettingDir { get; private set; }
        static private string m_SettingFile;
        static private Program instance;
        private string[] args;

        private Program(string[] args)
        {
            this.args = args;
        }
        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        [STAThread]
        static void Main(string[] args)
        {
            try
            {
                AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler(UnhandledException);
                instance = new Program(args);
                instance.Start();
            }
            catch (Exception ex)
            {
                ExceptionDialogBox dlg = new ExceptionDialogBox();
                dlg.Text = ex.ToString();
                dlg.ShowDialog();
                Process.GetCurrentProcess().Kill();
            }

        }

        public void Start()
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            // スプラッシュスクリーンを表示
            SplashScreen sscrn = new SplashScreen();
            sscrn.Show();
            sscrn.ProgressMsg = "Loading Entity Mappings...";
            sscrn.Update();
            Application.DoEvents();

            // NHibernateの初期化（エンティティマッピングをコンパイルさせる）
            try
            {
                SearchConfiguration.GetInstance().Initialize();
            }
            catch( Exception e )
            {
                ExceptionDialogBox dlg = new ExceptionDialogBox();
                dlg.Text = e.ToString();
                dlg.ShowDialog(); 
                return;
            }

            sscrn.ProgressMsg = "Loading User Settings and Corpus Information...";
            Application.DoEvents();

            // ユーザ設定のロード
            try
            {
                ProgramDir = Environment.CurrentDirectory;
                SettingDir = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
                SettingDir += @"\ChaKi.NET";
                if (!Directory.Exists(SettingDir))
                {
                    Directory.CreateDirectory(SettingDir);
                }
            }
            catch (Exception e)
            {
                MessageBox.Show(string.Format("{0}: {1}", e.Message, SettingDir));
            }
        
            m_SettingFile = SettingDir + @"\UserSettings.xml";
            try
            {
                UserSettings.GetInstance().Load(m_SettingFile);
            }
            catch (FileNotFoundException e)
            {
                // just reset content
            }
            catch (Exception e)
            {
                MessageBox.Show("Error reading user settings : {0}", e.Message);
            }

            Application.ApplicationExit += ApplicationExitHandler;
            Application.Idle += ApplicationIdleHandler;

            // メイン画面の表示
            try
            {
                MainForm = new MainForm();
                sscrn.Hide();
                Application.Run(MainForm);
            }
            catch (FileNotFoundException e)
            {
                string msg = string.Format("File(s) not found.\nPlease check installation.\n\n{0}", e.Message);
                MessageBox.Show(msg, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error );
                return;
            }
        }

        static void ApplicationExitHandler(object sender, EventArgs e)
        {
            Application.Idle -= ApplicationIdleHandler;

            UserSettings.GetInstance().Save(m_SettingFile);
        }

        static void ApplicationIdleHandler(object sender, EventArgs e)
        {
            MainForm.IdleUpdate();
            DepEditControl.UIUpdate();
        }

        private static void UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            ExceptionDialogBox dlg = new ExceptionDialogBox();
            dlg.Text = e.ExceptionObject.ToString();
            dlg.ShowDialog();
            Process.GetCurrentProcess().Kill();
        }
    }
}