unit uFileEmbedder;

interface

uses
  Windows, Classes;

type
  TFileEmbedder = class(TComponent)
  private
    // Variables
    FFileName: String;
    FData: TMemoryStream;
    FCompressed: Boolean;
    // Methods
    procedure ReadData(vStream: TStream);
    procedure WriteData(vStream: TStream);
    procedure SetFileName(const Value: String);
  protected
    // Methods
    procedure DefineProperties(vFiler: TFiler); override;
  public
    // Constructor & Destructor
    constructor Create(vOwner: TComponent); override;
    destructor Destroy; override;
    // Methods
    procedure SaveToFile(const vFileName: String);
    procedure SaveToStream(const vStream: TStream);
    // Property
    property Data: TMemoryStream read FData;
  published
    // Properties
    property FileName: String read FFileName write SetFileName;
    property Compressed: Boolean
      read FCompressed write FCompressed default True;
  end;

procedure Register;

implementation

uses
  SysUtils, Dialogs, ZlibEx;

procedure Register;
begin
  RegisterComponents('Sample', [TFileEmbedder]);
end;

{ TFileEmbedder }

constructor TFileEmbedder.Create(vOwner: TComponent);
(*
 * Tv  RXgN^
 *   vOwner  I[i[
 *)
begin
  inherited;

  FCompressed := True;
  FData := TMemoryStream.Create;
end;

procedure TFileEmbedder.DefineProperties(vFiler: TFiler);
begin
  inherited;

  vFiler.DefineBinaryProperty(
    'EmbededData',
    ReadData,
    WriteData,
    (FFileName <> ''));
end;

destructor TFileEmbedder.Destroy;
(*
 * Tv  fXgN^
 *)
begin
  FData.Free;

  inherited;
end;

procedure TFileEmbedder.ReadData(vStream: TStream);
(*
 * Tv  Data R|[lgXg[ǂݍ
 *   vStream  R|[lgXg[
 *)
var
  Size: Integer;
  Buff: Pointer;
  DecompressedBuf: Pointer;
  DecompressedSize: Integer;
begin
  vStream.Read(Size, SizeOf(Integer));

  FData.Clear;

  if (FCompressed) then begin
    GetMem(Buff, Size);
    try
      vStream.Read(PByte(Buff)^, Size);

      ZDecompress(Buff, Size, DecompressedBuf, DecompressedSize, 0);
      try
        FData.Size := DecompressedSize;

        CopyMemory(FData.Memory, DecompressedBuf, DecompressedSize);
      finally
        FreeMem(DecompressedBuf);
      end;
    finally
      FreeMem(Buff);
    end;
  end
  else
    FData.CopyFrom(vStream, Size);

  FData.Position := 0;
end;

procedure TFileEmbedder.SaveToFile(const vFileName: String);
(*
 * Tv  t@C FData ۑ
 *   vFileName  t@C
 *)
var
  FS: TFileStream;
begin
  FS := TFileStream.Create(vFileName, fmCreate);
  try
    SaveToStream(FS);
  finally
    FS.Free;
  end;
end;

procedure TFileEmbedder.SaveToStream(const vStream: TStream);
(*
 * Tv  Xg[ FData ۑ
 *   vStream  ۑXg[
 *)
begin
  vStream.CopyFrom(FData, 0);
end;

procedure TFileEmbedder.SetFileName(const Value: String);
(*
 * Tv  t@Cݒ肷
 *   Value  t@C
 *)
begin
  if (csDesigning in ComponentState) and (not FileExists(Value)) then begin
    ShowMessage(Value + ' was not found');
    Exit;
  end;

  FFileName := Value;
end;

procedure TFileEmbedder.WriteData(vStream: TStream);
(*
 * Tv  Data R|[lgXg[ɏ
 *   vStream  R|[lgXg[
 *)
var
  Size: Integer;
  Buff: Pointer;
  CompressedBuf: Pointer;
  CompressedSize: Integer;

  procedure Write0;
  begin
    Size := 0;
    vStream.Write(Size, SizeOf(Integer));
  end;

begin
  FData.Clear;

  if (not FileExists(FFileName)) then begin
    Write0;
    Exit;
  end;

  FData.LoadFromFile(FFileName);

  Size := FData.Size;

  if (Size = 0) then begin
    Write0;
    Exit;
  end;

  if (FCompressed) then begin
    Buff := FData.Memory;

    ZCompress(Buff, Size, CompressedBuf, CompressedSize);
    try
      vStream.Write(CompressedSize, SizeOf(Integer));
      vStream.Write(PByte(CompressedBuf)^, CompressedSize);
    finally
      FreeMem(CompressedBuf);
    end;
  end
  else begin
    vStream.Write(Size, SizeOf(Integer));
    vStream.CopyFrom(FData, 0);
  end;
end;

end.
