﻿/*
 * BezierCurves.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;

using bocoree;

namespace Boare.Cadencii {

    using boolean = System.Boolean;
    using Integer = Int32;

    /// <summary>
    /// AtatchedCurveをXMLシリアライズするためのクラス
    /// </summary>
    public class BezierCurves : ICloneable {
        public Vector<BezierChain> Dynamics;
        public Vector<BezierChain> Brethiness;
        public Vector<BezierChain> Brightness;
        public Vector<BezierChain> Clearness;
        public Vector<BezierChain> Opening;
        public Vector<BezierChain> GenderFactor;
        public Vector<BezierChain> PortamentoTiming;
        public Vector<BezierChain> VibratoRate;
        public Vector<BezierChain> VibratoDepth;
        public Vector<BezierChain> Harmonics;
        public Vector<BezierChain> FX2Depth;
        public Vector<BezierChain> Reso1Freq;
        public Vector<BezierChain> Reso1BW;
        public Vector<BezierChain> Reso1Amp;
        public Vector<BezierChain> Reso2Freq;
        public Vector<BezierChain> Reso2BW;
        public Vector<BezierChain> Reso2Amp;
        public Vector<BezierChain> Reso3Freq;
        public Vector<BezierChain> Reso3BW;
        public Vector<BezierChain> Reso3Amp;
        public Vector<BezierChain> Reso4Freq;
        public Vector<BezierChain> Reso4BW;
        public Vector<BezierChain> Reso4Amp;
        public Vector<BezierChain> PitchBend;
        public Vector<BezierChain> PitchBendSensitivity;
       
        public BezierCurves() {
            Dynamics = new Vector<BezierChain>();
            Brethiness = new Vector<BezierChain>();
            Brightness = new Vector<BezierChain>();
            Clearness = new Vector<BezierChain>();
            Opening = new Vector<BezierChain>();
            GenderFactor = new Vector<BezierChain>();
            PortamentoTiming = new Vector<BezierChain>();
            VibratoRate = new Vector<BezierChain>();
            VibratoDepth = new Vector<BezierChain>();
            Harmonics = new Vector<BezierChain>();
            FX2Depth = new Vector<BezierChain>();
            Reso1Freq = new Vector<BezierChain>();
            Reso1BW = new Vector<BezierChain>();
            Reso1Amp = new Vector<BezierChain>();
            Reso2Freq = new Vector<BezierChain>();
            Reso2BW = new Vector<BezierChain>();
            Reso2Amp = new Vector<BezierChain>();
            Reso3Freq = new Vector<BezierChain>();
            Reso3BW = new Vector<BezierChain>();
            Reso3Amp = new Vector<BezierChain>();
            Reso4Freq = new Vector<BezierChain>();
            Reso4BW = new Vector<BezierChain>();
            Reso4Amp = new Vector<BezierChain>();
            PitchBend = new Vector<BezierChain>();
            PitchBendSensitivity = new Vector<BezierChain>();
        }

        public BezierChain GetBezierChain( CurveType curve_type, int chain_id ) {
            Vector<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.size(); i++ ) {
                if ( list.get( i ).id == chain_id ) {
                    return list.get( i );
                }
            }
            return null;
        }

        public void SetBezierChain( CurveType curve_type, int chain_id, BezierChain item ) {
            Vector<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.size(); i++ ) {
                if ( list.get( i ).id == chain_id ) {
                    list.set( i, item );
                    break;
                }
            }
        }

        /// <summary>
        /// 指定した種類のコントロールカーブにベジエ曲線を追加します。
        /// AddBezierChainとの違い、オーバーラップする部分があれば自動的に結合されます。
        /// chainには2個以上のデータ点が含まれている必要がある
        /// </summary>
        /// <param name="curve"></param>
        /// <param name="chain"></param>
        public void MergeBezierChain( CurveType curve, BezierChain chain ) {
            if ( chain.points.size() <= 1 ) {
                return;
            }
            int chain_start = (int)chain.getStart();
            int chain_end = (int)chain.getEnd();

            // まず、全削除する必要のあるBezierChainを検索
            Vector<Integer> delete_list = new Vector<Integer>();
            Vector<BezierChain> src = this[curve];
            //foreach ( int id in this[curve].Keys ) {
            for ( int j = 0; j < src.size(); j++ ) {
                //BezierChain bc = this[curve][id];
                BezierChain bc = src.get( j );
                if ( bc.points.size() <= 0 ) {
                    continue;
                }
                int bc_start = (int)bc.getStart();
                int bc_end = (int)bc.getEnd();
                if ( chain_start <= bc_start && bc_end <= chain_end ) {
                    delete_list.add( bc.id );
                }
            }

            // 削除を実行
            for ( Iterator itr = delete_list.iterator(); itr.hasNext(); ){
                int id = (int)itr.next();
                Remove( curve, id );
                //this[curve].Remove( id );
            }

            // マージする必要があるかどうかを検査。
            boolean processed = true;
            while ( processed ) {
                processed = false;
                Vector<BezierChain> list = this[curve];
                //foreach ( int id in this[curve].Keys ) {
                for ( int j = 0; j < list.size(); j++ ) {
                    //BezierChain bc = this[curve][id];
                    BezierChain bc = list.get( j );
                    int id = bc.id;
                    int start = (int)bc.getStart();
                    int end = (int)bc.getEnd();

                    // 被っている箇所が2箇所以上ある可能性があるので、ifでヒットしてもbreakしない
                    if ( start < chain_start && chain_start <= end && end < chain_end ) {
                        // bcのchain_start ~ endを削除し、chain_startで結合
                        BezierChain bc_edit = bc.extractPartialBezier( start, chain_start );
                        bc_edit.id = bc.id;
                        int last = bc_edit.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points.get( last ).ControlRightType = BezierControlType.None;
                        chain.points.get( 0 ).ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        if ( bc_edit.points.get( last ).Base.Y == chain.points.get( 0 ).Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( bc_edit.points.get( last ).ControlLeftType != BezierControlType.None ) {
                                bc_edit.points.get( last ).ControlLeftType = BezierControlType.Master;
                            }
                            bc_edit.points.get( last ).ControlRight = chain.points.get( 0 ).controlLeft;
                            if ( chain.points.get( 0 ).ControlRightType != BezierControlType.None ) {
                                bc_edit.points.get( last ).ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }
                        for ( int i = copy_start; i < chain.points.size(); i++ ) {
                            chain.points.get( i ).ID = bc_edit.getNextId();
                            bc_edit.add( chain.points.get( i ) );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = bc_edit;
                        chain_start = (int)chain.getStart();
                        chain_end = (int)chain.getEnd();
                        processed = true;
                        break;
                    } else if ( chain_start < start && start <= chain_end && chain_end < end ) {
                        // bcのstart ~ chain_endを削除し、chain_endで結合
                        BezierChain bc_edit = bc.extractPartialBezier( chain_end, end );
                        bc_edit.id = bc.id;
                        int last = chain.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points.get( 0 ).ControlLeftType = BezierControlType.None;
                        chain.points.get( last ).ControlRightType = BezierControlType.None;

                        int copy_end = last;
                        if ( chain.points.get( last ).Base.Y == bc_edit.points.get( 0 ).Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain.points.get( last ).ControlLeftType != BezierControlType.None ) {
                                chain.points.get( last ).ControlLeftType = BezierControlType.Master;
                            }
                            chain.points.get( last ).ControlRight = bc_edit.points.get( 0 ).controlLeft;
                            if ( bc_edit.points.get( 0 ).ControlRightType != BezierControlType.None ) {
                                chain.points.get( last ).ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = last - 1;
                        }
                        for ( int i = 0; i <= copy_end; i++ ) {
                            chain.points.get( i ).ID = bc_edit.getNextId();
                            bc_edit.add( chain.points.get( i ) );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = bc_edit;
                        chain_start = (int)chain.getStart();
                        chain_end = (int)chain.getEnd();
                        processed = true;
                        break;
                    } else if ( start < chain_start && chain_end < end ) {
                        // bcのchain_start ~ chain_endをchainで置き換え
                        // left + chain + right
                        BezierChain left = bc.extractPartialBezier( start, chain_start );
                        BezierChain right = bc.extractPartialBezier( chain_end, end );
                        left.id = bc.id;

                        // 接合部ではステップ変化
                        left.points.get( left.Count - 1 ).ControlRightType = BezierControlType.None;
                        chain.points.get( 0 ).ControlLeftType = BezierControlType.None;
                        chain.points.get( chain.Count - 1 ).ControlRightType = BezierControlType.None;
                        right.points.get( 0 ).ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        int copy_end = chain.Count - 1;

                        if ( left.points.get( left.Count - 1 ).Base.Y == chain.points.get( 0 ).Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( left.points.get( left.Count - 1 ).ControlLeftType != BezierControlType.None ) {
                                left.points.get( left.Count - 1 ).ControlLeftType = BezierControlType.Master;
                            }
                            left.points.get( left.Count - 1 ).ControlRight = chain.points.get( 0 ).controlLeft;
                            if ( chain.points.get( 0 ).ControlRightType != BezierControlType.None ) {
                                left.points.get( left.Count - 1 ).ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }

                        if ( chain.points.get( chain.Count - 1 ).Base.Y == right.points.get( 0 ).Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain.points.get( chain.Count - 1 ).ControlLeftType != BezierControlType.None ) {
                                chain.points.get( chain.Count - 1 ).ControlLeftType = BezierControlType.Master;
                            }
                            chain.points.get( chain.Count - 1 ).ControlRight = right.points.get( 0 ).controlLeft;
                            if ( right.points.get( 0 ).ControlRightType != BezierControlType.None ) {
                                chain.points.get( chain.Count - 1 ).ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = chain.Count - 2;
                        }

                        // 追加
                        for ( int i = copy_start; i <= copy_end; i++ ) {
                            chain.points.get( i ).ID = left.getNextId();
                            left.add( chain.points.get( i ) );
                        }
                        for ( int i = 0; i < right.points.size(); i++ ) {
                            right.points.get( i ).ID = left.getNextId();
                            left.add( right.points.get( i ) );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = left;
                        chain_start = (int)chain.getStart();
                        chain_end = (int)chain.getEnd();
                        processed = true;
                        break;
                    }
                }
            }

            if ( !processed ) {
                chain.id = this.GetNextId( curve );
            }
            //this[curve].Add( chain.ID, chain );
            AddBezierChain( curve, chain, chain.id );
        }

        public boolean DeleteBeziers(
            Vector<CurveType> target_curve,
            int clock_start,
            int clock_end
        ) {
            boolean edited = false;
            for ( Iterator itr1 = target_curve.iterator(); itr1.hasNext(); ){
                CurveType curve = (CurveType)itr1.next();
                if ( curve.IsScalar || curve.IsAttachNote ) {
                    continue;
                }
                Vector<BezierChain> tmp = new Vector<BezierChain>();
                for ( Iterator itr = this[curve].iterator(); itr.hasNext(); ){
                    BezierChain bc = (BezierChain)itr.next();
                    int len = bc.points.size();
                    if ( len < 1 ) {
                        continue;
                    }
                    int chain_start = (int)bc.points.get( 0 ).Base.X;
                    int chain_end;
                    if ( len < 2 ) {
                        chain_end = chain_start;
                    } else {
                        chain_end = (int)bc.points.get( len - 1 ).Base.X;
                    }
                    if ( clock_end < chain_start && chain_start < clock_end && clock_end < chain_end ) {
                        // end ~ chain_endを残す
                        BezierChain chain = bc.extractPartialBezier( clock_end, chain_end );
                        chain.id = bc.id;
                        tmp.add( chain );
                        edited = true;
                    } else if ( chain_start <= clock_start && clock_end <= chain_end ) {
                        // chain_start ~ startとend ~ chain_endを残す
                        BezierChain chain1 = bc.extractPartialBezier( chain_start, clock_start );
                        chain1.id = bc.id;
                        BezierChain chain2 = bc.extractPartialBezier( clock_end, chain_end );
                        chain2.id = -1;  // 後で番号をつける
                        tmp.add( chain1 );
                        tmp.add( chain2 );
                        edited = true;
                    } else if ( chain_start < clock_start && clock_start < chain_end && clock_end <= chain_end ) {
                        // chain_start ~ startを残す
                        BezierChain chain = bc.extractPartialBezier( chain_start, clock_start );
                        chain.id = bc.id;
                        tmp.add( chain );
                        edited = true;
                    } else if ( clock_start <= chain_start && chain_end <= clock_end ) {
                        // 全体を削除
                        edited = true;
                    } else {
                        // 全体を残す
                        tmp.add( (BezierChain)bc.Clone() );
                    }
                }
                this[curve].clear();
                for ( Iterator itr = tmp.iterator(); itr.hasNext(); ){
                    BezierChain bc = (BezierChain)itr.next();
                    if ( bc.id >= 0 ) {
                        AddBezierChain( curve, bc, bc.id );
                    }
                }
                for ( Iterator itr = tmp.iterator(); itr.hasNext(); ){
                    BezierChain bc = (BezierChain)itr.next();
                    if ( bc.id < 0 ) {
                        bc.id = this.GetNextId( curve );
                        AddBezierChain( curve, bc, bc.id );
                    }
                }
            }
            return edited;
        }

        public void Remove( CurveType curve_type, int chain_id ) {
            Vector<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.size(); i++ ) {
                if ( list.get( i ).id == chain_id ) {
                    list.removeElementAt( i );
                    break;
                }
            }
        }

        /// <summary>
        /// 指定したコントロールカーブにベジエ曲線を追加します。
        /// </summary>
        /// <param name="curve_type"></param>
        /// <param name="chain"></param>
        public void AddBezierChain( CurveType curve_type, BezierChain chain, int chain_id ) {
            int index = curve_type.Index;
            BezierChain add = (BezierChain)chain.Clone();
            add.id = chain_id;
            this[curve_type].add( add );
        }

        public object Clone() {
            BezierCurves ret = new BezierCurves();
            foreach ( CurveType ct in AppManager.CURVE_USAGE ) {
                Vector<BezierChain> src = this[ct];
                ret[ct] = new Vector<BezierChain>();
                for ( int i = 0; i < src.size(); i++ ) {
                    ret[ct].add( (BezierChain)src.get( i ).Clone() );
                }
            }
            return ret;
        }

        public int GetNextId( CurveType curve_type ) {
            int index = curve_type.Index;
            Vector<BezierChain> bc = this[curve_type];
            int ret = bc.size();// m_curves[index].Count;
            /*while ( m_curves[index].ContainsKey( ret ) ) {
                ret++;
            }*/
            boolean found = true;
            while ( found ) {
                found = false;
                for ( int i = 0; i < bc.size(); i++ ) {
                    if ( bc.get( i ).id == ret ) {
                        found = true;
                        ret++;
                        break;
                    }
                }
            }
            return ret;
        }

        public Vector<BezierChain> get( CurveType curve ) {
            if ( curve.equals( CurveType.BRE ) ) {
                return Brethiness;
            } else if ( curve.equals( CurveType.BRI ) ) {
                return Brightness;
            } else if ( curve.equals( CurveType.CLE ) ) {
                return Clearness;
            } else if ( curve.equals( CurveType.DYN ) ) {
                return Dynamics;
            } else if ( curve.equals( CurveType.fx2depth ) ) {
                return FX2Depth;
            } else if ( curve.equals( CurveType.GEN ) ) {
                return GenderFactor;
            } else if ( curve.equals( CurveType.harmonics ) ) {
                return Harmonics;
            } else if ( curve.equals( CurveType.OPE ) ) {
                return Opening;
            } else if ( curve.equals( CurveType.POR ) ) {
                return PortamentoTiming;
            } else if ( curve.equals( CurveType.PIT ) ) {
                return PitchBend;
            } else if ( curve.equals( CurveType.PBS ) ) {
                return PitchBendSensitivity;
            } else if ( curve.equals( CurveType.reso1amp ) ) {
                return Reso1Amp;
            } else if ( curve.equals( CurveType.reso1bw ) ) {
                return Reso1BW;
            } else if ( curve.equals( CurveType.reso1freq ) ) {
                return Reso1Freq;
            } else if ( curve.equals( CurveType.reso2amp ) ) {
                return Reso2Amp;
            } else if ( curve.equals( CurveType.reso2bw ) ) {
                return Reso2BW;
            } else if ( curve.equals( CurveType.reso2freq ) ) {
                return Reso2Freq;
            } else if ( curve.equals( CurveType.reso3amp ) ) {
                return Reso3Amp;
            } else if ( curve.equals( CurveType.reso3bw ) ) {
                return Reso3BW;
            } else if ( curve.equals( CurveType.reso3freq ) ) {
                return Reso3Freq;
            } else if ( curve.equals( CurveType.reso4amp ) ) {
                return Reso4Amp;
            } else if ( curve.equals( CurveType.reso4bw ) ) {
                return Reso4BW;
            } else if ( curve.equals( CurveType.reso4freq ) ) {
                return Reso4Freq;
            } else if ( curve.equals( CurveType.VibratoDepth ) ) {
                return VibratoDepth;
            } else if ( curve.equals( CurveType.VibratoRate ) ) {
                return VibratoRate;
            } else {
                return null;
            }
        }

        public void set( CurveType curve, Vector<BezierChain> value ) {
            if ( curve.equals( CurveType.BRE ) ) {
                Brethiness = value;
            } else if ( curve.equals( CurveType.BRI ) ) {
                Brightness = value;
            } else if ( curve.equals( CurveType.CLE ) ) {
                Clearness = value;
            } else if ( curve.equals( CurveType.DYN ) ) {
                Dynamics = value;
            } else if ( curve.equals( CurveType.fx2depth ) ) {
                FX2Depth = value;
            } else if ( curve.equals( CurveType.GEN ) ) {
                GenderFactor = value;
            } else if ( curve.equals( CurveType.harmonics ) ) {
                Harmonics = value;
            } else if ( curve.equals( CurveType.OPE ) ) {
                Opening = value;
            } else if ( curve.equals( CurveType.POR ) ) {
                PortamentoTiming = value;
            } else if ( curve.equals( CurveType.PIT ) ) {
                PitchBend = value;
            } else if ( curve.equals( CurveType.PBS ) ) {
                PitchBendSensitivity = value;
            } else if ( curve.equals( CurveType.reso1amp ) ) {
                Reso1Amp = value;
            } else if ( curve.equals( CurveType.reso1bw ) ) {
                Reso1BW = value;
            } else if ( curve.equals( CurveType.reso1freq ) ) {
                Reso1Freq = value;
            } else if ( curve.equals( CurveType.reso2amp ) ) {
                Reso2Amp = value;
            } else if ( curve.equals( CurveType.reso2bw ) ) {
                Reso2BW = value;
            } else if ( curve.equals( CurveType.reso2freq ) ) {
                Reso2Freq = value;
            } else if ( curve.equals( CurveType.reso3amp ) ) {
                Reso3Amp = value;
            } else if ( curve.equals( CurveType.reso3bw ) ) {
                Reso3BW = value;
            } else if ( curve.equals( CurveType.reso3freq ) ) {
                Reso3Freq = value;
            } else if ( curve.equals( CurveType.reso4amp ) ) {
                Reso4Amp = value;
            } else if ( curve.equals( CurveType.reso4bw ) ) {
                Reso4BW = value;
            } else if ( curve.equals( CurveType.reso4freq ) ) {
                Reso4Freq = value;
            } else if ( curve.equals( CurveType.VibratoDepth ) ) {
                VibratoDepth = value;
            } else if ( curve.equals( CurveType.VibratoRate ) ) {
                VibratoRate = value;
            }
        }

        public Vector<BezierChain> this[CurveType curve] {
            get {
                return get( curve );
            }
            set {
                set( curve, value );
            }
        }

        /*public BezierCurves( AttachedCurve item ) {
            foreach ( CurveType ct in CURVE_USAGE ) {
                TreeMap<int, BezierChain> dict = item[ct];
                if ( dict != null ) {
                    this[ct] = new BezierChain[item[ct].Count];
                    int i = -1;
                    foreach ( int key in item[ct].Keys ) {
                        i++;
                        this[ct][i] = (BezierChain)item[ct][key].Clone();
                    }
                } else {
                    this[ct] = new BezierChain[0];
                }
            }
        }*/
    }

}
