﻿/*
 * PropertyPanel.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Windows.Forms.Design;
using System.Drawing;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    public delegate void CommandExecuteRequiredEventHandler( CadenciiCommand command );

    public partial class PropertyPanel : UserControl {
        public event CommandExecuteRequiredEventHandler CommandExecuteRequired;
        private List<VsqEventItemProxy> m_items;
        private int m_track;

        public PropertyPanel() {
            InitializeComponent();
            m_items = new List<VsqEventItemProxy>();
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont );
        }

        public bool Editing {
            get;
            set;
        }

        private void popGridItemExpandStatus() {
            if ( propertyGrid.SelectedGridItem == null ) {
                return;
            }

            GridItem root = findRootGridItem( propertyGrid.SelectedGridItem );
            if ( root == null ) {
                return;
            }

            popGridItemExpandStatusCore( root );
        }

        private void popGridItemExpandStatusCore( GridItem item ) {
            if ( item.Expandable ) {
                string s = getGridItemIdentifier( item );
                foreach ( ValuePair<string, bool> v in AppManager.EditorConfig.PropertyWindowStatus.ExpandStatus ) {
                    if ( v.Key == s ){
                        item.Expanded = v.Value;
                        break;
                    }
                }
            }
            foreach ( GridItem child in item.GridItems ) {
                popGridItemExpandStatusCore( child );
            }
        }

        private void pushGridItemExpandStatus() {
            if ( propertyGrid.SelectedGridItem == null ) {
                return;
            }

            GridItem root = findRootGridItem( propertyGrid.SelectedGridItem );
            if( root == null ){
                return;
            }

            pushGridItemExpandStatusCore( root );
        }

        private void pushGridItemExpandStatusCore( GridItem item ) {
            if ( item.Expandable ) {
                string s = getGridItemIdentifier( item );
                bool found = false;
                foreach( ValuePair<string, bool> v in AppManager.EditorConfig.PropertyWindowStatus.ExpandStatus ){
                    if ( v.Key == s ) {
                        found = true;
                        v.Value = item.Expanded;
                    }
                }
                if ( !found ) {
                    AppManager.EditorConfig.PropertyWindowStatus.ExpandStatus.Add( new ValuePair<string, bool>( s, item.Expanded ) );
                }
            }
            foreach( GridItem child in item.GridItems ){
                pushGridItemExpandStatusCore( child );
            }
        }

        public void UpdateValue( int track ) {
            m_track = track;
            m_items.Clear();

            // 現在のGridItemの展開状態を取得
            pushGridItemExpandStatus();

            // InternalIDを列挙
            List<int> items = new List<int>();
            foreach ( SelectedEventEntry item in AppManager.SelectedEvent.GetEnumerator() ) {
                if ( item.Track == track ) {
                    items.Add( item.Original.InternalID );
                }
            }

            // itemsの中身を列挙
            int count = 0;
            for ( Iterator itr = AppManager.VsqFile.Track[m_track].getNoteEventIterator(); itr.hasNext(); ) {
                VsqEvent ve = (VsqEvent)itr.next();
                if ( items.Contains( ve.InternalID ) ) {
                    count++;
                    m_items.Add( new VsqEventItemProxy( ve ) );
                }
                if ( count == items.Count ) {
                    break;
                }
            }

            object[] objs = new object[m_items.Count];
            for ( int i = 0; i < m_items.Count; i++ ) {
                objs[i] = m_items[i];
            }
            propertyGrid.SelectedObjects = objs;
            popGridItemExpandStatus();
            Editing = false;
        }

        private void propertyGrid_PropertyValueChanged( object s, PropertyValueChangedEventArgs e ) {
            string name = e.ChangedItem.PropertyDescriptor.Name;
            object old_value = e.OldValue;
            int len = propertyGrid.SelectedObjects.Length;
            VsqEvent[] items = new VsqEvent[len];
            for ( int i = 0; i < len; i++ ) {
                VsqEventItemProxy proxy = (VsqEventItemProxy)propertyGrid.SelectedObjects[i];

                items[i] = proxy.GetItemDifference();

                m_items[i].Original.Clock = proxy.Clock.Clock.getIntValue();
                m_items[i].Original.ID.DEMaccent = proxy.Accent;
                m_items[i].Original.ID.DEMdecGainRate = proxy.Decay;
                m_items[i].Original.ID.Dynamics = proxy.Velocity;
                m_items[i].Original.ID.Length = proxy.Length.getIntValue();
                m_items[i].Original.ID.LyricHandle.L0.setPhoneticSymbol( proxy.PhoneticSymbol );
                m_items[i].Original.ID.LyricHandle.L0.Phrase = proxy.Phrase;
                m_items[i].Original.ID.Note = proxy.Note.noteNumber;
                m_items[i].Original.ID.PMBendDepth = proxy.BendDepth;
                m_items[i].Original.ID.PMBendLength = proxy.BendLength;
                m_items[i].Original.ID.PMbPortamentoUse = proxy.GetPortamentoUsage();
                m_items[i].Original.ID.VibratoDelay = proxy.GetVibratoDelay();
                m_items[i].Original.UstEvent.PreUtterance = proxy.PreUtterance;
                m_items[i].Original.UstEvent.VoiceOverlap = proxy.Overlap;
                m_items[i].Original.UstEvent.Moduration = proxy.Moduration;
                m_items[i].Original.ID.d4mean = proxy.d4mean;
                m_items[i].Original.ID.pMeanEndingNote = proxy.pMeanEndingNote;
                m_items[i].Original.ID.pMeanOnsetFirstNote = proxy.pMeanOnsetFirstNote;
                m_items[i].Original.ID.vMeanNoteTransition = proxy.vMeanNoteTransition;
            }
            if ( CommandExecuteRequired != null ) {
                CadenciiCommand run = new CadenciiCommand( VsqCommand.generateCommandEventReplaceRange( m_track, items ) );
                CommandExecuteRequired( run );
            }
            propertyGrid.Refresh();
            Editing = false;
        }

        /// <summary>
        /// itemが属しているGridItemツリーの基点にある親を探します
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        private GridItem findRootGridItem( GridItem item ) {
            if ( item.Parent == null ) {
                return item;
            } else {
                return findRootGridItem( item.Parent );
            }
        }

        /// <summary>
        /// itemが属しているGridItemツリーの中で，itemを特定するための文字列を取得します
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        private string getGridItemIdentifier( GridItem item ) {
            if ( item.Parent == null ) {
                if ( item.PropertyDescriptor != null ) {
                    return item.PropertyDescriptor.Name;
                } else {
                    return item.Label;
                }
            } else {
                if ( item.PropertyDescriptor != null ) {
                    return getGridItemIdentifier( item.Parent ) + "@" + item.PropertyDescriptor.Name;
                } else {
                    return getGridItemIdentifier( item.Parent ) + "@" + item.Label;
                }
            }
        }

        private void propertyGrid_SelectedGridItemChanged( object sender, SelectedGridItemChangedEventArgs e ) {
            Editing = true;
        }

        private void propertyGrid_Enter( object sender, EventArgs e ) {
            Editing = true;
        }

        private void propertyGrid_Leave( object sender, EventArgs e ) {
            Editing = false;
        }
    }

}
