﻿/*
 * AppManager.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Windows.Forms;
using System.Xml.Serialization;
using Microsoft.CSharp;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;
/*
-コントロールカーブのテンプレートの登録＆貼付け機能
-選択したノートを指定したクロック数分一括シフトする機能
-前回レンダリングしたWAVEファイルをリサイクルする（起動ごとにレンダリングするのは面倒？）
-横軸を変更するショートカットキー
-音符の時間位置を秒指定で入力できるモード
-コントロールカーブのリアルタイムかつアナログ的な入力（ジョイスティックorゲームコントローラ）
-コントロールカーブエディタを幾つか積めるような機能
 */
namespace Boare.Cadencii {

    public delegate void BSimpleDelegate<T>( T arg );

    public static class AppManager {
        /// <summary>
        /// 鍵盤の表示幅(pixel)
        /// </summary>
        public const int _KEY_LENGTH = 68;
        private const string _CONFIG_FILE_NAME = "config.xml";
        private const string _CONFIG_DIR_NAME = "Cadencii";
        /// <summary>
        /// OSのクリップボードに貼り付ける文字列の接頭辞．これがついていた場合，クリップボードの文字列をCadenciiが使用できると判断する．
        /// </summary>
        private const string _CLIP_PREFIX = "CADENCIIOBJ";

        /// <summary>
        /// エディタの設定
        /// </summary>
        public static EditorConfig EditorConfig = new EditorConfig();
        /// <summary>
        /// AttachedCurve用のシリアライザ
        /// </summary>
        public static XmlSerializer XmlSerializerListBezierCurves = new XmlSerializer( typeof( AttachedCurve ) );
        public static Font BaseFont8 = null;
        public static Font BaseFont9 = null;
        /// <summary>
        /// ピアノロールの歌詞の描画に使用されるフォント。
        /// </summary>
        public static Font BaseFont10 = null;
        /// <summary>
        /// ピアノロールの歌詞の描画に使用されるフォント。（発音記号固定の物の場合）
        /// </summary>
        public static Font BaseFont10Bold = null;
        /// <summary>
        /// 歌詞を音符の（高さ方向の）真ん中に描画するためのオフセット
        /// </summary>
        public static int BaseFont10OffsetHeight = 0;
        public static int BaseFont8OffsetHeight = 0;
        public static int BaseFont9OffsetHeight = 0;
        public static PropertyPanel NotePropertyPanel;
        public static FormNoteProperty NotePropertyDlg;

        #region Static Readonly Fields
        public static readonly Color[] s_HILIGHT = new Color[] { 
            Color.FromArgb( 181, 220, 16 ),
            Color.FromArgb( 231, 244, 49 ),
            Color.FromArgb( 252, 230, 29 ),
            Color.FromArgb( 247, 171, 20 ),
            Color.FromArgb( 249, 94, 17 ),
            Color.FromArgb( 234, 27, 47 ),
            Color.FromArgb( 175, 20, 80 ),
            Color.FromArgb( 183, 24, 149 ),
            Color.FromArgb( 105, 22, 158 ),
            Color.FromArgb( 22, 36, 163 ),
            Color.FromArgb( 37, 121, 204 ),
            Color.FromArgb( 29, 179, 219 ),
            Color.FromArgb( 24, 239, 239 ),
            Color.FromArgb( 25, 206, 175 ),
            Color.FromArgb( 23, 160, 134 ),
            Color.FromArgb( 79, 181, 21 ) };
        public static readonly Color[] s_RENDER = new Color[]{
            Color.FromArgb( 19, 143, 52 ),
            Color.FromArgb( 158, 154, 18 ),
            Color.FromArgb( 160, 143, 23 ),
            Color.FromArgb( 145, 98, 15 ),
            Color.FromArgb( 142, 52, 12 ),
            Color.FromArgb( 142, 19, 37 ),
            Color.FromArgb( 96, 13, 47 ),
            Color.FromArgb( 117, 17, 98 ),
            Color.FromArgb( 62, 15, 99 ),
            Color.FromArgb( 13, 23, 84 ),
            Color.FromArgb( 25, 84, 132 ),
            Color.FromArgb( 20, 119, 142 ),
            Color.FromArgb( 19, 142, 139 ),
            Color.FromArgb( 17, 122, 102 ),
            Color.FromArgb( 13, 86, 72 ),
            Color.FromArgb( 43, 91, 12 ) };
        public static readonly string[] _USINGS = new string[] { "using System;",
                                             "using System.IO;",
                                             "using Boare.Lib.Vsq;",
                                             "using Boare.Cadencii;",
                                             "using bocoree;",
                                             "using Boare.Lib.Media;",
                                             "using Boare.Lib.AppUtil;",
                                             "using System.Windows.Forms;",
                                             "using System.Collections.Generic;",
                                             "using System.Drawing;",
                                             "using System.Text;",
                                             "using System.Xml.Serialization;" };
        public static readonly List<Keys> _SHORTCUT_ACCEPTABLE = new List<Keys>( new Keys[]{
            Keys.A,
            Keys.B,
            Keys.C,
            Keys.D,
            Keys.D0,
            Keys.D1,
            Keys.D2,
            Keys.D3,
            Keys.D4,
            Keys.D5,
            Keys.D6,
            Keys.D7,
            Keys.D8,
            Keys.D9,
            Keys.Down,
            Keys.E,
            Keys.F,
            Keys.F1,
            Keys.F2,
            Keys.F3,
            Keys.F4,
            Keys.F5,
            Keys.F6,
            Keys.F7,
            Keys.F8,
            Keys.F9,
            Keys.F10,
            Keys.F11,
            Keys.F12,
            Keys.F13,
            Keys.F14,
            Keys.F15,
            Keys.F16,
            Keys.F17,
            Keys.F18,
            Keys.F19,
            Keys.F20,
            Keys.F21,
            Keys.F22,
            Keys.F23,
            Keys.F24,
            Keys.G,
            Keys.H,
            Keys.I,
            Keys.J,
            Keys.K,
            Keys.L,
            Keys.Left,
            Keys.M,
            Keys.N,
            Keys.NumPad0,
            Keys.NumPad1,
            Keys.NumPad2,
            Keys.NumPad3,
            Keys.NumPad4,
            Keys.NumPad5,
            Keys.NumPad6,
            Keys.NumPad7,
            Keys.NumPad8,
            Keys.NumPad9,
            Keys.O,
            Keys.P,
            Keys.PageDown,
            Keys.PageUp,
            Keys.Q,
            Keys.R,
            Keys.Right,
            Keys.S,
            Keys.Space,
            Keys.T,
            Keys.U,
            Keys.Up,
            Keys.V,
            Keys.W,
            Keys.X,
            Keys.Y,
            Keys.Z } );
        public static readonly CurveType[] CURVE_USAGE = new CurveType[]{ CurveType.DYN,
                                                                          CurveType.BRE,
                                                                          CurveType.BRI,
                                                                          CurveType.CLE,
                                                                          CurveType.OPE,
                                                                          CurveType.GEN,
                                                                          CurveType.POR,
                                                                          CurveType.VibratoRate,
                                                                          CurveType.VibratoDepth,
                                                                          CurveType.Pitch,
                                                                          CurveType.harmonics,
                                                                          CurveType.fx2depth,
                                                                          CurveType.reso1amp,
                                                                          CurveType.reso1bw,
                                                                          CurveType.reso1freq,
                                                                          CurveType.reso2amp,
                                                                          CurveType.reso2bw,
                                                                          CurveType.reso2freq,
                                                                          CurveType.reso3amp,
                                                                          CurveType.reso3bw,
                                                                          CurveType.reso3freq,
                                                                          CurveType.reso4amp,
                                                                          CurveType.reso4bw,
                                                                          CurveType.reso4freq };
        #endregion

        #region Private Static Fields
        private static int s_base_tempo = 480000;
        private static SolidBrush s_hilight_brush = new SolidBrush( Color.CornflowerBlue );
        private static object s_locker;
        //private static XmlSerializer s_serizlizer = null;
        #endregion

        #region Imported from EditorManager
        private static int m_start_to_draw_y = 0;
#if !TREECOM
        private static VsqFileEx m_vsq;
#endif
        private static string m_file = "";
        private static int m_selected = 1;
        private static int m_current_clock = 0;
        private static bool m_playing = false;
        private static bool m_repeat_mode = false;
        private static bool m_saveavi = false;
        private static bool m_grid_visible = false;
        private static EditMode m_edit_mode = EditMode.None;
        /// <summary>
        /// トラックのレンダリングが必要かどうかを表すフラグ
        /// </summary>
        private static bool[] m_render_required = new bool[16] { false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false };
        /// <summary>
        /// トラックのオーバーレイ表示
        /// </summary>
        private static bool m_overlay = true;
        private static SelectedEventList m_selected_event = new SelectedEventList();
        /// <summary>
        /// 選択されているテンポ変更イベントのリスト
        /// </summary>
        private static Dictionary<int, SelectedTempoEntry> m_selected_tempo = new Dictionary<int, SelectedTempoEntry>();
        private static int m_last_selected_tempo = -1;
        /// <summary>
        /// 選択されている拍子変更イベントのリスト
        /// </summary>
        private static Dictionary<int, SelectedTimesigEntry> m_selected_timesig = new Dictionary<int, SelectedTimesigEntry>();
        private static int m_last_selected_timesig = -1;
        private static EditTool m_selected_tool = EditTool.Pencil;
#if !TREECOM
        private static List<ICommand> m_commands = new List<ICommand>();
        private static int m_command_position = -1;
#endif
        /// <summary>
        /// 選択されているベジエ点のリスト
        /// </summary>
        private static List<SelectedBezierPoint> m_selected_bezier = new List<SelectedBezierPoint>();
        /// <summary>
        /// 最後に選択されたベジエ点
        /// </summary>
        private static SelectedBezierPoint m_last_selected_bezier = new SelectedBezierPoint();
        /// <summary>
        /// 現在の演奏カーソルの位置(m_current_clockと意味は同じ。CurrentClockが変更されると、自動で更新される)
        /// </summary>
        private static PlayPositionSpecifier m_current_play_position;

        /// <summary>
        /// SelectedRegionが有効かどうかを表すフラグ
        /// </summary>
        public static bool SelectedRegionEnabled = false;
        /// <summary>
        /// Ctrlキーを押しながらのマウスドラッグ操作による選択が行われた範囲
        /// </summary>
        public static SelectedRegion SelectedRegion;
        /// <summary>
        /// 自動ノーマライズモードかどうかを表す値を取得、または設定します。
        /// </summary> 
        public static bool AutoNormalize = false;
        /// <summary>
        /// エンドマーカーの位置(clock)
        /// </summary>
        public static int EndMarker = 0;
        public static bool EndMarkerEnabled = false;
        /// <summary>
        /// x方向の表示倍率(pixel/clock)
        /// </summary>
        public static float ScaleX = 0.1f;
        /// <summary>
        /// スタートマーカーの位置(clock)
        /// </summary>
        public static int StartMarker = 0;
        /// <summary>
        /// Bezierカーブ編集モードが有効かどうかを表す
        /// </summary>
        public static bool m_is_curve_mode = false;
        public static bool StartMarkerEnabled = false;
        /// <summary>
        /// 再生時に自動スクロールするかどうか
        /// </summary>
        public static bool AutoScroll = true;
        /// <summary>
        /// 最初のバッファが書き込まれたかどうか
        /// </summary>
        public static bool FirstBufferWritten = false;
        /// <summary>
        /// リアルタイム再生で使用しようとしたVSTiが有効だったかどうか。
        /// </summary>
        public static bool RendererAvailable = false;
        /// <summary>
        /// プレビュー再生が開始された時刻
        /// </summary>
        public static DateTime PreviewStartedTime;
        /// <summary>
        /// 画面左端位置での、仮想画面上の仮想画面左端から測ったピクセル数．FormMain.hScroll.ValueとFormMain.trackBar.Valueで決まる．
        /// </summary>
        public static int StartToDrawX;
        public static string SelectedPaletteTool = "";
        public static ScreenStatus LastScreenStatus = new ScreenStatus();
        public static string m_id = "";

        /// <summary>
        /// グリッド線を表示するか否かを表す値が変更された時発生します
        /// </summary>
        public static event EventHandler GridVisibleChanged;
        public static event EventHandler PreviewStarted;
        public static event EventHandler PreviewAborted;
        public static event BSimpleDelegate<bool> SelectedEventChanged;
        public static event EventHandler SelectedToolChanged;
        /// <summary>
        /// CurrentClockプロパティの値が変更された時発生します
        /// </summary>
        public static event EventHandler CurrentClockChanged;

        private const string _TEMPDIR_NAME = "cadencii";

        /// <summary>
        /// pがrcの中にあるかどうかを判定します
        /// </summary>
        /// <param name="p"></param>
        /// <param name="rc"></param>
        /// <returns></returns>
        public static bool IsInRect( Point p, Rectangle rc ) {
            if ( rc.X <= p.X ) {
                if ( p.X <= rc.X + rc.Width ) {
                    if ( rc.Y <= p.Y ) {
                        if ( p.Y <= rc.Y + rc.Height ) {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        /// <summary>
        /// fontを使って文字を描画したとき，文字の縦方向の中心線と，描画時に指定した座標との間にどれだけのオフセットが生じるかを調べる
        /// </summary>
        /// <param name="font"></param>
        /// <returns></returns>
        public static unsafe int GetStringOffset( Font font ) {
            SizeF str_size = MeasureString( "Qjp", font );
            if ( str_size.Width <= 0 || str_size.Height <= 0 ) {
                return 0;
            }
            int draw_pos = (int)(str_size.Height / 2);
            Bitmap test = new Bitmap( (int)str_size.Width * 2, (int)str_size.Height * 2, System.Drawing.Imaging.PixelFormat.Format24bppRgb );
            using ( Graphics g = Graphics.FromImage( test ) ) {
                g.DrawString(
                    "Qjp",
                    font,
                    Brushes.Black,
                    new PointF( str_size.Width / 2, draw_pos ) );
            }
            BitmapData bdat = test.LockBits(
                new Rectangle( 0, 0, test.Width, test.Height ),
                ImageLockMode.ReadOnly,
                PixelFormat.Format24bppRgb );
            int stride = bdat.Stride;
            byte* dat = (byte*)bdat.Scan0;
            bool found = false;
            int firsty = draw_pos;
            for ( int y = 0; y < bdat.Height; y++ ) {
                int pos = y * stride - 3;
                for ( int x = 0; x < bdat.Width; x++ ) {
                    pos += 3;
                    if ( dat[pos] == 0 ) {
                        found = true;
                        break;
                    }
                }
                if ( found ) {
                    firsty = y;
                    break;
                }
            }
            found = false;
            int lasty = draw_pos;
            for ( int y = bdat.Height - 1; y >= 0; y-- ) {
                int pos = y * stride - 3;
                for ( int x = 0; x < bdat.Width; x++ ) {
                    pos++;
                    if ( dat[pos] == 0 ) {
                        found = true;
                        break;
                    }
                }
                if ( found ) {
                    lasty = y;
                    break;
                }
            }
            return (lasty + firsty) / 2 - draw_pos;
        }

        public static SizeF MeasureString( string item, Font font ) {
            using ( Bitmap test = new Bitmap( 10, 10 ) )
            using ( Graphics g = Graphics.FromImage( test ) ) {
                return g.MeasureString( item, font );
            }
        }

        /// <summary>
        /// 文字列itemをfontを用いて描画したとき、幅widthピクセルに収まるようにitemを調節したものを返します。
        /// 例えば"1 Voice"→"1 Voi..."ナド。
        /// </summary>
        /// <param name="item"></param>
        /// <param name="font"></param>
        /// <param name="width"></param>
        /// <returns></returns>
        public static string TrimString( string item, Font font, int width ) {
            string edited = item;
            int delete_count = item.Length;
            while ( true ) {
                SizeF measured = MeasureString( edited, font );
                if ( measured.Width <= width ) {
                    return edited;
                }
                delete_count -= 1;
                if ( delete_count > 0 ) {
                    edited = item.Substring( 0, delete_count ) + "...";
                } else {
                    return edited;
                }
            }
        }


        public static void DebugWriteLine( string message ) {
#if DEBUG
            Console.WriteLine( message );
#endif
        }

        /// <summary>
        /// FormMainを識別するID
        /// </summary>
        public static string ID {
            get {
                return m_id;
            }
        }

        public static string _( string id ) {
            return Boare.Lib.AppUtil.Messaging.GetMessage( id );
        }

        public static ScriptInvoker LoadScript( string file ) {
#if DEBUG
            AppManager.DebugWriteLine( "Common.LoadScript(string)" );
            AppManager.DebugWriteLine( "    File.GetLastWriteTimeUtc( file )=" + File.GetLastWriteTimeUtc( file ) );
#endif
            ScriptInvoker ret = new ScriptInvoker();
            ret.ScriptFile = file;
            ret.FileTimestamp = File.GetLastWriteTimeUtc( file );
            // スクリプトの記述のうち、以下のリストに当てはまる部分は空文字に置換される
            string config_file = ConfigFileNameFromScriptFileName( file );
            string script = "";
            using ( StreamReader sr = new StreamReader( file ) ) {
                script = sr.ReadToEnd();
            }
            foreach ( string s in AppManager._USINGS ) {
                script = script.Replace( s, "" );
            }

            string code = "";
            foreach ( string s in AppManager._USINGS ) {
                code += s;
            }
            code += "namespace Boare.CadenciiScript{";
            code += script;
            code += "}";
            ret.ErrorMessage = "";
            CompilerResults results;
            Assembly testAssembly = AppManager.CompileScript( code, out results );
            if ( testAssembly == null | results == null ) {
                ret.ErrorMessage = "failed compiling";
                return ret;
            }
            if ( results.Errors.Count != 0 ) {
                for ( int i = 0; i < results.Errors.Count; i++ ) {
                    ret.ErrorMessage += _( "line" ) + ":" + results.Errors[i].Line + " " + results.Errors[i].ErrorText + Environment.NewLine;
                }
                if ( results.Errors.HasErrors ) {
                    return ret;
                }
            }

            if ( testAssembly != null ) {
                foreach ( Type implemented in testAssembly.GetTypes() ) {
                    MethodInfo tmi = implemented.GetMethod( "Edit", new Type[] { typeof( VsqFile ) } );
                    if ( tmi != null && tmi.IsStatic && tmi.IsPublic && tmi.ReturnType.Equals( typeof( bool ) ) ) {
                        ret.ScriptType = implemented;
                        ret.EditVsqDelegate = (EditVsqScriptDelegate)Delegate.CreateDelegate( typeof( EditVsqScriptDelegate ), tmi );
                        ret.Serializer = new XmlStaticMemberSerializer( implemented );

                        if ( !File.Exists( config_file ) ) {
                            continue;
                        }

                        // 設定ファイルからDeserialize
                        using ( FileStream fs = new FileStream( config_file, FileMode.Open ) ) {
                            ret.Serializer.Deserialize( fs );
                        }
                    }
                    tmi = implemented.GetMethod( "Edit", new Type[] { typeof( VsqFileEx ) } );
                    if ( tmi != null && tmi.IsStatic && tmi.IsPublic && tmi.ReturnType.Equals( typeof( bool ) ) ) {
                        ret.ScriptType = implemented;
                        ret.EditVsqDelegate2 = (EditVsqScriptDelegate2)Delegate.CreateDelegate( typeof( EditVsqScriptDelegate2 ), tmi );
                        ret.Serializer = new XmlStaticMemberSerializer( implemented );

                        if ( !File.Exists( config_file ) ) {
                            continue;
                        }

                        // 設定ファイルからDeserialize
                        using ( FileStream fs = new FileStream( config_file, FileMode.Open ) ) {
                            ret.Serializer.Deserialize( fs );
                        }
                    }
                }
            }
            return ret;
        }

        /// <summary>
        /// スクリプトを実行します
        /// </summary>
        /// <param name="evsd"></param>
        public static bool InvokeScript( ScriptInvoker script_invoker ) {
            if ( script_invoker != null && (script_invoker.EditVsqDelegate != null || script_invoker.EditVsqDelegate2 != null) ) {
                try {
                    VsqFileEx work = (VsqFileEx)m_vsq.Clone();
                    bool ret = false;
                    if ( script_invoker.EditVsqDelegate != null ) {
                        ret = script_invoker.EditVsqDelegate.Invoke( work );
                    } else if ( script_invoker.EditVsqDelegate2 != null ) {
                        ret = script_invoker.EditVsqDelegate2.Invoke( work );
                    }
                    if ( !ret ) {
                        MessageBox.Show( _( "Script aborted" ), "Cadencii", MessageBoxButtons.OK, MessageBoxIcon.Information );
                    } else {
                        CadenciiCommand run = VsqFileEx.generateCommandReplace( work );
                        Register( m_vsq.executeCommand( run ) );
                    }
                    string config_file = ConfigFileNameFromScriptFileName( script_invoker.ScriptFile );
                    using ( FileStream fs = new FileStream( config_file, FileMode.Create ) ) {
                        script_invoker.Serializer.Serialize( fs );
                    }
                    return ret;
                } catch ( Exception ex ) {
                    MessageBox.Show( _( "Script runtime error:" ) + " " + ex, _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Information );
#if DEBUG
                    AppManager.DebugWriteLine( "    " + ex );
#endif
                }
            } else {
                MessageBox.Show( _( "Script compilation failed." ), _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Exclamation );
            }
            return false;
        }

        private static string ConfigFileNameFromScriptFileName( string script_file ) {
            string dir = Path.Combine( AppManager.ApplicationDataPath, "script" );
            if ( !Directory.Exists( dir ) ) {
                Directory.CreateDirectory( dir );
            }
            return Path.Combine( dir, Path.GetFileNameWithoutExtension( script_file ) + ".config" );
        }

        public static string TempWaveDir {
            get {
                string temp = Path.Combine( Path.GetTempPath(), _TEMPDIR_NAME );
                string dir = Path.Combine( temp, ID );
                if ( !Directory.Exists( temp ) ) {
                    Directory.CreateDirectory( temp );
                }
                if ( !Directory.Exists( dir ) ) {
                    Directory.CreateDirectory( dir );
                }
                return dir;
            }
        }

        public static bool IsCurveMode {
            get {
                return m_is_curve_mode;
            }
            set {
                bool old = m_is_curve_mode;
                m_is_curve_mode = value;
                if ( old != m_is_curve_mode && SelectedToolChanged != null ) {
                    SelectedToolChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

#if !TREECOM
        /// <summary>
        /// コマンドの履歴を削除します
        /// </summary>
        public static void ClearCommandBuffer() {
            m_commands.Clear();
            m_command_position = -1;
        }

        /// <summary>
        /// アンドゥ処理を行います
        /// </summary>
        public static void Undo() {
#if DEBUG
            AppManager.DebugWriteLine( "CommonConfig.Undo()" );
#endif
            if ( IsUndoAvailable ) {
                ICommand run_src = m_commands[m_command_position];
                CadenciiCommand run = (CadenciiCommand)run_src;
#if DEBUG
                AppManager.DebugWriteLine( "    command type=" + run.Type );
#endif
                if ( run.VsqCommand != null ) {
                    if ( run.VsqCommand.Type == VsqCommandType.DeleteTrack ) {
                        int track = (int)run.VsqCommand.Args[0];
                        if ( track == Selected && track >= 2 ) {
                            Selected = track - 1;
                        }
                    }
                }
                ICommand inv = m_vsq.executeCommand( run );
                m_commands[m_command_position] = inv;
                m_command_position--;
            }
        }

        /// <summary>
        /// リドゥ処理を行います
        /// </summary>
        public static void Redo() {
#if DEBUG
            AppManager.DebugWriteLine( "CommonConfig.Redo()" );
#endif
            if ( IsRedoAvailable ) {
                ICommand run_src = m_commands[m_command_position + 1];
                CadenciiCommand run = (CadenciiCommand)run_src;
                if ( run.VsqCommand != null ) {
                    if ( run.VsqCommand.Type == VsqCommandType.DeleteTrack ) {
                        int track = (int)run.Args[0];
                        if ( track == Selected && track >= 2 ) {
                            Selected = track - 1;
                        }
                    }
                }
                ICommand inv = m_vsq.executeCommand( run );
                m_commands[m_command_position + 1] = inv;
                m_command_position++;
            }
        }

        /// <summary>
        /// コマンドバッファに指定されたコマンドを登録します
        /// </summary>
        /// <param name="command"></param>
        public static void Register( ICommand command ) {
#if DEBUG
            AppManager.DebugWriteLine( "EditorManager.Register; command=" + command );
            AppManager.DebugWriteLine( "    m_commands.Count=" + m_commands.Count );
#endif
            if ( m_command_position == m_commands.Count - 1 ) {
                // 新しいコマンドバッファを追加する場合
                m_commands.Add( command );
                m_command_position = m_commands.Count - 1;
            } else {
                // 既にあるコマンドバッファを上書きする場合
                m_commands[m_command_position + 1] = command;
                for ( int i = m_commands.Count - 1; i >= m_command_position + 2; i-- ) {
                    m_commands.RemoveAt( i );
                }
                m_command_position++;
            }
        }

        /// <summary>
        /// リドゥ操作が可能かどうかを表すプロパティ
        /// </summary>
        public static bool IsRedoAvailable {
            get {
                if ( m_commands.Count > 0 && 0 <= m_command_position + 1 && m_command_position + 1 < m_commands.Count ) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        /// <summary>
        /// アンドゥ操作が可能かどうかを表すプロパティ
        /// </summary>
        public static bool IsUndoAvailable {
            get {
                if ( m_commands.Count > 0 && 0 <= m_command_position && m_command_position < m_commands.Count ) {
                    return true;
                } else {
                    return false;
                }
            }
        }
#endif

        public static EditTool SelectedTool {
            get {
                return m_selected_tool;
            }
            internal set {
                EditTool old = m_selected_tool;
                m_selected_tool = value;
                if ( old != m_selected_tool && SelectedToolChanged != null ) {
                    SelectedToolChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

        #region SelectedBezier
        public static IEnumerable<SelectedBezierPoint> GetSelectedBezierEnumerator() {
            for ( int i = 0; i < m_selected_bezier.Count; i++ ) {
                yield return m_selected_bezier[i];
            }
        }

        /// <summary>
        /// ベジエ点のリストに最後に追加された点の情報を取得します
        /// </summary>
        public static SelectedBezierPoint LastSelectedBezier {
            get {
                if ( m_last_selected_bezier.ChainID < 0 || m_last_selected_bezier.PointID < 0 ) {
                    return null;
                } else {
                    return m_last_selected_bezier;
                }
            }
        }

        /// <summary>
        /// 選択されているベジエ点のリストに、アイテムを追加します
        /// </summary>
        /// <param name="selected">追加する点</param>
        public static void AddSelectedBezier( SelectedBezierPoint selected ) {
            m_last_selected_bezier = selected;
            int index = -1;
            for ( int i = 0; i < m_selected_bezier.Count; i++ ) {
                if ( m_selected_bezier[i].ChainID == selected.ChainID &&
                    m_selected_bezier[i].PointID == selected.PointID ) {
                    index = i;
                    break;
                }
            }
            if ( index >= 0 ) {
                m_selected_bezier[index] = selected;
            } else {
                m_selected_bezier.Add( selected );
            }
        }

        /// <summary>
        /// 選択されているベジエ点のリストを初期化します
        /// </summary>
        public static void ClearSelectedBezier() {
            m_selected_bezier.Clear();
            m_last_selected_bezier.ChainID = -1;
            m_last_selected_bezier.PointID = -1;
        }
        #endregion

        #region SelectedTimesig
        /// <summary>
        /// 選択されている拍子変更イベントのリスト
        /// </summary>
        public static Dictionary<int, SelectedTimesigEntry> SelectedTimesig {
            get {
                return m_selected_timesig;
            }
        }

        public static SelectedTimesigEntry LastSelectedTimesig {
            get {
                if ( m_selected_timesig.ContainsKey( m_last_selected_timesig ) ) {
                    return m_selected_timesig[m_last_selected_timesig];
                } else {
                    return null;
                }
            }
        }

        public static int LastSelectedTimesigBarcount {
            get {
                return m_last_selected_timesig;
            }
        }

        public static void AddSelectedTimesig( int barcount ) {
            ClearSelectedEvent(); //ここ注意！
            ClearSelectedTempo();
            m_last_selected_timesig = barcount;
            if ( !m_selected_timesig.ContainsKey( barcount ) ) {
                foreach ( TimeSigTableEntry tte in m_vsq.TimesigTable ) {
                    if ( tte.BarCount == barcount ) {
                        m_selected_timesig.Add( barcount, new SelectedTimesigEntry( tte, (TimeSigTableEntry)tte.Clone() ) );
                        break;
                    }
                }
            }
        }

        public static void ClearSelectedTimesig() {
            m_selected_timesig.Clear();
            m_last_selected_timesig = -1;
        }
        #endregion

        #region SelectedTempo
        /// <summary>
        /// 選択されているテンポ変更イベントのリスト
        /// </summary>
        public static Dictionary<int, SelectedTempoEntry> SelectedTempo {
            get {
                return m_selected_tempo;
            }
        }

        public static SelectedTempoEntry LastSelectedTempo {
            get {
                if ( m_selected_tempo.ContainsKey( m_last_selected_tempo ) ) {
                    return m_selected_tempo[m_last_selected_tempo];
                } else {
                    return null;
                }
            }
        }

        public static int LastSelectedTempoClock {
            get {
                return m_last_selected_tempo;
            }
        }

        public static void AddSelectedTempo( int clock ) {
            ClearSelectedEvent(); //ここ注意！
            ClearSelectedTimesig();
            m_last_selected_tempo = clock;
            if ( !m_selected_tempo.ContainsKey( clock ) ) {
                foreach ( TempoTableEntry tte in m_vsq.TempoTable ) {
                    if ( tte.Clock == clock ) {
                        m_selected_tempo.Add( clock, new SelectedTempoEntry( tte, (TempoTableEntry)tte.Clone() ) );
                        break;
                    }
                }
            }
        }

        public static void ClearSelectedTempo() {
            m_selected_tempo.Clear();
            m_last_selected_tempo = -1;
        }
        #endregion

        #region SelectedEvent
        public static SelectedEventList SelectedEvent {
            get {
                return m_selected_event;
            }
        }

        public static void RemoveSelectedEvent( int id ) {
            RemoveSelectedEventCor( id, false );
        }

        public static void RemoveSelectedEventSilent( int id ) {
            RemoveSelectedEventCor( id, true );
        }

        private static void RemoveSelectedEventCor( int id, bool silent ) {
            m_selected_event.Remove( id );
            NotePropertyPanel.UpdateValue( m_selected );
        }

        public static void RemoveSelectedEventRange( int[] ids ) {
            m_selected_event.RemoveRange( ids );
            NotePropertyPanel.UpdateValue( m_selected );
        }

        public static void AddSelectedEventRange( int[] ids ) {
            ClearSelectedTempo();
            ClearSelectedTimesig();
            List<int> list = new List<int>( ids );
            VsqEvent[] index = new VsqEvent[ids.Length];
            int count = 0;
            for ( Iterator itr = m_vsq.Track[m_selected].getEventIterator(); itr.hasNext(); ) {
                VsqEvent ev = (VsqEvent)itr.next();
                int find = list.FindIndex( new Predicate<int>( delegate( int arg ){ return arg == ev.InternalID; } ) );
                if ( 0 <= find ) {
                    index[find] = ev;
                    count++;
                }
                if ( count == ids.Length ) {
                    break;
                }
            }
            for ( int i = 0; i < index.Length; i++ ) {
                m_selected_event.Add( new SelectedEventEntry( m_selected, index[i], (VsqEvent)index[i].Clone() ) );
            }
            if ( SelectedEventChanged != null ) {
                SelectedEventChanged( false );
            }
            NotePropertyPanel.UpdateValue( m_selected );
        }

        public static void AddSelectedEvent( int id ) {
            AddSelectedEventCor( id, false );
        }

        public static void AddSelectedEventSilent( int id ) {
            AddSelectedEventCor( id, true );
        }

        private static void AddSelectedEventCor( int id, bool silent ) {
            ClearSelectedTempo();
            ClearSelectedTimesig();
            for ( Iterator itr = m_vsq.Track[m_selected].getEventIterator(); itr.hasNext(); ) {
                VsqEvent ev = (VsqEvent)itr.next();
                if ( ev.InternalID == id ) {
                    m_selected_event.Add( new SelectedEventEntry( m_selected, ev, (VsqEvent)ev.Clone() ) );
                    if ( !silent && SelectedEventChanged != null ) {
                        SelectedEventChanged( false );
                    }
                    break;
                }
            }
            if ( !silent ) {
                NotePropertyPanel.UpdateValue( m_selected );
            }
        }

        public static void ClearSelectedEvent() {
            m_selected_event.Clear();
            NotePropertyPanel.UpdateValue( m_selected );
            if ( SelectedEventChanged != null ) {
                SelectedEventChanged( true );
            }
        }
        #endregion

        public static bool Overlay {
            get {
                return m_overlay;
            }
            set {
                m_overlay = value;
            }
        }

        public static bool getRenderRequired( int track ) {
            return m_render_required[track - 1];
        }

        public static void setRenderRequired( int track, bool value ) {
            m_render_required[track - 1] = value;
        }

        /// <summary>
        /// 現在の編集モード
        /// </summary>
        public static EditMode EditMode {
            get {
                return m_edit_mode;
            }
            set {
                m_edit_mode = value;
            }
        }

        /// <summary>
        /// グリッドを表示するか否かを表すフラグを取得または設定します
        /// </summary>
        public static bool GridVisible {
            get {
                return m_grid_visible;
            }
            set {
                if ( value != m_grid_visible ) {
                    m_grid_visible = value;
                    if ( GridVisibleChanged != null ) {
                        GridVisibleChanged( typeof( AppManager ), new EventArgs() );
                    }
                }
            }
        }

        /// <summary>
        /// 現在のプレビューがリピートモードであるかどうかを表す値を取得または設定します
        /// </summary>
        public static bool IsRepeatMode {
            get {
                return m_repeat_mode;
            }
            set {
                m_repeat_mode = value;
            }
        }

        /// <summary>
        /// 現在プレビュー中かどうかを示す値を取得または設定します
        /// </summary>
        public static bool Playing {
            get {
                return m_playing;
            }
            set {
                bool previous = m_playing;
                m_playing = value;
                if ( previous != m_playing ) {
                    if ( m_playing && PreviewStarted != null ) {
                        int clock = CurrentClock;
                        PreviewStarted( typeof( AppManager ), new EventArgs() );
                    } else if ( !m_playing && PreviewAborted != null ) {
                        PreviewAborted( typeof( AppManager ), new EventArgs() );
                    }
                }
            }
        }

        /// <summary>
        /// _vsq_fileにセットされたvsqファイルの名前を取得します。
        /// </summary>
        public static string FileName {
            get {
                return m_file;
            }
        }

        public static void SaveTo( string file ) {
            if ( m_vsq != null ) {
                string path = Path.GetDirectoryName( file );
                string file2 = Path.Combine( path, Path.GetFileNameWithoutExtension( file ) + ".vsq" );
                m_vsq.writeAsXml( file );
                m_vsq.write( file2 );
                m_file = file;
                EditorConfig.PushRecentFiles( m_file );
            }
        }

        /// <summary>
        /// 位置clockにおけるテンポ、拍子、小節数を取得します
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="numerator"></param>
        /// <param name="denominator"></param>
        /// <param name="bar_count"></param>
        /// <param name="tempo"></param>
        public static void GetTempoAndTimeSignature(
            int clock,
            out int numerator,
            out int denominator,
            out int bar_count,
            out int tempo,
            out int time_sig_index,
            out int tempo_index,
            out int last_clock ) {
            numerator = 4;
            denominator = 4;
            bar_count = 0;
            tempo = 480000;
            time_sig_index = -1;
            tempo_index = -1;
            last_clock = 0;
            if ( m_vsq != null ) {
                int index = -1;
                for ( int i = 0; i < m_vsq.TimesigTable.Count; i++ ) {
                    if ( m_vsq.TimesigTable[i].Clock > clock ) {
                        index = i - 1;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    denominator = m_vsq.TimesigTable[index].Denominator;
                    numerator = m_vsq.TimesigTable[index].Numerator;
                    time_sig_index = index;
                    last_clock = m_vsq.TimesigTable[index].Clock;
                }
                bar_count = m_vsq.getBarCountFromClock( clock );

                index = -1;
                for ( int i = 0; i < m_vsq.TempoTable.Count; i++ ) {
                    if ( m_vsq.TempoTable[i].Clock > clock ) {
                        index = i - 1;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    tempo = m_vsq.TempoTable[index].Tempo;
                    tempo_index = index;
                }
            }
        }

        /// <summary>
        /// 現在の演奏マーカーの位置を取得または設定します。
        /// </summary>
        public static int CurrentClock {
            get {
                return m_current_clock;
            }
            set {
                int old = m_current_clock;
                m_current_clock = value;
                int barcount = m_vsq.getBarCountFromClock( m_current_clock );
                int bar_top_clock = m_vsq.getClockFromBarCount( barcount );
                int numerator = 4;
                int denominator = 4;
                m_vsq.getTimesigAt( m_current_clock, out numerator, out denominator );
                int clock_per_beat = 480 / 4 * denominator;
                int beat = (m_current_clock - bar_top_clock) / clock_per_beat;
                m_current_play_position.BarCount = barcount - m_vsq.getPreMeasure() + 1;
                m_current_play_position.Beat = beat + 1;
                m_current_play_position.Clock = m_current_clock - bar_top_clock - clock_per_beat * beat;
                m_current_play_position.Denominator = denominator;
                m_current_play_position.Numerator = numerator;
                m_current_play_position.Tempo = m_vsq.getTempoAt( m_current_clock );
                if ( old != m_current_clock && CurrentClockChanged != null ) {
                    CurrentClockChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

        /// <summary>
        /// 現在の演奏カーソルの位置(m_current_clockと意味は同じ。CurrentClockが変更されると、自動で更新される)
        /// </summary>
        public static PlayPositionSpecifier PlayPosition {
            get {
                return m_current_play_position;
            }
        }

        /// <summary>
        /// 現在選択されているトラックを取得または設定します
        /// </summary>
        public static int Selected {
            get {
                int tracks = m_vsq.Track.Count;
                if ( tracks <= m_selected ) {
                    m_selected = tracks - 1;
                }
                return m_selected;
            }
            set {
                m_selected = value;
            }
        }

        /// <summary>
        /// vsqファイルを読込みます
        /// </summary>
        /// <param name="file"></param>
        public static void ReadVsq( string file ) {
            m_selected = 1;
            m_file = file;
            VsqFileEx newvsq = null;
            try {
                newvsq = VsqFileEx.readFromXml( file );
            } catch ( Exception ex ) {
#if DEBUG
                AppManager.DebugWriteLine( "EditorManager.ReadVsq; ex=" + ex );
#endif
                return;
            }
            if ( newvsq == null ) {
                return;
            }
            m_vsq = newvsq;
            for ( int i = 0; i < m_render_required.Length; i++ ) {
                if ( i < m_vsq.Track.Count - 1 ) {
                    m_render_required[i] = true;
                } else {
                    m_render_required[i] = false;
                }
            }
            StartMarker = m_vsq.getPreMeasureClocks();
            int bar = m_vsq.getPreMeasure() + 1;
            EndMarker = m_vsq.getClockFromBarCount( bar );
            if ( m_vsq.Track.Count >= 1 ) {
                m_selected = 1;
            } else {
                m_selected = -1;
            }
        }

#if !TREECOM
        /// <summary>
        /// vsqファイル。
        /// </summary>
        public static VsqFileEx VsqFile {
            get {
                return m_vsq;
            }
        }
#endif

        public static void SetVsqFile( VsqFileEx vsq ) {
            m_vsq = vsq;
            for ( int i = 0; i < m_render_required.Length; i++ ) {
                if ( i < m_vsq.Track.Count - 1 ) {
                    m_render_required[i] = true;
                } else {
                    m_render_required[i] = false;
                }
            }
            m_file = "";
            StartMarker = m_vsq.getPreMeasureClocks();
            int bar = m_vsq.getPreMeasure() + 1;
            EndMarker = m_vsq.getClockFromBarCount( bar );
        }

        /// <summary>
        /// 現在表示されているピアノロール画面の右上の、仮想スクリーン上座標で見たときのy座標(pixel)
        /// </summary>
        public static int StartToDrawY {
            get {
                return m_start_to_draw_y;
            }
            set {
                m_start_to_draw_y = value;
            }
        }
        #endregion

        public static void Init() {
            LoadConfig();
            VSTiProxy.Init();
            s_locker = new object();
            SymbolTable.loadDictionary();
            VSTiProxy.CurrentUser = "";

            #region Apply User Dictionary Configuration
            List<ValuePair<string, bool>> current = new List<ValuePair<string, bool>>();
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                current.Add( new ValuePair<string, bool>( SymbolTable.getSymbolTable( i ).getName(), false ) );
            }
            List<ValuePair<string, bool>> config_data = new List<ValuePair<string, bool>>();
            for ( int i = 0; i < EditorConfig.UserDictionaries.Count; i++ ) {
                string[] spl = EditorConfig.UserDictionaries[i].Split( "\t".ToCharArray(), 2 );
                config_data.Add( new ValuePair<string, bool>( spl[0], (spl[1] == "T" ? true : false) ) );
#if DEBUG
                AppManager.DebugWriteLine( "    " + spl[0] + "," + spl[1] );
#endif
            }
            List<KeyValuePair<string, bool>> common = new List<KeyValuePair<string, bool>>();
            for ( int i = 0; i < config_data.Count; i++ ) {
                for ( int j = 0; j < current.Count; j++ ) {
                    if ( config_data[i].Key == current[j].Key ) {
                        current[j].Value = true; //こっちのboolは、AppManager.EditorConfigのUserDictionariesにもKeyが含まれているかどうかを表すので注意
                        common.Add( new KeyValuePair<string, bool>( config_data[i].Key, config_data[i].Value ) );
                        break;
                    }
                }
            }
            for ( int i = 0; i < current.Count; i++ ) {
                if ( !current[i].Value ) {
                    common.Add( new KeyValuePair<string, bool>( current[i].Key, false ) );
                }
            }
            SymbolTable.changeOrder( common.ToArray() );
            #endregion

            Boare.Lib.AppUtil.Messaging.LoadMessages();
            Boare.Lib.AppUtil.Messaging.Language = EditorConfig.Language;

            KeySoundPlayer.Init();
            PaletteToolServer.Init();

#if !TREECOM
            m_id = bocoree.misc.getmd5( DateTime.Now.ToBinary().ToString() );
            string log = Path.Combine( TempWaveDir, "run.log" );
#endif
            NotePropertyPanel = new PropertyPanel();
            NotePropertyDlg = new FormNoteProperty();
            NotePropertyDlg.Controls.Add( NotePropertyPanel );
            NotePropertyPanel.Dock = DockStyle.Fill;
        }

        public static string GetShortcutDisplayString( Keys[] keys ) {
            string ret = "";
            List<Keys> list = new List<Keys>( keys );
            if ( list.Contains( Keys.Menu ) ) {
                ret = new string( '\x2318', 1 );
            }
            if ( list.Contains( Keys.Control ) ) {
                ret += (ret == "" ? "" : "+") + "Ctrl";
            }
            if ( list.Contains( Keys.Shift ) ) {
                ret += (ret == "" ? "" : "+") + "Shift";
            }
            if ( list.Contains( Keys.Alt ) ) {
                ret += (ret == "" ? "" : "+") + "Alt";
            }
            List<Keys> list2 = new List<Keys>();
            foreach ( Keys key in keys ) {
                AppManager.DebugWriteLine( "    " + key );
                if ( key != Keys.Control && key != Keys.Shift && key != Keys.Alt ) {
                    list2.Add( key );
                }
            }
            list2.Sort();
            for ( int i = 0; i < list2.Count; i++ ) {
                ret += (ret == "" ? "" : "+") + GetKeyDisplayString( list2[i] );
            }
            return ret;
        }

        private static string GetKeyDisplayString( Keys key ) {
            switch ( key ) {
                case Keys.PageDown:
                    return "PgDn";
                case Keys.PageUp:
                    return "PgUp";
                case Keys.D0:
                    return "0";
                case Keys.D1:
                    return "1";
                case Keys.D2:
                    return "2";
                case Keys.D3:
                    return "3";
                case Keys.D4:
                    return "4";
                case Keys.D5:
                    return "5";
                case Keys.D6:
                    return "6";
                case Keys.D7:
                    return "7";
                case Keys.D8:
                    return "8";
                case Keys.D9:
                    return "9";
                case Keys.Menu:
                    return new string( '\x2318', 1 );
                default:
                    return key.ToString();
            }
        }

        /// <summary>
        /// オブジェクトをシリアライズし，クリップボードに格納するための文字列を作成します
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        private static string GetSerializedText( object obj ) {
            string str = "";
            using ( MemoryStream ms = new MemoryStream() ) {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize( ms, obj );
                ms.Seek( 0, SeekOrigin.Begin );
                byte[] arr = new byte[ms.Length];
                ms.Read( arr, 0, arr.Length );
                str = _CLIP_PREFIX + ":" + obj.GetType().FullName + ":" + Convert.ToBase64String( arr );
            }
            return str;
        }

        /// <summary>
        /// クリップボードに格納された文字列を元に，デシリアライズされたオブジェクトを取得します
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        private static object GetDeserializedObjectFromText( string s ) {
            if ( s.StartsWith( _CLIP_PREFIX ) ) {
                int index = s.IndexOf( ":" );
                index = s.IndexOf( ":", index + 1 );
                object ret = null;
                try {
                    using ( MemoryStream ms = new MemoryStream( Convert.FromBase64String( s.Substring( index + 1 ) ) ) ) {
                        BinaryFormatter bf = new BinaryFormatter();
                        ret = bf.Deserialize( ms );
                    }
                } catch {
                    ret = null;
                }
                return ret;
            } else {
                return null;
            }
        }

        public static void ClearClipBoard() {
            if ( Clipboard.ContainsText() ) {
                if ( Clipboard.GetText().StartsWith( _CLIP_PREFIX ) ) {
                    Clipboard.Clear();
                }
            }
        }

        public static void SetClipboard( ClipboardEntry item ) {
            string clip = GetSerializedText( item );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }

        #region VsqEvent用のクリップボード管理
        public static List<VsqEvent> GetCopiedEvent() {
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
#if DEBUG
                    AppManager.DebugWriteLine( "typename=" + typename );
#endif
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        List<VsqEvent> ret = null;
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            ret = ce.Event;
                        } catch {
                            ret = null;
                        }
                        if ( ret != null ) {
                            return ret;
                        }
                    }
                }
            }
            return new List<VsqEvent>();
        }

        public static void SetCopiedEvent( List<VsqEvent> item ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Event = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region TempoTableEntry用のクリップボード管理
        public static List<TempoTableEntry> GetCopiedTempo( out int copy_started_clock ) {
            List<TempoTableEntry> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Tempo;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new List<TempoTableEntry>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedTempo( List<TempoTableEntry> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Tempo = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region TimeSigTableEntry用のクリップボード管理
        public static List<TimeSigTableEntry> GetCopiedTimesig( out int copy_started_clock ) {
            List<TimeSigTableEntry> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Timesig;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new List<TimeSigTableEntry>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedTimesig( List<TimeSigTableEntry> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Timesig = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region CurveType, List<BPPair>用のクリップボード管理
        public static Dictionary<CurveType, List<BPPair>> GetCopiedCurve( out int copy_started_clock ) {
            Dictionary<CurveType, List<BPPair>> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Curve;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new Dictionary<CurveType, List<BPPair>>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedCurve( Dictionary<CurveType, List<BPPair>> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Curve = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region CurveType, List<BezierChain>用のクリップボード管理
        public static Dictionary<CurveType, List<BezierChain>> GetCopiedBezier( out int copy_started_clock ) {
            Dictionary<CurveType, List<BezierChain>> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Bezier;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new Dictionary<CurveType, List<BezierChain>>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedBezier( Dictionary<CurveType, List<BezierChain>> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Bezier = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        public static Assembly CompileScript( string code, out CompilerResults results ) {
            CSharpCodeProvider provider = new CSharpCodeProvider();
            string path = Application.StartupPath;
            CompilerParameters parameters = new CompilerParameters( new string[] {
                Path.Combine( path, "Boare.Lib.Vsq.dll" ),
                Path.Combine( path, "Cadencii.exe" ),
                Path.Combine( path, "Boare.Lib.Media.dll" ),
                Path.Combine( path, "Boare.Lib.AppUtil.dll" ),
                Path.Combine( path, "bocoree.dll" ) } );
            parameters.ReferencedAssemblies.Add( "System.Windows.Forms.dll" );
            parameters.ReferencedAssemblies.Add( "System.dll" );
            parameters.ReferencedAssemblies.Add( "System.Drawing.dll" );
            parameters.ReferencedAssemblies.Add( "System.Xml.dll" );
            parameters.GenerateInMemory = true;
            parameters.GenerateExecutable = false;
            parameters.IncludeDebugInformation = true;
            try {
                results = provider.CompileAssemblyFromSource( parameters, code );
                return results.CompiledAssembly;
            } catch ( Exception ex ){
#if DEBUG
                AppManager.DebugWriteLine( "AppManager.CompileScript; ex=" + ex );
#endif
                results = null;
                return null;
            }
        }

        /// <summary>
        /// アプリケーションデータの保存位置を取得します
        /// Gets the path for application data
        /// </summary>
        public static string ApplicationDataPath {
            get {
                string dir = Path.Combine( Environment.GetFolderPath( Environment.SpecialFolder.LocalApplicationData ), "Boare" );
                if ( !Directory.Exists( dir ) ) {
                    Directory.CreateDirectory( dir );
                }
                string dir2 = Path.Combine( dir, _CONFIG_DIR_NAME );
                if ( !Directory.Exists( dir2 ) ) {
                    Directory.CreateDirectory( dir2 );
                }
                return dir2;
            }
        }

        /// <summary>
        /// 位置クオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetPositionQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.PositionQuantize, EditorConfig.PositionQuantizeTriplet );
        }

        /// <summary>
        /// 音符長さクオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetLengthQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.LengthQuantize, EditorConfig.LengthQuantizeTriplet );
        }

        public static void SaveConfig() {
            // ユーザー辞書の情報を取り込む
            EditorConfig.UserDictionaries.Clear();
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                EditorConfig.UserDictionaries.Add( SymbolTable.getSymbolTable( i ).getName() + "\t" + (SymbolTable.getSymbolTable( i ).isEnabled() ? "T" : "F") );
            }

            string file = Path.Combine( ApplicationDataPath, _CONFIG_FILE_NAME );
            try {
                EditorConfig.Serialize( EditorConfig, file );
            } catch {
            }
        }

        public static void LoadConfig() {
            string config_file = Path.Combine( ApplicationDataPath, _CONFIG_FILE_NAME );
            EditorConfig ret = null;
            if ( File.Exists( config_file ) ) {
                try {
                    ret = EditorConfig.Deserialize( EditorConfig, config_file );
                } catch {
                    ret = null;
                }
            } else {
                config_file = Path.Combine( Application.StartupPath, _CONFIG_FILE_NAME );
                if ( File.Exists( config_file ) ) {
                    try {
                        ret = EditorConfig.Deserialize( EditorConfig, config_file );
                    } catch {
                        ret = null;
                    }
                }
            }
            if ( ret == null ) {
                ret = new EditorConfig();
            }
            EditorConfig = ret;
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                SymbolTable st = SymbolTable.getSymbolTable( i );
                bool found = false;
                foreach ( string s in EditorConfig.UserDictionaries ) {
                    string[] spl = s.Split( "\t".ToCharArray(), 2 );
                    if ( st.getName() == spl[0] ) {
                        found = true;
                        break;
                    }
                }
                if ( !found ) {
                    EditorConfig.UserDictionaries.Add( st.getName() + "\tT" );
                }
            }
            MidiPlayer.DeviceGeneral = (uint)EditorConfig.MidiDeviceGeneral.PortNumber;
            MidiPlayer.DeviceMetronome = (uint)EditorConfig.MidiDeviceMetronome.PortNumber;
            MidiPlayer.NoteBell = EditorConfig.MidiNoteBell;
            MidiPlayer.NoteNormal = EditorConfig.MidiNoteNormal;
            MidiPlayer.PreUtterance = EditorConfig.MidiPreUtterance;
            MidiPlayer.ProgramBell = EditorConfig.MidiProgramBell;
            MidiPlayer.ProgramNormal = EditorConfig.MidiProgramNormal;
            MidiPlayer.RingBell = EditorConfig.MidiRingBell;
        }

        public static VsqID getSingerIDUtau( string name ) {
            VsqID ret = new VsqID( 0 );
            ret.type = VsqIDType.Singer;
            int index = -1;
            for ( int i = 0; i < EditorConfig.UtauSingers.Count; i++ ) {
                if ( EditorConfig.UtauSingers[i].VOICENAME == name ) {
                    index = i;
                    break;
                }
            }
            if ( index >= 0 ) {
                SingerConfig sc = EditorConfig.UtauSingers[index];
                int lang = 0;//utauは今のところ全部日本語
                ret.IconHandle = new IconHandle();
                ret.IconHandle.IconID = "$0701" + index.ToString( "0000" );
                ret.IconHandle.IDS = sc.VOICENAME;
                ret.IconHandle.Index = 0;
                ret.IconHandle.Language = lang;
                ret.IconHandle.Length = 1;
                ret.IconHandle.Original = sc.Original;
                ret.IconHandle.Program = sc.Program;
                ret.IconHandle.Caption = "";
                return ret;
            } else {
                ret.IconHandle = new IconHandle();
                ret.IconHandle.Program = 0;
                ret.IconHandle.Language = 0;
                ret.IconHandle.IconID = "$0701" + 0.ToString( "0000" );
                ret.IconHandle.IDS = "Unknown";
                ret.type = VsqIDType.Singer;
                return ret;
            }
        }

        public static SingerConfig getSingerInfoUtau( int program_change ) {
            if ( 0 <= program_change && program_change < EditorConfig.UtauSingers.Count ) {
                return EditorConfig.UtauSingers[program_change];
            } else {
                return null;
            }
        }

        public static string _VERSION {
            get {
                string prefix = "";
                string rev = "";
                // $Id: AppManager.cs 299 2009-07-01 16:54:59Z kbinani $
                string id = GetAssemblyConfigurationAttribute();
                string[] spl0 = id.Split( " ".ToCharArray(), StringSplitOptions.RemoveEmptyEntries );
                if ( spl0.Length >= 3 ) {
                    string s = spl0[2];
#if DEBUG
                    AppManager.DebugWriteLine( "AppManager.get__VERSION; s=" + s );
#endif
                    string[] spl = s.Split( " ".ToCharArray(), StringSplitOptions.RemoveEmptyEntries );
                    if ( spl.Length > 0 ) {
                        rev = spl[0];
                    }
                }
                if ( rev == "" ) {
                    rev = "?";
                }
#if DEBUG
                prefix = "(rev: " + rev + "; build: debug)";
#else
                prefix = "(rev: " + rev + "; build: release)";
#endif
                return GetAssemblyFileVersion( typeof( AppManager ) ) + " " + prefix;
            }
        }

        public static string GetAssemblyConfigurationAttribute() {
            Assembly a = Assembly.GetAssembly( typeof( AppManager ) );
            AssemblyConfigurationAttribute attr = (AssemblyConfigurationAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyConfigurationAttribute ) );
#if DEBUG
            AppManager.DebugWriteLine( "GetAssemblyConfigurationAttribute; attr.Configuration=" + attr.Configuration );
#endif
            return attr.Configuration;
        }

        public static string GetAssemblyFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return afva.Version;
        }

        public static string GetAssemblyNameAndFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return a.GetName().Name + " v" + afva.Version;
        }

        public static SolidBrush HilightBrush {
            get {
                return s_hilight_brush;
            }
        }

        public static Color HilightColor {
            get {
                return s_hilight_brush.Color;
            }
            set {
                s_hilight_brush = new SolidBrush( value );
            }
        }

        /// <summary>
        /// ベースとなるテンポ。
        /// </summary>
        public static int BaseTempo {
            get {
                return s_base_tempo;
            }
            set {
                s_base_tempo = value;
            }
        }
    }

}
