﻿/*
 * VsqUtil.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.IO;
using System.Text;
using System.Collections.Generic;

using bocoree;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// Static library to treat the information of VOCALOID/VOCALOID2 system.
    /// </summary>
    public static class VocaloSysUtil {
        private static bool s_initialized = false;

        private static string s_dll_path2 = "";
        private static string s_exp_db_dir2 = "";
        private static SingerConfigSys s_singer_config2;
        private static string s_editor_path2 = "";

        private static string s_dll_path1 = "";
        private static string s_exp_db_dir1 = "";
        private static SingerConfigSys s_singer_config1;
        private static string s_editor_path1 = "";
        
        public static double getAmplifyCoeffFromPanLeft( int pan ) {
            return pan / -64.0 + 1.0;
        }

        public static double getAmplifyCoeffFromPanRight( int pan ) {
            return pan / 64.0 + 1.0;
        }

        public static double getAmplifyCoeffFromFeder( int feder ) {
            return Math.Exp( -1.26697245e-02 + 1.18448420e-01 * feder / 10.0 );
        }

        /// <summary>
        /// Gets the list of installed VOCALOID2 singers which is installed in the computer.
        /// </summary>
        /// <returns></returns>
        public static SingerConfig[] getInstalledSingers2() {
            return s_singer_config2.getInstalledSingers();
        }

        public static SingerConfig[] getInstalledSingers1() {
            return s_singer_config1.getInstalledSingers();
        }

        /// <summary>
        /// Gets the list of singer configs.
        /// </summary>
        /// <returns></returns>
        public static SingerConfig[] getSingerConfigs2() {
            return s_singer_config2.getSingerConfigs();
        }

        public static SingerConfig[] getSingerConfigs1() {
            return s_singer_config1.getSingerConfigs();
        }

        /// <summary>
        /// Gets the path of VOCALOID Editor.
        /// </summary>
        public static string getEditorPath2() {
            if ( !s_initialized ) {
                init();
            }
            return s_editor_path2;
        }

        public static string getEditorPath1() {
            if ( !s_initialized ) {
                init();
            }
            return s_editor_path1;
        }

        /// <summary>
        /// Gets the voice language of specified program change
        /// </summary>
        /// <param name="program_change">program change to get the voice language</param>
        /// <returns></returns>
        public static VsqVoiceLanguage getLanguage2( string singer ) {
            VocaloSingerConfigSys sys2 = (VocaloSingerConfigSys)s_singer_config2;
            return sys2.getLanguage( singer );
        }

        /// <summary>
        /// Gets the name of original singer of specified program change.
        /// </summary>
        /// <param name="program_change"></param>
        /// <returns></returns>
        public static string getOriginalSinger2( string singer ) {
            VocaloSingerConfigSys sys2 = (VocaloSingerConfigSys)s_singer_config2;
            return sys2.getOriginalSinger( singer );
        }

        public static string getOriginalSinger1( string singer ) {
            VocaloSingerConfigSys sys1 = (VocaloSingerConfigSys)s_singer_config1;
            return sys1.getOriginalSinger( singer );
        }

        /// <summary>
        /// Gets the VsqID of program change.
        /// </summary>
        /// <param name="program_change"></param>
        /// <returns></returns>
        public static VsqID getSingerID2( string singer ) {
            return s_singer_config2.getSingerID( singer );
        }

        public static VsqID getSingerID1( string singer ) {
            return s_singer_config1.getSingerID( singer );
        }

        /// <summary>
        /// Gets the singer information of pecified program change.
        /// </summary>
        /// <param name="program_change"></param>
        /// <returns></returns>
        public static SingerConfig getSingerInfo2( string singer ) {
            if ( s_singer_config2 == null ) {
                loadSingerConfigs();
            }
            return s_singer_config2.getSingerInfo( singer );
        }

        public static SingerConfig getSingerInfo1( string singer ) {
            if ( s_singer_config1 == null ) {
                loadSingerConfigs();
            }
            return s_singer_config1.getSingerInfo( singer );
        }

        /// <summary>
        /// Loads the singer config from ExpDbPath.
        /// This method will automatically called when GetSingerInfo method or GetSingerID method is called in first time.
        /// </summary>
        public static void loadSingerConfigs() {
            if ( s_singer_config2 == null ) {
                s_singer_config2 = new VocaloSingerConfigSys( getExpDbPath2() );
            }
            if ( s_singer_config1 == null ) {
                s_singer_config1 = new VocaloSingerConfigSys( getExpDbPath1() );
            }
        }

        /// <summary>
        /// Gets the path of VOCALOID2 Playback VSTi dll.
        /// </summary>
        /// <returns></returns>
        public static string getDllPathVsti2() {
            if ( !s_initialized ) {
                init();
            }
            return s_dll_path2;
        }

        /// <summary>
        /// Gets the path of VOCALOID1 Playback VSTi dll.
        /// </summary>
        /// <returns></returns>
        public static string getDllPathVsti1() {
            if ( !s_initialized ) {
                init();
            }
            return s_dll_path1;
        }

        /// <summary>
        /// Initializes the library.
        /// </summary>
        private static void init() {
            try {
#if DEBUG
                bocoree.debug.push_log( "VsqUtil+Init()" );
#endif

                // vocaloid2 dll path
                Microsoft.Win32.RegistryKey regkey_dll = null;
                regkey_dll = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2\\APPLICATION", false );
                if ( regkey_dll == null ) {
                    regkey_dll = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2_DEMO\\APPLICATION", false );
                }
                if ( regkey_dll != null ) {
                    string[] keys = regkey_dll.GetSubKeyNames();
                    for ( int i = 0; i < keys.Length; i++ ) {
                        Microsoft.Win32.RegistryKey key = regkey_dll.OpenSubKey( keys[i], false );
                        if ( key != null ) {
                            string name = (string)key.GetValue( "PATH" );
                            if ( name.ToLower().EndsWith( "\\vocaloid2.dll" ) ) {
                                s_dll_path2 = name;
                            } else if ( name.ToLower().EndsWith( "\\vocaloid2_demo.dll" ) ) {
                                s_dll_path2 = name;
                            } else if ( name.ToLower().EndsWith( "\\vocaloid2.exe" ) ) {
                                s_editor_path2 = name;
                            }
                            key.Close();
                        }
                    }
                    regkey_dll.Close();
                }
                
                // vocaloid1 dll path
                Microsoft.Win32.RegistryKey regkey_dll1 = null;
                regkey_dll1 = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID\\APPLICATION", false );
                if ( regkey_dll1 != null ){
                    string[] keys = regkey_dll1.GetSubKeyNames();
                    for ( int i = 0; i < keys.Length; i++ ) {
                        Microsoft.Win32.RegistryKey key = regkey_dll1.OpenSubKey( keys[i], false );
                        if ( key != null ) {
                            string name = (string)key.GetValue( "PATH" );
                            if ( name.ToLower().EndsWith( "\\vocaloid.dll" ) ) {
                                s_dll_path1 = name;
                            } else if ( name.ToLower().EndsWith( "\\vocaloid.exe" ) ) {
                                s_editor_path1 = name;
                            }
                            key.Close();
                        }
                    }
                    regkey_dll1.Close();
                }

                // voicedbdir for vocaloid2
                Microsoft.Win32.RegistryKey regkey_voicedb2 = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2\\DATABASE\\VOICE", false );
                if ( regkey_voicedb2 == null ) {
                    regkey_voicedb2 = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID2_DEMO\\DATABASE\\VOICE", false );
                }
                if ( regkey_voicedb2 != null ) {
                    string[] keys = regkey_voicedb2.GetSubKeyNames();
                    for ( int i = 0; i < keys.Length; i++ ) {
                        Microsoft.Win32.RegistryKey key = regkey_voicedb2.OpenSubKey( keys[i], false );
                        if ( key != null ) {
                            string name = (string)key.GetValue( "INSTALLDIR" );
                            if ( name.ToLower().EndsWith( "\\voicedbdir" ) ) {
                                s_exp_db_dir2 = name;
                                break;
                            }
                            key.Close();
                        }
                    }
                    regkey_voicedb2.Close();
                }

                // voicedbdir for vocaloid1
                Microsoft.Win32.RegistryKey regkey_voicedb1 = Microsoft.Win32.Registry.LocalMachine.OpenSubKey( "SOFTWARE\\VOCALOID\\DATABASE\\VOICE", false );
                if ( regkey_voicedb1 != null ) {
                    string name = (string)regkey_voicedb1.GetValue( "VOICEDIR" );
                    if ( name != "" && name.ToLower().EndsWith( "\\voicedbdir" ) ) {
                        s_exp_db_dir1 = name;
                    }
                    regkey_voicedb1.Close();
                }

            } catch ( Exception ex ) {
                bocoree.debug.push_log( "VsqUtil.Init" );
                bocoree.debug.push_log( "    ex=" + ex );
            } finally {
                s_initialized = true;
#if DEBUG
                bocoree.debug.push_log( "    s_dll_path1=" + s_dll_path1 );
                bocoree.debug.push_log( "    s_exp_db_dir1=" + s_exp_db_dir1 );
                bocoree.debug.push_log( "    s_editor_path1=" + s_editor_path1 );
                bocoree.debug.push_log( "    s_dll_path2=" + s_dll_path2 );
                bocoree.debug.push_log( "    s_exp_db_dir2=" + s_exp_db_dir2 );
                bocoree.debug.push_log( "    s_editor_path2=" + s_editor_path2 );
#endif
            }
        }

        /// <summary>
        /// Gets the path of directories in which singer expression database is installed.
        /// </summary>
        /// <returns></returns>
        public static string getExpDbPath2() {
            if ( !s_initialized ) {
                init();
            }
            return s_exp_db_dir2;
        }

        public static string getExpDbPath1() {
            if ( !s_initialized ) {
                init();
            }
            return s_exp_db_dir1;
        }
    }

}
