﻿/*
 * BezierPoint.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Drawing;
using System.Xml.Serialization;
using System.ComponentModel;

namespace Boare.Cadencii {

    /// <summary>
    /// ベジエ曲線を構成するデータ点。
    /// </summary>
    [Serializable]
    public class BezierPoint : IComparable<BezierPoint>, ICloneable {
        PointD m_base;
        internal PointD controlLeft;
        internal PointD controlRight;
        BezierControlType m_type_left;
        BezierControlType m_type_right;
        [NonSerialized]
        int m_id;

        [XmlIgnore]
        public int ID {
            get {
                return m_id;
            }
            internal set {
                m_id = value;
            }
        }

        public override String ToString() {
            return "m_base=" + m_base.X + "," + m_base.Y + "\n" +
                "m_control_left=" + controlLeft.X + "," + controlLeft.Y + "\n" +
                "m_control_right=" + controlRight.X + "," + controlRight.Y + "\n" +
                "m_type_left=" + m_type_left + "\n" +
                "m_type_right=" + m_type_right + "\n";
        }

        private BezierPoint() {
        }

        public BezierPoint( PointD p1 ) : this( p1.X, p1.Y ) {
        }

        public BezierPoint( double x, double y ) {
            PointD p = new PointD( x, y );
            m_base = p;
            controlLeft = p;
            controlRight = p;
            m_type_left = BezierControlType.None;
            m_type_right = BezierControlType.None;
        }

        public BezierPoint( PointD p1, PointD left, PointD right ) {
            m_base = p1;
            controlLeft = new PointD( left.X - m_base.X, left.Y - m_base.Y );
            controlRight = new PointD( right.X - m_base.X, right.Y - m_base.Y );
            m_type_left = BezierControlType.None;
            m_type_right = BezierControlType.None;
        }

        public object Clone() {
            BezierPoint result = new BezierPoint( this.Base, this.ControlLeft, this.ControlRight );
            result.controlLeft = this.controlLeft;
            result.controlRight = this.controlRight;
            result.m_type_left = this.m_type_left;
            result.m_type_right = this.m_type_right;
            result.m_id = this.m_id;
            return result;
        }

        public int CompareTo( BezierPoint item ) {
            double thisx = this.Base.X;
            double itemx = item.Base.X;
            if ( thisx > itemx ) {
                return 1;
            } else if ( thisx < itemx ) {
                return -1;
            } else {
                if ( this.ID > item.ID ) {
                    return 1;
                } else if ( this.ID < item.ID ) {
                    return -1;
                } else {
                    return 0;
                }
            }
        }

        public PointD Base {
            get {
                return getBase();
            }
            set {
                setBase( value );
            }
        }

        public PointD getBase() {
            return m_base;
        }

        public void setBase( PointD value ) {
            m_base = value;
        }

        public void SetPosition( BezierPickedSide picked_side, PointD new_position ) {
            if ( picked_side == BezierPickedSide.Base ) {
                this.Base = new_position;
            } else if ( picked_side == BezierPickedSide.Left ) {
                this.controlLeft = new PointD( new_position.X - this.Base.X, new_position.Y - this.Base.Y );
            } else {
                this.controlRight = new PointD( new_position.X - this.Base.X, new_position.Y - this.Base.Y );
            }
        }

        public PointD GetPosition( BezierPickedSide picked_side ) {
            if ( picked_side == BezierPickedSide.Base ) {
                return this.Base;
            } else if ( picked_side == BezierPickedSide.Left ) {
                return this.ControlLeft;
            } else {
                return this.ControlRight;
            }
        }

        public BezierControlType GetControlType( BezierPickedSide picked_side ) {
            if ( picked_side == BezierPickedSide.Left ) {
                return this.ControlLeftType;
            } else if ( picked_side == BezierPickedSide.Right ) {
                return this.ControlRightType;
            } else {
                return BezierControlType.None;
            }
        }

        public PointD ControlLeft {
            get {
                if ( m_type_left != BezierControlType.None ) {
                    return new PointD( m_base.X + controlLeft.X, m_base.Y + controlLeft.Y );
                } else {
                    return m_base;
                }
            }
            set {
                setControlLeft( value );
            }
        }

        public void setControlLeft( PointD value ) {
            controlLeft = new PointD( value.X - m_base.X, value.Y - m_base.Y );
        }

        public PointD ControlRight {
            get {
                if ( m_type_right != BezierControlType.None ) {
                    return new PointD( m_base.X + controlRight.X, m_base.Y + controlRight.Y );
                } else {
                    return m_base;
                }
            }
            set {
                controlRight = new PointD( value.X - m_base.X, value.Y - m_base.Y );
            }
        }

        public BezierControlType ControlLeftType {
            get {
                return m_type_left;
            }
            set {
                setControlLeftType( value );
            }
        }

        public void setControlLeftType( BezierControlType value ) {
            m_type_left = value;
            if ( m_type_left == BezierControlType.Master && m_type_right != BezierControlType.None ) {
                m_type_right = BezierControlType.Master;
            }
        }

        public void setControlRightType( BezierControlType value ) {
            m_type_right = value;
            if ( m_type_right == BezierControlType.Master && m_type_left != BezierControlType.None ) {
                m_type_left = BezierControlType.Master;
            }
        }

        public BezierControlType ControlRightType {
            get {
                return m_type_right;
            }
            set {
                setControlRightType( value );
            }
        }
    }

}
