﻿/*
 * BezierCurves.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;

namespace Boare.Cadencii {

    /// <summary>
    /// AtatchedCurveをXMLシリアライズするためのクラス
    /// </summary>
    public class BezierCurves : ICloneable {
        public List<BezierChain> Dynamics;
        public List<BezierChain> Brethiness;
        public List<BezierChain> Brightness;
        public List<BezierChain> Clearness;
        public List<BezierChain> Opening;
        public List<BezierChain> GenderFactor;
        public List<BezierChain> PortamentoTiming;
        public List<BezierChain> Pitch;
        public List<BezierChain> VibratoRate;
        public List<BezierChain> VibratoDepth;
        public List<BezierChain> Harmonics;
        public List<BezierChain> FX2Depth;
        public List<BezierChain> Reso1Freq;
        public List<BezierChain> Reso1BW;
        public List<BezierChain> Reso1Amp;
        public List<BezierChain> Reso2Freq;
        public List<BezierChain> Reso2BW;
        public List<BezierChain> Reso2Amp;
        public List<BezierChain> Reso3Freq;
        public List<BezierChain> Reso3BW;
        public List<BezierChain> Reso3Amp;
        public List<BezierChain> Reso4Freq;
        public List<BezierChain> Reso4BW;
        public List<BezierChain> Reso4Amp;
        public int MasterTuningInCent = 0;
       
        public BezierCurves() {
            Dynamics = new List<BezierChain>();
            Brethiness = new List<BezierChain>();
            Brightness = new List<BezierChain>();
            Clearness = new List<BezierChain>();
            Opening = new List<BezierChain>();
            GenderFactor = new List<BezierChain>();
            PortamentoTiming = new List<BezierChain>();
            Pitch = new List<BezierChain>();
            VibratoRate = new List<BezierChain>();
            VibratoDepth = new List<BezierChain>();
            Harmonics = new List<BezierChain>();
            FX2Depth = new List<BezierChain>();
            Reso1Freq = new List<BezierChain>();
            Reso1BW = new List<BezierChain>();
            Reso1Amp = new List<BezierChain>();
            Reso2Freq = new List<BezierChain>();
            Reso2BW = new List<BezierChain>();
            Reso2Amp = new List<BezierChain>();
            Reso3Freq = new List<BezierChain>();
            Reso3BW = new List<BezierChain>();
            Reso3Amp = new List<BezierChain>();
            Reso4Freq = new List<BezierChain>();
            Reso4BW = new List<BezierChain>();
            Reso4Amp = new List<BezierChain>();
        }

        public BezierChain GetBezierChain( CurveType curve_type, int chain_id ) {
            List<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.Count; i++ ) {
                if ( list[i].ID == chain_id ) {
                    return list[i];
                }
            }
            return null;
        }

        public void SetBezierChain( CurveType curve_type, int chain_id, BezierChain item ) {
            List<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.Count; i++ ) {
                if ( list[i].ID == chain_id ) {
                    list[i] = item;
                    break;
                }
            }
        }

        /// <summary>
        /// 指定した種類のコントロールカーブにベジエ曲線を追加します。
        /// AddBezierChainとの違い、オーバーラップする部分があれば自動的に結合されます。
        /// chainには2個以上のデータ点が含まれている必要がある
        /// </summary>
        /// <param name="curve"></param>
        /// <param name="chain"></param>
        public void MergeBezierChain( CurveType curve, BezierChain chain ) {
            if ( chain.points.Count <= 1 ) {
                return;
            }
            int chain_start = (int)chain.Start;
            int chain_end = (int)chain.End;

            // まず、全削除する必要のあるBezierChainを検索
            List<int> delete_list = new List<int>();
            List<BezierChain> src = this[curve];
            //foreach ( int id in this[curve].Keys ) {
            for ( int j = 0; j < src.Count; j++ ) {
                //BezierChain bc = this[curve][id];
                BezierChain bc = src[j];
                if ( bc.points.Count <= 0 ) {
                    continue;
                }
                int bc_start = (int)bc.Start;
                int bc_end = (int)bc.End;
                if ( chain_start <= bc_start && bc_end <= chain_end ) {
                    delete_list.Add( bc.ID );
                }
            }

            // 削除を実行
            foreach ( int id in delete_list ) {
                Remove( curve, id );
                //this[curve].Remove( id );
            }

            // マージする必要があるかどうかを検査。
            bool processed = true;
            while ( processed ) {
                processed = false;
                List<BezierChain> list = this[curve];
                //foreach ( int id in this[curve].Keys ) {
                for ( int j = 0; j < list.Count; j++ ) {
                    //BezierChain bc = this[curve][id];
                    BezierChain bc = list[j];
                    int id = bc.ID;
                    int start = (int)bc.Start;
                    int end = (int)bc.End;

                    // 被っている箇所が2箇所以上ある可能性があるので、ifでヒットしてもbreakしない
                    if ( start < chain_start && chain_start <= end && end < chain_end ) {
                        // bcのchain_start ~ endを削除し、chain_startで結合
                        BezierChain bc_edit = bc.ExtractPartialBezier( start, chain_start );
                        bc_edit.ID = bc.ID;
                        int last = bc_edit.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points[last].ControlRightType = BezierControlType.None;
                        chain.points[0].ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        if ( bc_edit.points[last].Base.Y == chain.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( bc_edit.points[last].ControlLeftType != BezierControlType.None ) {
                                bc_edit.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            bc_edit.points[last].ControlRight = chain.points[0].m_control_left;
                            if ( chain.points[0].ControlRightType != BezierControlType.None ) {
                                bc_edit.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }
                        for ( int i = copy_start; i < chain.points.Count; i++ ) {
                            chain.points[i].ID = bc_edit.GetNextId();
                            bc_edit.Add( chain.points[i] );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = bc_edit;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    } else if ( chain_start < start && start <= chain_end && chain_end < end ) {
                        // bcのstart ~ chain_endを削除し、chain_endで結合
                        BezierChain bc_edit = bc.ExtractPartialBezier( chain_end, end );
                        bc_edit.ID = bc.ID;
                        int last = chain.Count - 1;

                        // 接合部分では、制御点無しでステップ変化する
                        bc_edit.points[0].ControlLeftType = BezierControlType.None;
                        chain.points[last].ControlRightType = BezierControlType.None;

                        int copy_end = last;
                        if ( chain.points[last].Base.Y == bc_edit.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain[last].ControlLeftType != BezierControlType.None ) {
                                chain.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            chain.points[last].ControlRight = bc_edit.points[0].m_control_left;
                            if ( bc_edit.points[0].ControlRightType != BezierControlType.None ) {
                                chain.points[last].ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = last - 1;
                        }
                        for ( int i = 0; i <= copy_end; i++ ) {
                            chain.points[i].ID = bc_edit.GetNextId();
                            bc_edit.Add( chain.points[i] );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = bc_edit;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    } else if ( start < chain_start && chain_end < end ) {
                        // bcのchain_start ~ chain_endをchainで置き換え
                        // left + chain + right
                        BezierChain left = bc.ExtractPartialBezier( start, chain_start );
                        BezierChain right = bc.ExtractPartialBezier( chain_end, end );
                        left.ID = bc.ID;

                        // 接合部ではステップ変化
                        left.points[left.Count - 1].ControlRightType = BezierControlType.None;
                        chain.points[0].ControlLeftType = BezierControlType.None;
                        chain.points[chain.Count - 1].ControlRightType = BezierControlType.None;
                        right.points[0].ControlLeftType = BezierControlType.None;

                        int copy_start = 0;
                        int copy_end = chain.Count - 1;

                        if ( left.points[left.Count - 1].Base.Y == chain.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( left.points[left.Count - 1].ControlLeftType != BezierControlType.None ) {
                                left.points[left.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            left.points[left.Count - 1].ControlRight = chain.points[0].m_control_left;
                            if ( chain.points[0].ControlRightType != BezierControlType.None ) {
                                left.points[left.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            copy_start = 1;
                        }

                        if ( chain.points[chain.Count - 1].Base.Y == right.points[0].Base.Y ) {
                            // bcの終点とchainの始点の座標が一致している場合
                            if ( chain.points[chain.Count - 1].ControlLeftType != BezierControlType.None ) {
                                chain.points[chain.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            chain.points[chain.Count - 1].ControlRight = right.points[0].m_control_left;
                            if ( right.points[0].ControlRightType != BezierControlType.None ) {
                                chain.points[chain.Count - 1].ControlLeftType = BezierControlType.Master;
                            }
                            copy_end = chain.Count - 2;
                        }

                        // 追加
                        for ( int i = copy_start; i <= copy_end; i++ ) {
                            chain.points[i].ID = left.GetNextId();
                            left.Add( chain.points[i] );
                        }
                        for ( int i = 0; i < right.points.Count; i++ ) {
                            right.points[i].ID = left.GetNextId();
                            left.Add( right.points[i] );
                        }
                        //this[curve].Remove( id );
                        Remove( curve, id );
                        chain = left;
                        chain_start = (int)chain.Start;
                        chain_end = (int)chain.End;
                        processed = true;
                        break;
                    }
                }
            }

            if ( !processed ) {
                chain.ID = this.GetNextId( curve );
            }
            //this[curve].Add( chain.ID, chain );
            AddBezierChain( curve, chain, chain.ID );
        }

        public bool DeleteBeziers(
            List<CurveType> target_curve,
            int clock_start,
            int clock_end
        ) {
            bool edited = false;
            foreach ( CurveType curve in target_curve ) {
                if ( curve.IsScalar || curve.IsAttachNote ) {
                    continue;
                }
                List<BezierChain> tmp = new List<BezierChain>();
                foreach ( BezierChain bc in this[curve] ) {
                    int len = bc.points.Count;
                    if ( len < 1 ) {
                        continue;
                    }
                    int chain_start = (int)bc.points[0].Base.X;
                    int chain_end;
                    if ( len < 2 ) {
                        chain_end = chain_start;
                    } else {
                        chain_end = (int)bc.points[len - 1].Base.X;
                    }
                    if ( clock_end < chain_start && chain_start < clock_end && clock_end < chain_end ) {
                        // end ~ chain_endを残す
                        BezierChain chain = bc.ExtractPartialBezier( clock_end, chain_end );
                        chain.ID = bc.ID;
                        tmp.Add( chain );
                        edited = true;
                    } else if ( chain_start <= clock_start && clock_end <= chain_end ) {
                        // chain_start ~ startとend ~ chain_endを残す
                        BezierChain chain1 = bc.ExtractPartialBezier( chain_start, clock_start );
                        chain1.ID = bc.ID;
                        BezierChain chain2 = bc.ExtractPartialBezier( clock_end, chain_end );
                        chain2.ID = -1;  // 後で番号をつける
                        tmp.Add( chain1 );
                        tmp.Add( chain2 );
                        edited = true;
                    } else if ( chain_start < clock_start && clock_start < chain_end && clock_end <= chain_end ) {
                        // chain_start ~ startを残す
                        BezierChain chain = bc.ExtractPartialBezier( chain_start, clock_start );
                        chain.ID = bc.ID;
                        tmp.Add( chain );
                        edited = true;
                    } else if ( clock_start <= chain_start && chain_end <= clock_end ) {
                        // 全体を削除
                        edited = true;
                    } else {
                        // 全体を残す
                        tmp.Add( (BezierChain)bc.Clone() );
                    }
                }
                this[curve].Clear();
                foreach ( BezierChain bc in tmp ) {
                    if ( bc.ID >= 0 ) {
                        AddBezierChain( curve, bc, bc.ID );
                    }
                }
                foreach ( BezierChain bc in tmp ) {
                    if ( bc.ID < 0 ) {
                        bc.ID = this.GetNextId( curve );
                        AddBezierChain( curve, bc, bc.ID );
                    }
                }
            }
            return edited;
        }

        public void Remove( CurveType curve_type, int chain_id ) {
            List<BezierChain> list = this[curve_type];
            for ( int i = 0; i < list.Count; i++ ) {
                if ( list[i].ID == chain_id ) {
                    list.RemoveAt( i );
                    break;
                }
            }
        }

        /// <summary>
        /// 指定したコントロールカーブにベジエ曲線を追加します。
        /// </summary>
        /// <param name="curve_type"></param>
        /// <param name="chain"></param>
        public void AddBezierChain( CurveType curve_type, BezierChain chain, int chain_id ) {
            int index = curve_type.Index;
            BezierChain add = (BezierChain)chain.Clone();
            add.ID = chain_id;
            this[curve_type].Add( add );
        }

        public object Clone() {
            BezierCurves ret = new BezierCurves();
            ret.MasterTuningInCent = MasterTuningInCent;
            foreach ( CurveType ct in AppManager.CURVE_USAGE ) {
                List<BezierChain> src = this[ct];
                ret[ct] = new List<BezierChain>();
                for ( int i = 0; i < src.Count; i++ ) {
                    ret[ct].Add( (BezierChain)src[i].Clone() );
                }
            }
            return ret;
        }

        public int GetNextId( CurveType curve_type ) {
            int index = curve_type.Index;
            List<BezierChain> bc = this[curve_type];
            int ret = bc.Count;// m_curves[index].Count;
            /*while ( m_curves[index].ContainsKey( ret ) ) {
                ret++;
            }*/
            bool found = true;
            while ( found ) {
                found = false;
                for ( int i = 0; i < bc.Count; i++ ) {
                    if ( bc[i].ID == ret ) {
                        found = true;
                        ret++;
                        break;
                    }
                }
            }
            return ret;
        }

        public List<BezierChain> this[CurveType curve] {
            get {
                if ( curve == CurveType.BRE ) {
                    return Brethiness;
                } else if ( curve == CurveType.BRI ) {
                    return Brightness;
                } else if ( curve == CurveType.CLE ) {
                    return Clearness;
                } else if ( curve == CurveType.DYN ) {
                    return Dynamics;
                } else if ( curve == CurveType.fx2depth ) {
                    return FX2Depth;
                } else if ( curve == CurveType.GEN ) {
                    return GenderFactor;
                } else if ( curve == CurveType.harmonics ) {
                    return Harmonics;
                } else if ( curve == CurveType.OPE ) {
                    return Opening;
                } else if ( curve == CurveType.Pitch ) {
                    return Pitch;
                } else if ( curve == CurveType.POR ) {
                    return PortamentoTiming;
                } else if ( curve == CurveType.reso1amp ) {
                    return Reso1Amp;
                } else if ( curve == CurveType.reso1bw ) {
                    return Reso1BW;
                } else if ( curve == CurveType.reso1freq ) {
                    return Reso1Freq;
                } else if ( curve == CurveType.reso2amp ) {
                    return Reso2Amp;
                } else if ( curve == CurveType.reso2bw ) {
                    return Reso2BW;
                } else if ( curve == CurveType.reso2freq ) {
                    return Reso2Freq;
                } else if ( curve == CurveType.reso3amp ) {
                    return Reso3Amp;
                } else if ( curve == CurveType.reso3bw ) {
                    return Reso3BW;
                } else if ( curve == CurveType.reso3freq ) {
                    return Reso3Freq;
                } else if ( curve == CurveType.reso4amp ) {
                    return Reso4Amp;
                } else if ( curve == CurveType.reso4bw ) {
                    return Reso4BW;
                } else if ( curve == CurveType.reso4freq ) {
                    return Reso4Freq;
                } else if ( curve == CurveType.VibratoDepth ) {
                    return VibratoDepth;
                } else if ( curve == CurveType.VibratoRate ) {
                    return VibratoRate;
                } else {
                    return null;
                }
            }
            set {
                if ( curve == CurveType.BRE ) {
                    Brethiness = value;
                } else if ( curve == CurveType.BRI ) {
                    Brightness = value;
                } else if ( curve == CurveType.CLE ) {
                    Clearness = value;
                } else if ( curve == CurveType.DYN ) {
                    Dynamics = value;
                } else if ( curve == CurveType.fx2depth ) {
                    FX2Depth = value;
                } else if ( curve == CurveType.GEN ) {
                    GenderFactor = value;
                } else if ( curve == CurveType.harmonics ) {
                    Harmonics = value;
                } else if ( curve == CurveType.OPE ) {
                    Opening = value;
                } else if ( curve == CurveType.Pitch ) {
                    Pitch = value;
                } else if ( curve == CurveType.POR ) {
                    PortamentoTiming = value;
                } else if ( curve == CurveType.reso1amp ) {
                    Reso1Amp = value;
                } else if ( curve == CurveType.reso1bw ) {
                    Reso1BW = value;
                } else if ( curve == CurveType.reso1freq ) {
                    Reso1Freq = value;
                } else if ( curve == CurveType.reso2amp ) {
                    Reso2Amp = value;
                } else if ( curve == CurveType.reso2bw ) {
                    Reso2BW = value;
                } else if ( curve == CurveType.reso2freq ) {
                    Reso2Freq = value;
                } else if ( curve == CurveType.reso3amp ) {
                    Reso3Amp = value;
                } else if ( curve == CurveType.reso3bw ) {
                    Reso3BW = value;
                } else if ( curve == CurveType.reso3freq ) {
                    Reso3Freq = value;
                } else if ( curve == CurveType.reso4amp ) {
                    Reso4Amp = value;
                } else if ( curve == CurveType.reso4bw ) {
                    Reso4BW = value;
                } else if ( curve == CurveType.reso4freq ) {
                    Reso4Freq = value;
                } else if ( curve == CurveType.VibratoDepth ) {
                    VibratoDepth = value;
                } else if ( curve == CurveType.VibratoRate ) {
                    VibratoRate = value;
                }
            }
        }

        /*public BezierCurves( AttachedCurve item ) {
            foreach ( CurveType ct in CURVE_USAGE ) {
                Dictionary<int, BezierChain> dict = item[ct];
                if ( dict != null ) {
                    this[ct] = new BezierChain[item[ct].Count];
                    int i = -1;
                    foreach ( int key in item[ct].Keys ) {
                        i++;
                        this[ct][i] = (BezierChain)item[ct][key].Clone();
                    }
                } else {
                    this[ct] = new BezierChain[0];
                }
            }
        }*/
    }

}
