﻿/*
 * EditorManager.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Windows.Forms;
using System.Xml.Serialization;
using Microsoft.CSharp;

using Boare.Lib.AppUtil;
using Boare.Lib.Media;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    public delegate bool EditVsqScriptDelegate( VsqFile vsq );

    public partial class EditorManager {
        private int m_start_to_draw_y = 0;
#if !TREECOM
        private VsqFileEx m_vsq;
#endif
        private string m_file = "";
        private int m_selected = 1;
        private int m_current_clock = 0;
        private bool m_playing = false;
        private bool m_repeat_mode = false;
        private bool m_saveavi = false;
        private bool m_grid_visible = false;
        private EditMode m_edit_mode = EditMode.None;
        /*/// <summary>
        /// トラックのON/OFFを表す
        /// </summary>
        private bool[] m_is_on = new bool[16] { true, true, true, true, true, true, true, true, true, true, true, true, true, true, true, true };*/
        /// <summary>
        /// トラックのレンダリングが必要かどうかを表すフラグ
        /// </summary>
        private bool[] m_render_required = new bool[16] { false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false };
        /// <summary>
        /// トラックのオーバーレイ表示
        /// </summary>
        private bool m_overlay = true;
        private SelectedEventList/* Dictionary<int, SelectedEventEntry>*/ m_selected_event = new SelectedEventList();
        //private int m_last_selected_event_id = -1;
        /// <summary>
        /// 選択されているテンポ変更イベントのリスト
        /// </summary>
        private Dictionary<int, SelectedTempoEntry> m_selected_tempo = new Dictionary<int, SelectedTempoEntry>();
        private int m_last_selected_tempo = -1;
        /// <summary>
        /// 選択されている拍子変更イベントのリスト
        /// </summary>
        private Dictionary<int, SelectedTimesigEntry> m_selected_timesig = new Dictionary<int, SelectedTimesigEntry>();
        private int m_last_selected_timesig = -1;
        private EditTool m_selected_tool = EditTool.Pencil;
#if !TREECOM
        private List<ICommand> m_commands = new List<ICommand>();
        private int m_command_position = -1;
#endif
        /// <summary>
        /// 選択されているベジエ点のリスト
        /// </summary>
        private List<SelectedBezierPoint> m_selected_bezier = new List<SelectedBezierPoint>();
        /// <summary>
        /// 最後に選択されたベジエ点
        /// </summary>
        private SelectedBezierPoint m_last_selected_bezier = new SelectedBezierPoint();
        /// <summary>
        /// 現在の演奏カーソルの位置(m_current_clockと意味は同じ。CurrentClockが変更されると、自動で更新される)
        /// </summary>
        private PlayPositionSpecifier m_current_play_position;
        /// <summary>
        /// プレビュー再生のスピード
        /// </summary>
        private float m_speed = 1.0f;

        /// <summary>
        /// SelectedRegionが有効かどうかを表すフラグ
        /// </summary>
        public bool SelectedRegionEnabled = false;
        /// <summary>
        /// Ctrlキーを押しながらのマウスドラッグ操作による選択が行われた範囲
        /// </summary>
        public SelectedRegion SelectedRegion;
        /// <summary>
        /// 自動ノーマライズモードかどうかを表す値を取得、または設定します。
        /// </summary> 
        public bool AutoNormalize = false;
        /// <summary>
        /// エンドマーカーの位置(clock)
        /// </summary>
        public int EndMarker = 0;
        public bool EndMarkerEnabled = false;
        /// <summary>
        /// x方向の表示倍率(pixel/clock)
        /// </summary>
        public float ScaleX = 0.1f;
        /// <summary>
        /// スタートマーカーの位置(clock)
        /// </summary>
        public int StartMarker = 0;
        /// <summary>
        /// Bezierカーブ編集モードが有効かどうかを表す
        /// </summary>
        public bool m_is_curve_mode = false;
        public bool StartMarkerEnabled = false;
        /// <summary>
        /// 再生時に自動スクロールするかどうか
        /// </summary>
        public bool AutoScroll = true;
        /// <summary>
        /// 最初のバッファが書き込まれたかどうか
        /// </summary>
        public bool FirstBufferWritten = false;
        /// <summary>
        /// リアルタイム再生で使用しようとしたVSTiが有効だったかどうか。
        /// </summary>
        public bool RendererAvailable = false;
        /// <summary>
        /// プレビュー再生が開始された時刻
        /// </summary>
        public DateTime PreviewStartedTime;
        /// <summary>
        /// 画面左端位置での、仮想画面上の仮想画面左端から測ったピクセル数．FormMain.hScroll.ValueとFormMain.trackBar.Valueで決まる．
        /// </summary>
        public int StartToDrawX;
        public string SelectedPaletteTool = "";
        public ScreenStatus LastScreenStatus = new ScreenStatus();
        public string m_id = "";

        /// <summary>
        /// グリッド線を表示するか否かを表す値が変更された時発生します
        /// </summary>
        public event EventHandler GridVisibleChanged;
        public event EventHandler PreviewStarted;
        public event EventHandler PreviewAborted;
        public event BSimpleDelegate<bool> SelectedEventChanged;
        public event EventHandler SelectedToolChanged;
        /// <summary>
        /// CurrentClockプロパティの値が変更された時発生します
        /// </summary>
        public event EventHandler CurrentClockChanged;
        /// <summary>
        /// このインスタンスが担当するFormMainが閉じられた時発生します
        /// </summary>
        public event AppManager.MainFormClosedEventHandler MainFormClosed;
        /// <summary>
        /// このインスタンスが担当するFormMainにおいてEditorConfigの内容が変更されたとき発生します
        /// </summary>
        public event AppManager.EditorConfigChangedEventHandler EditorConfigChanged;
        /// <summary>
        /// このインスタンスが担当するFormMainの表示を，EditorConfigの内容に応じて変更することが求められたとき発生します
        /// </summary>
        public event EventHandler ApplyEditorConfigRequired;

        private const string _TEMPDIR_NAME = "cadencii";

#if !TREECOM
        public EditorManager(){
            m_id = bocoree.misc.getmd5( DateTime.Now.ToBinary().ToString() );
            string log = Path.Combine( TempWaveDir, "run.log" );
#if DEBUG
            Common.DebugWriteLine( "EditorManager..ctor; log=" + log );
#endif
            bocoree.debug.force_logfile_path( log );
        }
#endif
        /// <summary>
        /// FormMainを識別するID
        /// </summary>
        public string ID {
            get {
                return m_id;
            }
        }

        public string _( string id ) {
            return Boare.Lib.AppUtil.Messaging.GetMessage( id );
        }

        public ScriptInvoker LoadScript( string file ) {
#if DEBUG
            Common.DebugWriteLine( "Common.LoadScript(string)" );
            Common.DebugWriteLine( "    File.GetLastWriteTimeUtc( file )=" + File.GetLastWriteTimeUtc( file ) );
#endif
            ScriptInvoker ret = new ScriptInvoker();
            ret.ScriptFile = file;
            ret.FileTimestamp = File.GetLastWriteTimeUtc( file );
            // スクリプトの記述のうち、以下のリストに当てはまる部分は空文字に置換される
            string config_file = ConfigFileNameFromScriptFileName( file );
            string script = "";
            using ( StreamReader sr = new StreamReader( file ) ) {
                script = sr.ReadToEnd();
            }
            foreach ( string s in AppManager._USINGS ) {
                script = script.Replace( s, "" );
            }

            string code = "";
            foreach ( string s in AppManager._USINGS ) {
                code += s;
            }
            code += "namespace Boare.CadenciiScript{";
            code += "    static class Script{";
            code += "        private static EditorManager s_manager;";
            code += "        public static EditorManager Manager{";
            code += "            get{";
            code += "                return s_manager;";
            code += "            }";
            code += "        }";
            code += "        public static void Init( EditorManager manager ){";
            code += "            s_manager = manager;";
            code += "        }";
            code += "    }";
            code += script;
            code += "}";
            ret.ErrorMessage = "";
            CompilerResults results;
            Assembly testAssembly = AppManager.CompileScript( code, out results );
            if ( results.Errors.Count != 0 ) {
                for ( int i = 0; i < results.Errors.Count; i++ ) {
                    ret.ErrorMessage += _( "line" ) + ":" + results.Errors[i].Line + " " + results.Errors[i].ErrorText + Environment.NewLine;
                }
                if ( results.Errors.HasErrors ) {
                    return ret;
                }
            }

            //EditVsqScriptDelegate ret = null;
            if ( testAssembly != null ) {
                foreach ( Type implemented in testAssembly.GetTypes() ) {
                    MethodInfo tmi = implemented.GetMethod( "Edit", new Type[] { typeof( VsqFile ) } );
                    if ( tmi != null && tmi.IsStatic && tmi.IsPublic && tmi.ReturnType.Equals( typeof( bool ) ) ) {
                        ret.ScriptType = implemented;
                        ret.EditVsqDelegate = (EditVsqScriptDelegate)Delegate.CreateDelegate( typeof( EditVsqScriptDelegate ), tmi );
                        ret.Serializer = new XmlStaticMemberSerializer( implemented );

                        if ( !File.Exists( config_file ) ) {
                            continue;
                        }

                        // 設定ファイルからDeserialize
                        using ( FileStream fs = new FileStream( config_file, FileMode.Open ) ) {
                            ret.Serializer.Deserialize( fs );
                        }
                    }
                    MethodInfo init = implemented.GetMethod( "Init", new Type[]{ typeof( EditorManager ) } );
                    if ( init != null && init.IsPublic && init.IsStatic && init.ReturnType.Equals( typeof( void ) ) ) {
                        InitVsqScriptDelegate ivsd = (InitVsqScriptDelegate)Delegate.CreateDelegate( typeof( InitVsqScriptDelegate ), init );
                        ivsd.Invoke( this );
                    }
                }
            }
            return ret;
        }

        /// <summary>
        /// スクリプトを実行します
        /// </summary>
        /// <param name="evsd"></param>
        public bool InvokeScript( ScriptInvoker script_invoker ) {
            if ( script_invoker != null && script_invoker.EditVsqDelegate != null ) {
                try {
                    VsqFileEx work = (VsqFileEx)m_vsq.Clone();
                    bool ret = script_invoker.EditVsqDelegate.Invoke( work );
                    if ( !ret ) {
                        MessageBox.Show( _( "Script aborted" ), "Cadencii", MessageBoxButtons.OK, MessageBoxIcon.Information );
                    } else {
                        CadenciiCommand run = VsqFileEx.generateCommandReplace( work );
                        Register( m_vsq.executeCommand( run ) );
                    }
                    string config_file = ConfigFileNameFromScriptFileName( script_invoker.ScriptFile );
                    using ( FileStream fs = new FileStream( config_file, FileMode.Create ) ) {
                        script_invoker.Serializer.Serialize( fs );
                    }
                    return ret;
                } catch ( Exception ex ) {
                    MessageBox.Show( _( "Script runtime error:" ) + " " + ex, _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Information );
#if DEBUG
                    Common.DebugWriteLine( "    " + ex );
#endif
                }
            } else {
                MessageBox.Show( _( "Script compilation failed." ), _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Exclamation );
            }
            return false;
        }

        private static string ConfigFileNameFromScriptFileName( string script_file ) {
            string dir = Path.Combine( AppManager.ApplicationDataPath, "script" );
            if ( !Directory.Exists( dir ) ) {
                Directory.CreateDirectory( dir );
            }
            return Path.Combine( dir, Path.GetFileNameWithoutExtension( script_file ) + ".config" );
        }

        /// <summary>
        /// プレビュー再生のスピード
        /// </summary>
        public float Speed {
            get {
                return m_speed;
            }
            set {
                m_speed = value;
            }
        }

        public string TempWaveDir {
            get {
                string temp = Path.Combine( Path.GetTempPath(), _TEMPDIR_NAME );
                string dir = Path.Combine( temp, ID );
                if ( !Directory.Exists( temp ) ) {
                    Directory.CreateDirectory( temp );
                }
                if ( !Directory.Exists( dir ) ) {
                    Directory.CreateDirectory( dir );
                }
                return dir;
            }
        }

        /// <summary>
        /// (FormMain宛てに)EditorConfigの内容に応じて表示内容を変更することを要求します
        /// </summary>
        public void SendApplyEditorConfigMessage() {
            if ( ApplyEditorConfigRequired != null ) {
                ApplyEditorConfigRequired( this, null );
            }
        }

        /// <summary>
        /// (FormRoot宛てに)EditorConfigが変更されたことを知らせます
        /// </summary>
        public void SendEditorConfigChangedMessage() {
            if ( EditorConfigChanged != null ) {
                EditorConfigChanged( this );
            }
        }

        /// <summary>
        /// (FormRoot宛てに)FormMainが閉じられたことを知らせます
        /// </summary>
        public void SendWindowClosedMessage() {
            if ( MainFormClosed != null ) {
                MainFormClosed( this );
            }
        }

        public bool IsCurveMode {
            get {
                return m_is_curve_mode;
            }
            set {
                bool old = m_is_curve_mode;
                m_is_curve_mode = value;
                if ( old != m_is_curve_mode && SelectedToolChanged != null ) {
                    SelectedToolChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

#if !TREECOM
        /// <summary>
        /// コマンドの履歴を削除します
        /// </summary>
        public void ClearCommandBuffer() {
            m_commands.Clear();
            m_command_position = -1;
        }

        /// <summary>
        /// アンドゥ処理を行います
        /// </summary>
        public void Undo() {
#if DEBUG
            Common.DebugWriteLine( "CommonConfig.Undo()" );
#endif
            if ( IsUndoAvailable ) {
                ICommand run_src = m_commands[m_command_position];
                CadenciiCommand run = (CadenciiCommand)run_src;
#if DEBUG
                Common.DebugWriteLine( "    command type=" + run.Type );
#endif
                if ( run.VsqCommand != null ) {
                    if ( run.VsqCommand.Type == VsqCommandType.DeleteTrack ) {
                        int track = (int)run.VsqCommand.Args[0];
                        if ( track == Selected && track >= 2 ) {
                            Selected = track - 1;
                        }
                    }
                }
                ICommand inv = m_vsq.executeCommand( run );
                m_commands[m_command_position] = inv;
                m_command_position--;
            }
        }

        /// <summary>
        /// リドゥ処理を行います
        /// </summary>
        public void Redo() {
#if DEBUG
            Common.DebugWriteLine( "CommonConfig.Redo()" );
#endif
            if ( IsRedoAvailable ) {
                ICommand run_src = m_commands[m_command_position + 1];
                CadenciiCommand run = (CadenciiCommand)run_src;
                if ( run.VsqCommand != null ) {
                    if ( run.VsqCommand.Type == VsqCommandType.DeleteTrack ) {
                        int track = (int)run.Args[0];
                        if ( track == Selected && track >= 2 ) {
                            Selected = track - 1;
                        }
                    }
                }
                ICommand inv = m_vsq.executeCommand( run );
                m_commands[m_command_position + 1] = inv;
                m_command_position++;
            }
        }

        /// <summary>
        /// コマンドバッファに指定されたコマンドを登録します
        /// </summary>
        /// <param name="command"></param>
        public void Register( ICommand command ) {
#if DEBUG
            Common.DebugWriteLine( "EditorManager.Register; command=" + command );
            Common.DebugWriteLine( "    m_commands.Count=" + m_commands.Count );
#endif
            if ( m_command_position == m_commands.Count - 1 ) {
                // 新しいコマンドバッファを追加する場合
                m_commands.Add( command );
                m_command_position = m_commands.Count - 1;
            } else {
                // 既にあるコマンドバッファを上書きする場合
                m_commands[m_command_position + 1] = command;
                for ( int i = m_commands.Count - 1; i >= m_command_position + 2; i-- ) {
                    m_commands.RemoveAt( i );
                }
                m_command_position++;
            }
        }

        /// <summary>
        /// リドゥ操作が可能かどうかを表すプロパティ
        /// </summary>
        public bool IsRedoAvailable {
            get {
                if ( m_commands.Count > 0 && 0 <= m_command_position + 1 && m_command_position + 1 < m_commands.Count ) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        /// <summary>
        /// アンドゥ操作が可能かどうかを表すプロパティ
        /// </summary>
        public bool IsUndoAvailable {
            get {
                if ( m_commands.Count > 0 && 0 <= m_command_position && m_command_position < m_commands.Count ) {
                    return true;
                } else {
                    return false;
                }
            }
        }
#endif

        public EditTool SelectedTool {
            get {
                return m_selected_tool;
            }
            internal set {
                EditTool old = m_selected_tool;
                m_selected_tool = value;
                if ( old != m_selected_tool && SelectedToolChanged != null ) {
                    SelectedToolChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

        #region SelectedBezier
        public IEnumerable<SelectedBezierPoint> GetSelectedBezierEnumerator() {
            for ( int i = 0; i < m_selected_bezier.Count; i++ ) {
                yield return m_selected_bezier[i];
            }
        }

        /// <summary>
        /// ベジエ点のリストに最後に追加された点の情報を取得します
        /// </summary>
        public SelectedBezierPoint LastSelectedBezier {
            get {
                if ( m_last_selected_bezier.ChainID < 0 || m_last_selected_bezier.PointID < 0 ) {
                    return null;
                } else {
                    return m_last_selected_bezier;
                }
            }
        }

        /// <summary>
        /// 選択されているベジエ点のリストに、アイテムを追加します
        /// </summary>
        /// <param name="selected">追加する点</param>
        public void AddSelectedBezier( SelectedBezierPoint selected ) {
            m_last_selected_bezier = selected;
            int index = -1;
            for ( int i = 0; i < m_selected_bezier.Count; i++ ) {
                if ( m_selected_bezier[i].ChainID == selected.ChainID &&
                    m_selected_bezier[i].PointID == selected.PointID ) {
                    index = i;
                    break;
                }
            }
            if ( index >= 0 ) {
                m_selected_bezier[index] = selected;
            } else {
                m_selected_bezier.Add( selected );
            }
        }

        /// <summary>
        /// 選択されているベジエ点のリストを初期化します
        /// </summary>
        public void ClearSelectedBezier() {
            m_selected_bezier.Clear();
            m_last_selected_bezier.ChainID = -1;
            m_last_selected_bezier.PointID = -1;
        }
        #endregion

        #region SelectedTimesig
        /// <summary>
        /// 選択されている拍子変更イベントのリスト
        /// </summary>
        public Dictionary<int, SelectedTimesigEntry> SelectedTimesig {
            get {
                return m_selected_timesig;
            }
        }

        public SelectedTimesigEntry LastSelectedTimesig {
            get {
                if ( m_selected_timesig.ContainsKey( m_last_selected_timesig ) ) {
                    return m_selected_timesig[m_last_selected_timesig];
                } else {
                    return null;
                }
            }
        }

        public int LastSelectedTimesigBarcount {
            get {
                return m_last_selected_timesig;
            }
        }

        public void AddSelectedTimesig( int barcount ) {
            ClearSelectedEvent(); //ここ注意！
            ClearSelectedTempo();
            m_last_selected_timesig = barcount;
            if ( !m_selected_timesig.ContainsKey( barcount ) ) {
                foreach ( TimeSigTableEntry tte in m_vsq.getTimeSigList() ) {
                    if ( tte.BarCount == barcount ) {
                        m_selected_timesig.Add( barcount, new SelectedTimesigEntry( tte, (TimeSigTableEntry)tte.Clone() ) );
                        break;
                    }
                }
            }
        }

        public void ClearSelectedTimesig() {
            m_selected_timesig.Clear();
            m_last_selected_timesig = -1;
        }
        #endregion

        #region SelectedTempo
        /// <summary>
        /// 選択されているテンポ変更イベントのリスト
        /// </summary>
        public Dictionary<int, SelectedTempoEntry> SelectedTempo {
            get {
                return m_selected_tempo;
            }
        }

        public SelectedTempoEntry LastSelectedTempo {
            get {
                if ( m_selected_tempo.ContainsKey( m_last_selected_tempo ) ) {
                    return m_selected_tempo[m_last_selected_tempo];
                } else {
                    return null;
                }
            }
        }

        public int LastSelectedTempoClock {
            get {
                return m_last_selected_tempo;
            }
        }

        public void AddSelectedTempo( int clock ) {
            ClearSelectedEvent(); //ここ注意！
            ClearSelectedTimesig();
            m_last_selected_tempo = clock;
            if ( !m_selected_tempo.ContainsKey( clock ) ) {
                foreach ( TempoTableEntry tte in m_vsq.getTempoList() ) {
                    if ( tte.Clock == clock ) {
                        m_selected_tempo.Add( clock, new SelectedTempoEntry( tte, (TempoTableEntry)tte.Clone() ) );
                        break;
                    }
                }
            }
        }

        public void ClearSelectedTempo() {
            m_selected_tempo.Clear();
            m_last_selected_tempo = -1;
        }
        #endregion

        #region SelectedEvent
        public SelectedEventList SelectedEvent {
            get {
                return m_selected_event;
            }
        }

        public void AddSelectedEvent( int id ) {
            ClearSelectedTempo();
            ClearSelectedTimesig();
            for ( Iterator itr = m_vsq.Track[m_selected].getEventIterator(); itr.hasNext(); ) {
                VsqEvent ev = (VsqEvent)itr.next();
                if ( ev.InternalID == id ) {
                    m_selected_event.Add( new SelectedEventEntry( m_selected, ev, (VsqEvent)ev.Clone() ) );
                    if ( SelectedEventChanged != null ) {
                        SelectedEventChanged( false );
                    }
                    break;
                }
            }
        }

        public void ClearSelectedEvent() {
            m_selected_event.Clear();
            if ( SelectedEventChanged != null ) {
                SelectedEventChanged( true );
            }
        }
        #endregion

        public bool Overlay {
            get {
                return m_overlay;
            }
            set {
                m_overlay = value;
            }
        }

        public bool getRenderRequired( int track ) {
            return m_render_required[track - 1];
        }

        public void setRenderRequired( int track, bool value ) {
            m_render_required[track - 1] = value;
        }

        /// <summary>
        /// 現在の編集モード
        /// </summary>
        public EditMode EditMode {
            get {
                return m_edit_mode;
            }
            set {
                m_edit_mode = value;
            }
        }

        /// <summary>
        /// グリッドを表示するか否かを表すフラグを取得または設定します
        /// </summary>
        public bool GridVisible {
            get {
                return m_grid_visible;
            }
            set {
                if ( value != m_grid_visible ) {
                    m_grid_visible = value;
                    if ( GridVisibleChanged != null ) {
                        GridVisibleChanged( typeof( AppManager ), new EventArgs() );
                    }
                }
            }
        }

        /// <summary>
        /// 現在のプレビューがリピートモードであるかどうかを表す値を取得または設定します
        /// </summary>
        public bool IsRepeatMode {
            get {
                return m_repeat_mode;
            }
            set {
                m_repeat_mode = value;
            }
        }

        /// <summary>
        /// 現在プレビュー中かどうかを示す値を取得または設定します
        /// </summary>
        public bool Playing {
            get {
                return m_playing;
            }
            set {
                bool previous = m_playing;
                m_playing = value;
                if ( previous != m_playing ) {
                    if ( m_playing && PreviewStarted != null ) {
                        int clock = CurrentClock;
                        PreviewStarted( this, new EventArgs() );
                    } else if ( !m_playing && PreviewAborted != null ) {
                        PreviewAborted( this, new EventArgs() );
                    }
                }
            }
        }

        /// <summary>
        /// _vsq_fileにセットされたvsqファイルの名前を取得します。
        /// </summary>
        public string FileName {
            get {
                return m_file;
            }
        }

        public void SaveTo( string file ) {
            if ( m_vsq != null ) {
                string path = Path.GetDirectoryName( file );
                string file2 = Path.Combine( path, Path.GetFileNameWithoutExtension( file ) + ".vsq" );
                m_vsq.writeAsXml( file );
                m_vsq.write( file2 );
                m_file = file;
            }
        }

        /// <summary>
        /// 位置clockにおけるテンポ、拍子、小節数を取得します
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="numerator"></param>
        /// <param name="denominator"></param>
        /// <param name="bar_count"></param>
        /// <param name="tempo"></param>
        public void GetTempoAndTimeSignature(
            int clock,
            out int numerator,
            out int denominator,
            out int bar_count,
            out int tempo,
            out int time_sig_index,
            out int tempo_index,
            out int last_clock ) {
            numerator = 4;
            denominator = 4;
            bar_count = 0;
            tempo = 480000;
            time_sig_index = -1;
            tempo_index = -1;
            last_clock = 0;
            if ( m_vsq != null ) {
                int index = -1;
                for ( int i = 0; i < m_vsq.getTimeSigList().Count; i++ ) {
                    if ( m_vsq.getTimeSigList()[i].Clock > clock ) {
                        index = i - 1;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    denominator = m_vsq.getTimeSigList()[index].Denominator;
                    numerator = m_vsq.getTimeSigList()[index].Numerator;
                    time_sig_index = index;
                    last_clock = m_vsq.getTimeSigList()[index].Clock;
                }
                bar_count = m_vsq.getBarCountFromClock( clock );

                index = -1;
                for ( int i = 0; i < m_vsq.getTempoList().Count; i++ ) {
                    if ( m_vsq.getTempoList()[i].Clock > clock ) {
                        index = i - 1;
                        break;
                    }
                }
                if ( index >= 0 ) {
                    tempo = m_vsq.getTempoList()[index].Tempo;
                    tempo_index = index;
                }
            }
        }

        /// <summary>
        /// 現在の演奏マーカーの位置を取得または設定します。
        /// </summary>
        public int CurrentClock {
            get {
                return m_current_clock;
            }
            set {
                int old = m_current_clock;
                m_current_clock = value;
                int barcount = m_vsq.getBarCountFromClock( m_current_clock );
                int bar_top_clock = m_vsq.getClockFromBarCount( barcount );
                int numerator = 4;
                int denominator = 4;
                m_vsq.getTimesigAt( m_current_clock, out numerator, out denominator );
                int clock_per_beat = 480 / 4 * denominator;
                int beat = (m_current_clock - bar_top_clock) / clock_per_beat;
                m_current_play_position.BarCount = barcount - m_vsq.getPreMeasure() + 1;
                m_current_play_position.Beat = beat + 1;
                m_current_play_position.Clock = m_current_clock - bar_top_clock - clock_per_beat * beat;
                m_current_play_position.Denominator = denominator;
                m_current_play_position.Numerator = numerator;
                m_current_play_position.Tempo = m_vsq.getTempoAt( m_current_clock );
                if ( old != m_current_clock && CurrentClockChanged != null ) {
                    CurrentClockChanged( typeof( AppManager ), new EventArgs() );
                }
            }
        }

        /// <summary>
        /// 現在の演奏カーソルの位置(m_current_clockと意味は同じ。CurrentClockが変更されると、自動で更新される)
        /// </summary>
        public PlayPositionSpecifier PlayPosition {
            get {
                return m_current_play_position;
            }
        }

        /// <summary>
        /// 現在選択されているトラックを取得または設定します
        /// </summary>
        public int Selected {
            get {
                int tracks = m_vsq.Track.Count;
                if ( tracks <= m_selected ) {
                    m_selected = tracks - 1;
                }
                return m_selected;
            }
            set {
                m_selected = value;
            }
        }

        /// <summary>
        /// vsqファイルを読込みます
        /// </summary>
        /// <param name="file"></param>
        public void ReadVsq( string file ) {
            m_selected = 1;
            m_file = file;
            VsqFileEx newvsq = null;
            try {
                newvsq = VsqFileEx.readFromXml( file );
            } catch ( Exception ex ) {
#if DEBUG
                Common.DebugWriteLine( "EditorManager.ReadVsq; ex=" + ex );
#endif
                return;
            }
            if ( newvsq == null ) {
                return;
            }
            m_vsq = newvsq;
            for ( int i = 0; i < m_render_required.Length; i++ ) {
                if ( i < m_vsq.Track.Count - 1 ) {
                    m_render_required[i] = true;
                } else {
                    m_render_required[i] = false;
                }
            }
            StartMarker = m_vsq.getPreMeasureClocks();
            int bar = m_vsq.getPreMeasure() + 1;
            EndMarker = m_vsq.getClockFromBarCount( bar );
            if ( m_vsq.Track.Count >= 1 ) {
                m_selected = 1;
            } else {
                m_selected = -1;
            }
        }

#if !TREECOM
        /// <summary>
        /// vsqファイル。
        /// </summary>
        public VsqFileEx VsqFile {
            get {
                return m_vsq;
            }
        }
#endif

        public void SetVsqFile( VsqFileEx vsq ) {
            m_vsq = vsq;
            for ( int i = 0; i < m_render_required.Length; i++ ) {
                if ( i < m_vsq.Track.Count - 1 ) {
                    m_render_required[i] = true;
                } else {
                    m_render_required[i] = false;
                }
            }
            m_file = "";
            StartMarker = m_vsq.getPreMeasureClocks();
            int bar = m_vsq.getPreMeasure() + 1;
            EndMarker = m_vsq.getClockFromBarCount( bar );
        }

        /// <summary>
        /// 現在表示されているピアノロール画面の右上の、仮想スクリーン上座標で見たときのy座標(pixel)
        /// </summary>
        public int StartToDrawY {
            get {
                return m_start_to_draw_y;
            }
            set {
                m_start_to_draw_y = value;
            }
        }
    }

}
