﻿/*
 * PaletteToolServer.cs
 * Copyright (C) 2009-2010 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.CodeDom.Compiler;
using System.IO;
using System.Reflection;
using System.Windows.Forms;
using Boare.Lib.Vsq;
using Boare.Lib.AppUtil;
using bocoree;
using bocoree.util;

namespace Boare.Cadencii {

    using boolean = System.Boolean;

    public static class PaletteToolServer {
        public static TreeMap<String, object> LoadedTools = new TreeMap<String, object>();

        public static void Init() {
            String path = Path.Combine( Application.StartupPath, "tool" );
            if ( !Directory.Exists( path ) ) {
                return;
            }

            FileInfo[] files = new DirectoryInfo( path ).GetFiles( "*.txt" );
            foreach ( FileInfo file in files ) {
                String code = "";
                using ( StreamReader sr = new StreamReader( file.FullName ) ){
                    code += sr.ReadToEnd();
                }
                CompilerResults results = AppManager.compileScript( code );
                if ( results == null ) {
                    continue;
                }
                Assembly asm = null;
                try {
                    asm = results.CompiledAssembly;
                } catch ( Exception ex ) {
                    continue;
                }
                foreach ( Type t in asm.GetTypes() ) {
                    if ( t.IsClass && t.IsPublic && !t.IsAbstract && t.GetInterface( typeof( IPaletteTool ).FullName ) != null ) {
                        try {
#if DEBUG
                            AppManager.debugWriteLine( "t.FullName=" + t.FullName );
#endif
                            object instance = asm.CreateInstance( t.FullName );
                            String dir = Path.Combine( AppManager.getApplicationDataPath(), "tool" );
                            String cfg = Path.GetFileNameWithoutExtension( file.FullName ) + ".config";
                            String config = Path.Combine( dir, cfg );
                            if ( PortUtil.isFileExists( config ) ) {
                                XmlStaticMemberSerializer xsms = new XmlStaticMemberSerializer( instance.GetType() );
                                FileStream fs = null;
                                boolean errorOnDeserialize = false;
                                try {
                                    fs = new FileStream( config, FileMode.Open, FileAccess.Read );
                                    try {
                                        xsms.Deserialize( fs );
                                    } catch ( Exception ex ) {
                                        errorOnDeserialize = true;
                                    }
                                } catch ( Exception ex ) {
                                    Console.Error.WriteLine( "PaletteToolServer#Init; ex=" + ex );
                                } finally {
                                    if ( fs != null ) {
                                        try {
                                            fs.Close();
                                        } catch ( Exception ex2 ) {
                                            Console.Error.WriteLine( "PaletteToolServer#Init; ex2=" + ex2 );
                                        }
                                    }
                                }
                                if ( errorOnDeserialize ) {
                                    try {
                                        File.Delete( config );
                                    } catch ( Exception ex ) {
                                        Console.Error.WriteLine( "PaletteToolServer#Init; ex=" + ex );
                                    }
                                }
                            }
                            String id = Path.GetFileNameWithoutExtension( file.FullName );
                            LoadedTools.put( id, instance );
#if DEBUG
                            AppManager.debugWriteLine( "PaletteToolServer.Init; id=" + id );
#endif
                        } catch {
                        }
                    }
                }
            }
        }

        private static String _( String id ) {
            return Messaging.getMessage( id );
        }

        public static boolean InvokePaletteTool( String id, int track, int[] vsq_event_intrenal_ids, MouseButtons button ) {
            if ( LoadedTools.containsKey( id ) ) {
                VsqTrack item = (VsqTrack)AppManager.getVsqFile().Track.get( track ).clone();
                object objPal = LoadedTools.get( id );
                if ( objPal == null ) {
                    return false;
                }
                if ( !(objPal is IPaletteTool) ) {
                    return false;
                }
                IPaletteTool pal = (IPaletteTool)objPal;
                boolean edited = false;
                try {
                    edited = pal.edit( item, vsq_event_intrenal_ids, button );
                } catch ( Exception ex ) {
                    AppManager.reportError( ex, 
                                            string.Format( _( "Palette tool '{0}' reported an error.\nPlease copy the exception text and report it to developper." ), id ),
                                            -1 );
                    Console.Error.WriteLine( "PaletteToolServer#InvokePaletteTool; ex=" + ex );
                    edited = false;
                }
                if ( edited ) {
                    CadenciiCommand run = VsqFileEx.generateCommandTrackReplace( track, item, AppManager.getVsqFile().AttachedCurves.get( track - 1 ) );
                    AppManager.register( AppManager.getVsqFile().executeCommand( run ) );
                }
                return edited;
            } else {
                return false;
            }
        }
    }

}
