﻿/*
 * FileDialog.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Runtime.InteropServices;

using bocoree;

namespace Boare.Cadencii {

    public partial class FileDialog : Form {
        private string m_initial_directory = "";
        private string m_current_directory = "";
        private SortType m_sort_type = SortType.Name;
        private List<string> m_history = new List<string>();
        private bool m_multiselect = false;
        private DialogMode m_mode = DialogMode.Open;
        private List<string> m_selected_item = new List<string>();
        private List<string> m_filter = new List<string>();

        public enum DialogMode {
            Open,
            Save,
        }

        private enum SortType {
            Name,
            Size,
            Type,
            Date,
        }

        public FileDialog( DialogMode mode ) {
            m_mode = mode;
            InitializeComponent();
            if ( m_current_directory == "" ) {
                m_current_directory = InitialDirectory;
            }
            UpdateFileList();
#if DEBUG
            OpenFileDialog ofd = new OpenFileDialog();
#endif
        }

        public int FilterIndex {
            get {
                return comboFileType.SelectedIndex + 1;
            }
            set {
                if ( 0 <= value - 1 && value - 1 < comboFileType.Items.Count ) {
                    comboFileType.SelectedIndex = value - 1;
                }
            }
        }

        public string Title {
            get {
                return this.Text;
            }
            set {
                this.Text = value;
            }
        }

        private string CurrentFilterExtension {
            get {
                if ( m_filter.Count <= 0 ) {
                    return "";
                } else {
                    if ( 0 <= comboFileType.SelectedIndex && comboFileType.SelectedIndex < m_filter.Count ) {
                        string filter = m_filter[comboFileType.SelectedIndex];
                        string[] spl = filter.Split( '|' );
                        if ( spl.Length < 2 ) {
                            return "";
                        }
                        int index = spl[1].LastIndexOf( "." );
                        if ( index < 0 ) {
                            return "";
                        }
                        filter = spl[1].Substring( index );
                        if ( filter.StartsWith( ".*" ) ) {
                            return "";
                        }
                        return filter;
                    } else {
                        return "";
                    }
                }
            }
        }

        public string Filter {
            set {
                string[] spl = value.Split( '|' );
                if ( spl.Length % 2 != 0 ) {
                    throw new ApplicationException( "invalid filter format" );
                }
                m_filter.Clear();
                comboFileType.Items.Clear();
                for ( int i = 0; i < spl.Length; i += 2 ) {
                    m_filter.Add( spl[i] + "|" + spl[i + 1] );
                    comboFileType.Items.Add( spl[i] );
                }
                if ( spl.Length > 0 ) {
                    comboFileType.SelectedIndex = 0;
                }
            }
            get {
                string ret = "";
                for ( int i = 0; i < m_filter.Count; i++ ) {
                    ret += (i == 0 ? "" : "|") + m_filter[i];
                }
                return ret;
            }
        }

        private void UpdateFileList() {
            listFiles.Items.Clear();
            if ( listFiles.SmallImageList != null ) {
                listFiles.SmallImageList.Images.Clear();
            } else {
                listFiles.SmallImageList = new ImageList();
            }
            listFiles.SmallImageList.ColorDepth = ColorDepth.Depth32Bit;
            listFiles.SmallImageList.ImageSize = new Size( 16, 16 );
            if ( listFiles.LargeImageList != null ) {
                listFiles.LargeImageList.Images.Clear();
            } else {
                listFiles.LargeImageList = new ImageList();
            }
            listFiles.LargeImageList.ColorDepth = ColorDepth.Depth32Bit;

            if ( m_current_directory == "" ) {
                m_current_directory = InitialDirectory;
            }
            if ( m_current_directory == "MY_COMPUTER" ) {
                listFiles.SmallImageList.Images.Add( "DRIVE", Properties.Resources.drive );
                string[] drives = Environment.GetLogicalDrives();
                for ( int i = 0; i < drives.Length; i++ ) {
                    listFiles.Items.Add( drives[i], "DRIVE" );
                }
            } else {
                // ディレクトリ一覧
                string[] dirnames = Directory.GetDirectories( m_current_directory );
                List<DirectoryInfo> dirs = new List<DirectoryInfo>();
                for ( int i = 0; i < dirnames.Length; i++ ) {
                    dirs.Add( new DirectoryInfo( dirnames[i] ) );
                }
                bool changed = true;
                while ( changed ) {
                    changed = false;
                    for ( int i = 0; i < dirs.Count - 1; i++ ) {
                        bool swap = false;
                        switch ( m_sort_type ) {
                            case SortType.Name:
                                swap = dirs[i].Name.CompareTo( dirs[i + 1].Name ) > 0;
                                break;
                            case SortType.Date:
                                swap = dirs[i].LastWriteTimeUtc > dirs[i + 1].LastWriteTimeUtc;
                                break;
                        }
                        if ( swap ) {
                            DirectoryInfo di = dirs[i];
                            dirs[i] = dirs[i + 1];
                            dirs[i + 1] = di;
                            changed = true;
                        }
                    }
                }
                listFiles.SmallImageList.Images.Add( "FOLDER", Properties.Resources.folder );

                for ( int i = 0; i < dirs.Count; i++ ) {
                    listFiles.Items.Add( dirs[i].Name, "FOLDER" );
                }

                // ファイル一覧
                string[] filenames = Directory.GetFiles( m_current_directory );
                //Dictionary<string, string> extensions = new Dictionary<string, string>();
                List<FileInfo> files = new List<FileInfo>();
                string filter = CurrentFilterExtension;
                for ( int i = 0; i < filenames.Length; i++ ) {
                    if ( filter == "" ) {
                        files.Add( new FileInfo( filenames[i] ) );
                    } else {
                        string ext = Path.GetExtension( filenames[i] ).ToLower();
                        if ( ext == filter ) {
                            files.Add( new FileInfo( filenames[i] ) );
                        }
                    }
                    /*
                    if ( ext != "" ) {
                        if ( !extensions.ContainsKey( ext ) ) {
                            extensions.Add( ext, filenames[i] );
                        }
                    }*/
                }
                changed = true;
                while ( changed ) {
                    changed = false;
                    for ( int i = 0; i < files.Count - 1; i++ ) {
                        bool swap = false;
                        switch ( m_sort_type ) {
                            case SortType.Name:
                                swap = files[i].Name.CompareTo( files[i + 1].Name ) > 0;
                                break;
                            case SortType.Date:
                                swap = files[i].LastWriteTimeUtc > files[i + 1].LastWriteTimeUtc;
                                break;
                        }
                        if ( swap ) {
                            FileInfo di = files[i];
                            files[i] = files[i + 1];
                            files[i + 1] = di;
                            changed = true;
                        }
                    }
                }

                /*foreach ( string key in extensions.Keys ) {
                    listFiles.SmallImageList.Images.Add( key, Icon.ExtractAssociatedIcon( extensions[key] ) );
                }*/
                for ( int i = 0; i < files.Count; i++ ) {
                    string ext = Path.GetExtension( files[i].FullName ).ToLower();
                    listFiles.SmallImageList.Images.Add( files[i].FullName, Icon.ExtractAssociatedIcon( files[i].FullName ) );
                    //if ( extensions.ContainsKey( ext ) ) {
                        listFiles.Items.Add( files[i].Name, files[i].FullName );
                    //} else {
                        //listFiles.Items.Add( files[i].Name );
                    //}
                }
            }
        }

        public bool Multiselect {
            get {
                return m_multiselect;
            }
            set {
                m_multiselect = value;
                listFiles.MultiSelect = m_multiselect;
            }
        }

        public string InitialDirectory {
            get {
                if ( m_initial_directory == "" ) {
                    m_initial_directory = Environment.GetFolderPath( Environment.SpecialFolder.MyDocuments );
                    m_current_directory = m_initial_directory;
                }
                return m_initial_directory;
            }
            set {
                m_initial_directory = value;
                m_current_directory = m_initial_directory;
            }
        }

        private void FileDialog_Load( object sender, EventArgs e ) {
            UpdateFileList();
        }

        private void btnUp_Click( object sender, EventArgs e ) {
            m_history.Add( m_current_directory );
            if ( m_current_directory == "MY_COMPUTER" ) {
                m_current_directory = Environment.GetFolderPath( Environment.SpecialFolder.Desktop );
            } else if ( Path.GetPathRoot( m_current_directory ).ToLower() == m_current_directory.ToLower() ) {
                m_current_directory = "MY_COMPUTER";
            } else {
                m_current_directory = Path.GetDirectoryName( m_current_directory );
            }
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void UpdateButtonStatus() {
            btnUp.Enabled = m_current_directory != Environment.GetFolderPath( Environment.SpecialFolder.Desktop );
            btnPrev.Enabled = m_history.Count > 0;
            chkDesktop.Checked = false;
            chkMyComputer.Checked = false;
            //radioMyNetwork.Checked = false;
            chkPersonal.Checked = false;
            //radioRecent.Checked = false;
            btnNew.Enabled = (m_current_directory != "MY_COMPUTER") && (m_current_directory != "MY_NETWORK");
            listFiles.MultiSelect = (m_multiselect && m_current_directory != "MY_COMPUTER" && m_current_directory != "MY_NETWORK");
            if ( m_current_directory == Environment.GetFolderPath( Environment.SpecialFolder.Desktop ) ) {
                chkDesktop.Checked = true;
                return;
            }
            if ( m_current_directory == "MY_COMPUTER" ) {
                chkMyComputer.Checked = true;
                return;
            }
            if ( m_current_directory == Environment.GetFolderPath( Environment.SpecialFolder.MyDocuments ) ) {
                chkPersonal.Checked = true;
                return;
            }
            /*if ( m_current_directory == Environment.GetFolderPath( Environment.SpecialFolder.Recent ) ) {
                radioRecent.Checked = true;
                return;
            }
            if ( m_current_directory == "MY_NETWORK" ) {
                radioMyNetwork.Checked = true;
                return;
            }*/
        }

        private void btnPrev_Click( object sender, EventArgs e ) {
            if ( m_history.Count > 0 ) {
                m_current_directory = m_history[m_history.Count - 1];
                m_history.RemoveAt( m_history.Count - 1 );
            }
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void radioRecent_Click( object sender, EventArgs e ) {
            m_history.Add( m_current_directory );
            m_current_directory = Environment.GetFolderPath( Environment.SpecialFolder.Recent );
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void radioDesktop_Click( object sender, EventArgs e ) {
            m_history.Add( m_current_directory );
            m_current_directory = Environment.GetFolderPath( Environment.SpecialFolder.Desktop );
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void radioPersonal_Click( object sender, EventArgs e ) {
            m_history.Add( m_current_directory );
            m_current_directory = Environment.GetFolderPath( Environment.SpecialFolder.MyDocuments );
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void radioMyComputer_Click( object sender, EventArgs e ) {
            m_history.Add( m_current_directory );
            m_current_directory = "MY_COMPUTER";
            UpdateFileList();
            UpdateButtonStatus();
        }

        private void listFiles_PreviewKeyDown( object sender, PreviewKeyDownEventArgs e ) {
            if ( listFiles.SelectedItems.Count != 1 ) {
                return;
            }
            if ( e.KeyCode == Keys.Enter ) {
                ProcessListViewItemClicked( listFiles.SelectedItems[0] );
            }
        }

        private void listFiles_MouseDoubleClick( object sender, MouseEventArgs e ) {
            ListViewItem item = listFiles.GetItemAt( e.X, e.Y );
            if ( item != null ) {
                ProcessListViewItemClicked( item );
            }
        }

        private void ProcessListViewItemClicked( ListViewItem listviewitem ) {
            string item = listviewitem.Text;
            if ( m_current_directory == "MY_COMPUTER" ) {
                bool available = true;
                try {
                    string[] dirs = Directory.GetDirectories( item );
                } catch {
                    available = false;
                }
                if ( !available ) {
                    return;
                }
                m_history.Add( m_current_directory );
                m_current_directory = item;
            } else if ( m_current_directory == "MY_NETWORK" ) {
                //todo:
            } else {
                string next = Path.Combine( m_current_directory, item );
                if ( Directory.Exists( next ) ) {
                    m_history.Add( m_current_directory );
                    m_current_directory = next;
                } else if ( File.Exists( next ) ) {
                    DialogResult = DialogResult.OK;
                    this.Close();
                }
            }
            UpdateFileList();
            UpdateButtonStatus();
        }

        public string[] FileNames {
            get {
                string[] ret = new string[m_selected_item.Count];
                for ( int i = 0; i < m_selected_item.Count; i++ ) {
                    ret[i] = Path.Combine( m_current_directory, m_selected_item[i] );
                }
                return ret;
            }
        }

        public string FileName {
            get {
                if ( m_selected_item.Count == 0 ) {
                    string ret = m_current_directory;
                    if ( m_current_directory == "MY_COMPUTER" || m_current_directory == "MY_NETWORK" ) {
                        ret = "";
                    }
                    return ret;
                } else {
                    return Path.Combine( m_current_directory, m_selected_item[0] );
                }
            }
            set {
                if ( value != "" ) {
                    m_initial_directory = Path.GetDirectoryName( value );
                    m_current_directory = m_initial_directory;
                    comboFileName.Text = Path.GetFileName( value );
                }
            }
        }

        private void listFiles_ItemSelectionChanged( object sender, ListViewItemSelectionChangedEventArgs e ) {
            StringBuilder sb = new StringBuilder();
            int count = listFiles.SelectedItems.Count;
            if ( count <= 0 ) {
                m_selected_item.Clear();
                return;
            } else if ( count == 1 ) {
                ListViewItem lvi = listFiles.SelectedItems[0];
                if ( lvi.ImageKey != "DRIVE" && lvi.ImageKey != "FOLDER" ) {
                    m_selected_item.Clear();
                    m_selected_item.Add( lvi.Text );
                    comboFileName.Text = lvi.Text;
                }
            } else {
                m_selected_item.Clear();
                bool first = true;
                foreach ( ListViewItem lvi in listFiles.SelectedItems ) {
                    if ( lvi.ImageKey != "DRIVE" && lvi.ImageKey != "FOLDER" ) {
                        m_selected_item.Add( lvi.Text );
                        sb.Append( (first ? "" : " ") + "\"" + lvi.Text + "\"" );
                        first = false;
                    }
                }
                comboFileName.Text = sb.ToString();
                sb = null;
            }
        }

        private void comboFileType_SelectedIndexChanged( object sender, EventArgs e ) {
            UpdateFileList();
        }
    }

}
