﻿/*
 * VocaloSingerConfigSys.s
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System.IO;
using System.Collections.Generic;

namespace Boare.Lib.Vsq {

    public class VocaloSingerConfigSys {
        /// <summary>
        /// Maximum number of singers which can be allowed in VOCALOID/VOCALOID2 system.
        /// </summary>
        public const uint MAX_SINGERS = 0x4000;

        private List<SingerConfig> m_singer_configs = new List<SingerConfig>();
        private List<SingerConfig> m_installed_singers = new List<SingerConfig>();

        /// <summary>
        /// Loads the list of singer configuration from the specified path.
        /// Keys in the dictionary "s_singer_configs" represents the program change of each singer.
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        public VocaloSingerConfigSys( string path, string[] voiceidstrs ) {
#if DEBUG
            bocoree.debug.push_log( "VocaloSingerConfigSys..ctor" );
#endif
            string map_file = Path.Combine( path, "voice.map" );
            if ( !File.Exists( map_file ) ) {
                return;
            }
            using ( FileStream fs = new FileStream( map_file, FileMode.Open, FileAccess.Read ) ) {
                byte[] dat = new byte[8];
                fs.Seek( 0x20, SeekOrigin.Begin );
                for ( int i = 0; i < MAX_SINGERS; i++ ) {
                    fs.Read( dat, 0, 8 );
                    ulong value = makelong_le( dat );
                    if ( value >= 1 ) {
                        string file = Path.Combine( path, "vvoice" + value + ".vvd" );
                        if ( File.Exists( file ) ) {
                            SingerConfig sc = SingerConfig.readSingerConfig( file, (int)(value - 1) );
                            sc.Program = i;
#if DEBUG
                            bocoree.debug.push_log( "    i=" + i + "; sc=" + sc.ToString() );
#endif
                            m_singer_configs.Add( sc );
                        }
                    }
                }
            }
            /*List<string> voiceidstrs = new List<string>();
            foreach ( SingerConfig sc in m_singer_configs ) {
                if ( !voiceidstrs.Contains( sc.VOICEIDSTR ) ) {
                    voiceidstrs.Add( sc.VOICEIDSTR );
                }
            }*/
            foreach ( string dir in voiceidstrs ) {
                //string dir = Path.Combine( path, s );
                string[] files = Directory.GetFiles( dir, "*.vvd" );
                foreach ( string s2 in files ) {
                    string file = Path.Combine( dir, s2 );
                    if ( File.Exists( file ) ) {
                        m_installed_singers.Add( SingerConfig.readSingerConfig( file, -1 ) );
                    }
                }
            }
#if DEBUG
            bocoree.debug.push_log( "    m_singer_configs" );
            foreach ( SingerConfig sc in m_singer_configs ) {
                bocoree.debug.push_log( "        VOICENAME=" + sc.VOICENAME );
            }
            bocoree.debug.push_log( "    m_installed_singers" );
            foreach ( SingerConfig sc in m_installed_singers ) {
                bocoree.debug.push_log( "        VOICENAME=" + sc.VOICENAME );
            }
#endif
        }

        /// <summary>
        /// Gets the list of installed singers which is installed in the computer.
        /// </summary>
        /// <returns></returns>
        public SingerConfig[] getInstalledSingers() {
            return m_installed_singers.ToArray();
        }

        /// <summary>
        /// Gets the name of original singer of specified program change.
        /// </summary>
        /// <param name="singer"></param>
        /// <returns></returns>
        public string getOriginalSinger( string singer ) {
            SingerConfig sc = getSingerInfo( singer );
            if ( sc != null ) {
                string voiceidstr = sc.VOICEIDSTR;
                foreach ( SingerConfig installed in m_installed_singers ) {
                    if ( installed.VOICEIDSTR == voiceidstr ) {
                        return installed.VOICENAME;
                    }
                }
            }
            return "";
        }

        /// <summary>
        /// Gets the singer information of pecified program change.
        /// </summary>
        /// <param name="singer"></param>
        /// <returns></returns>
        public SingerConfig getSingerInfo( string singer ) {
            for ( int i = 0; i < m_singer_configs.Count; i++ ) {
                if ( m_singer_configs[i].VOICENAME == singer ) {
                    return m_singer_configs[i];
                }
            }
            return null;
        }

        /// <summary>
        /// Gets the list of singer configs.
        /// </summary>
        /// <returns></returns>
        public SingerConfig[] getSingerConfigs() {
            SingerConfig[] ret = new SingerConfig[m_singer_configs.Count];
            for ( int i = 0; i < m_singer_configs.Count; i++ ) {
                ret[i] = (SingerConfig)m_singer_configs[i].Clone();
            }
            return ret;
        }

        /// <summary>
        /// Gets the VsqID of program change.
        /// </summary>
        /// <param name="singer"></param>
        /// <returns></returns>
        public VsqID getSingerID( string singer ) {
            VsqID ret = new VsqID( 0 );
            ret.type = VsqIDType.Singer;
            SingerConfig sc = getSingerInfo( singer );
            if ( sc == null ) {
                sc = new SingerConfig();
            }
            int lang = 0;
            foreach ( SingerConfig sc2 in m_installed_singers ) {
                if ( sc.VOICEIDSTR == sc2.VOICEIDSTR ) {
                    lang = (int)getLanguageFromName( sc.VOICENAME );
                    break;
                }
            }
            ret.IconHandle = new IconHandle();
            ret.IconHandle.IconID = "$0701" + sc.Program.ToString( "0000" );
            ret.IconHandle.IDS = sc.VOICENAME;
            ret.IconHandle.Index = 0;
            ret.IconHandle.Language = lang;
            ret.IconHandle.Length = 1;
            ret.IconHandle.Original = sc.Original;
            ret.IconHandle.Program = sc.Program;
            ret.IconHandle.Caption = "";
            return ret;
        }

        /// <summary>
        /// Gets the voice language of specified program change
        /// </summary>
        /// <param name="name">name of singer</param>
        /// <returns></returns>
        private static VsqVoiceLanguage getLanguageFromName( string name ) {
            switch ( name ) {
                case "MEIKO":
                case "Miku":
                case "Rin":
                case "Len":
                case "Rin_ACT2":
                case "Len_ACT2":
                case "Gackpoid":
                case "Luka_JPN":
                case "Megpoid":
                    return VsqVoiceLanguage.Japanese;
                case "Sweet_Ann":
                case "Prima":
                case "Luka_ENG":
                    return VsqVoiceLanguage.English;
            }
            return VsqVoiceLanguage.Default;
        }

        /// <summary>
        /// Gets the voice language of specified program change
        /// </summary>
        /// <param name="singer">name of singer to get the voice language</param>
        /// <returns></returns>
        public VsqVoiceLanguage getLanguage( string singer ) {
            string name = getOriginalSinger( singer );
            return getLanguageFromName( name );
        }

        /// <summary>
        /// Transform the byte array(length=8) to unsigned long, assuming that the byte array is little endian.
        /// </summary>
        /// <param name="oct"></param>
        /// <returns></returns>
        private static ulong makelong_le( byte[] oct ) {
            return (ulong)oct[7] << 56 | (ulong)oct[6] << 48 | (ulong)oct[5] << 40 | (ulong)oct[4] << 32 | (ulong)oct[3] << 24 | (ulong)oct[2] << 16 | (ulong)oct[1] << 8 | (ulong)oct[0];
        }
    }

}
