﻿/*
 * EditorConfig.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Xml.Serialization;
using System.IO;
using System.Windows.Forms;
using System.Drawing;

using bocoree;
using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    /// <summary>
    /// Cadenciiの環境設定
    /// </summary>
    public class EditorConfig {
        [XmlItemDescription( "Default value of pre-measure" )]
        public int DefaultPreMeasure = 4;
        //[Obsolete][XmlItemDescription( "Program change of default singer")]
        //public int DefaultSinger = 0;
        public string DefaultSingerName = "Miku";
        [XmlItemDescription( "Default scale for width direction in pixel/clock" )]
        public int DefaultXScale = 65;
        public string BaseFontName = "MS UI Gothic";
        [XmlItemDescription( "Font size in GraphicsUnit.Point unit" )]
        public float BaseFontSize = 9.0f;
        public string ScreenFontName = "MS UI Gothic";
        public string CounterFontName = "Arial";
        [XmlItemDescription( "Mouse wheel speed" )]
        public int WheelOrder = 20;
        [XmlItemDescription( "Fix play cursor to the center of screen" )]
        public bool CursorFixed = false;
        /// <summary>
        /// RecentFilesに登録することの出来る最大のファイル数
        /// </summary>
        [XmlItemDescription( "Capacity of RecentFiles" )]
        public int NumRecentFiles = 5;
        /// <summary>
        /// 最近使用したファイルのリスト
        /// </summary>
        [XmlItemDescription( "List of files recentry used" )]
        public List<string> RecentFiles = new List<string>();
        public int DefaultPMBendDepth = 8;
        public int DefaultPMBendLength = 0;
        public int DefaultPMbPortamentoUse = 3;
        public int DefaultDEMdecGainRate = 50;
        public int DefaultDEMaccent = 50;
        [XmlItemDescription( "Drawing lyrics to screen or not" )]
        public bool ShowLyric = true;
        public bool ShowExpLine = true;
        public DefaultVibratoLength DefaultVibratoLength = DefaultVibratoLength.L66;
        public AutoVibratoMinLength AutoVibratoMinimumLength = AutoVibratoMinLength.L1;
        public VibratoType AutoVibratoType = VibratoType.NormalType1;
        public bool EnableAutoVibrato = true;
        public int PxTrackHeight = 14;
        public int MouseDragIncrement = 50;
        public int MouseDragMaximumRate = 600;
        public bool MixerVisible = false;
        public int PreSendTime = 500;
        public ClockResolution ControlCurveResolution = ClockResolution.L30;
        public string Language = "";
        /// <summary>
        /// マウスの操作などの許容範囲。プリメジャーにPxToleranceピクセルめり込んだ入力を行っても、エラーにならない。(補正はされる)
        /// </summary>
        [XmlItemDescription( "Tolerance of mouse operation (in pixel)" )]
        public int PxTolerance = 10;
        /// <summary>
        /// マウスホイールでピアノロールを水平方向にスクロールするかどうか。
        /// </summary>
        [XmlItemDescription( "Whether scroll horizontally with mouse-wheel" )]
        public bool ScrollHorizontalOnWheel = true;
        /// <summary>
        /// 画面描画の最大フレームレート
        /// </summary>
        [XmlItemDescription( "Maximum frame rate(in frames per second)" )]
        public int MaximumFrameRate = 15;
        /// <summary>
        /// ユーザー辞書のOn/Offと順序
        /// </summary>
        public List<string> UserDictionaries = new List<string>();
        /// <summary>
        /// 実行環境
        /// </summary>
        public Platform Platform = Platform.Windows;
        /// <summary>
        /// toolStripToolの表示位置
        /// </summary>
        public ToolStripLocation ToolEditTool = new ToolStripLocation( new Point( 3, 50 ), ToolStripLocation.ParentPanel.Top );
        /// <summary>
        /// toolStripPositionの表示位置
        /// </summary>
        public ToolStripLocation ToolPositionLocation = new ToolStripLocation( new Point( 3, 0 ), ToolStripLocation.ParentPanel.Top );
        /// <summary>
        /// toolStripMeasureの表示位置
        /// </summary>
        public ToolStripLocation ToolMeasureLocation = new ToolStripLocation( new Point( 3, 25 ), ToolStripLocation.ParentPanel.Top );
        public ToolStripLocation ToolFileLocation = new ToolStripLocation( new Point( 3, 75 ), ToolStripLocation.ParentPanel.Top );
        public ToolStripLocation ToolPaletteLocation = new ToolStripLocation( new Point( 3, 100 ), ToolStripLocation.ParentPanel.Top );
        public bool WindowMaximized = false;
        public Rectangle WindowRect = new Rectangle( 0, 0, 756, 616 );
        /// <summary>
        /// hScrollのスクロールボックスの最小幅(px)
        /// </summary>
        public int MinimumScrollHandleWidth = 20;
        /// <summary>
        /// 発音記号入力モードを，維持するかどうか
        /// </summary>
        public bool KeepLyricInputMode = false;
        /// <summary>
        /// 最後に使用したVSQファイルへのパス
        /// </summary>
        public string LastVsqPath = "";
        /// <summary>
        /// ピアノロールの何もないところをクリックした場合、右クリックでもプレビュー音を再生するかどうか
        /// </summary>
        public bool PlayPreviewWhenRightClick = false;
        /// <summary>
        /// ゲームコントローラで、異なるイベントと識別する最小の時間間隔(millisec)
        /// </summary>
        public int GameControlerMinimumEventInterval = 100;
        /// <summary>
        /// カーブの選択範囲もクオンタイズするかどうか
        /// </summary>
        public bool CurveSelectingQuantized = true;

        private QuantizeMode m_position_quantize = QuantizeMode.p32;
        private bool m_position_quantize_triplet = false;
        private QuantizeMode m_length_quantize = QuantizeMode.p32;
        private bool m_length_quantize_triplet = false;
        private int m_mouse_hover_time = 500;
        private int m_gamectrl_tr = 0;
        private int m_gamectrl_o = 1;
        private int m_gamectrl_x = 2;
        private int m_gamectrl_re = 3;
        private int m_gamectrl_L1 = 4;
        private int m_gamectrl_R1 = 5;
        private int m_gamectrl_L2 = 6;
        private int m_gamectrl_R2 = 7;
        private int m_gamectrl_select = 8;
        private int m_gamectrl_start = 9;
        private int m_gamectrl_stickL = 10;
        private int m_gamectrl_stickR = 11;
        public bool CurveVisibleVelocity = true;
        public bool CurveVisibleAccent = true;
        public bool CurveVisibleDecay = true;
        public bool CurveVisibleVibratoRate = true;
        public bool CurveVisibleVibratoDepth = true;
        public bool CurveVisibleDynamics = true;
        public bool CurveVisibleBreathiness = true;
        public bool CurveVisibleBrightness = true;
        public bool CurveVisibleClearness = true;
        public bool CurveVisibleOpening = true;
        public bool CurveVisibleGendorfactor = true;
        public bool CurveVisiblePortamento = true;
        //public bool CurveVisiblePitchbend = true;
        //public bool CurveVisiblePitchbendSensitivity = true;
        public bool CurveVisiblePitch = true;
        public bool CurveVisibleHarmonics = false;
        public bool CurveVisibleFx2Depth = false;
        public bool CurveVisibleReso1 = false;
        public bool CurveVisibleReso2 = false;
        public bool CurveVisibleReso3 = false;
        public bool CurveVisibleReso4 = false;
        public int GameControlPovRight = 9000;
        public int GameControlPovLeft = 27000;
        public int GameControlPovUp = 0;
        public int GameControlPovDown = 18000;
        /// <summary>
        /// wave波形を表示するかどうか
        /// </summary>
        public bool ViewWaveform = false;
        /// <summary>
        /// キーボードからの入力に使用するデバイス
        /// </summary>
        public int MidiInPort = -1;
        public RgbColor PianorollColorVocalo2Black = new RgbColor( 212, 212, 212 );
        public RgbColor PianorollColorVocalo2White = new RgbColor( 240, 240, 240 );
        public RgbColor PianorollColorVocalo1Black = new RgbColor( 210, 205, 172 );
        public RgbColor PianorollColorVocalo1White = new RgbColor( 240, 235, 214 );
        public RgbColor PianorollColorUtauBlack = new RgbColor( 212, 212, 212 );
        public RgbColor PianorollColorUtauWhite = new RgbColor( 240, 240, 240 );
        public RgbColor PianorollColorVocalo1Bar = new RgbColor( 161, 157, 136 );
        public RgbColor PianorollColorVocalo1Beat = new RgbColor( 209, 204, 172 );
        public RgbColor PianorollColorVocalo2Bar = new RgbColor( 161, 157, 136 );
        public RgbColor PianorollColorVocalo2Beat = new RgbColor( 209, 204, 172 );
        public RgbColor PianorollColorUtauBar = new RgbColor( 255, 64, 255 );
        public RgbColor PianorollColorUtauBeat = new RgbColor( 128, 128, 255 );
        [XmlItemDescription( "Show actual pitch line or not" )]
        public bool ViewAtcualPitch = false;
        [XmlItemDescription( "Invoke resampler with Wine" )]
        public bool InvokeUtauCoreWithWine = false;
        public List<SingerConfig> UtauSingers = new List<SingerConfig>();
        public string PathResampler = "";
        public string PathWavtool = "";
        public bool UseCustomFileDialog = false;
        /// <summary>
        /// ベジエ制御点を掴む時の，掴んだと判定する際の誤差．制御点の外輪からPxToleranceBezierピクセルずれていても，掴んだと判定する．
        /// </summary>
        public int PxToleranceBezier = 10;
        /// <summary>
        /// 歌詞入力においてローマ字が入力されたとき，Cadencii側でひらがなに変換するかどうか
        /// </summary>
        public bool SelfDeRomanization = false;
        /// <summary>
        /// openMidiDialogで最後に選択された拡張子
        /// </summary>
        public string LastUsedExtension = ".vsq";
        /// <summary>
        /// ミキサーダイアログを常に手前に表示するかどうか
        /// </summary>
        public bool MixerTopMost = false;


        /// <summary>
        /// PositionQuantize, PositionQuantizeTriplet, LengthQuantize, LengthQuantizeTripletの描くプロパティのいずれかが
        /// 変更された時発生します
        /// </summary>
        public static event EventHandler QuantizeModeChanged;

        public Font BaseFont {
            get {
                return new Font( BaseFontName, BaseFontSize, GraphicsUnit.Point );
            }
        }

        /// <summary>
        /// Button index of Left Stick
        /// </summary>
        public int GameControlStirckL{
            get {
                return m_gamectrl_stickL;
            }
            set {
                m_gamectrl_stickL = value;
            }
        }

        /// <summary>
        /// Button index of Right Stick
        /// </summary>
        public int GameControlStirckR {
            get {
                return m_gamectrl_stickR;
            }
            set {
                m_gamectrl_stickR = value;
            }
        }

        /// <summary>
        /// Button index of "START"
        /// </summary>
        public int GameControlStart {
            get {
                return m_gamectrl_start;
            }
            set {
                m_gamectrl_start = value;
            }
        }

        /// <summary>
        /// Button index of "R2"
        /// </summary>
        public int GameControlR2 {
            get {
                return m_gamectrl_R2;
            }
            set {
                m_gamectrl_R2 = value;
            }
        }

        /// <summary>
        /// Button index of "R1"
        /// </summary>
        public int GameControlR1 {
            get {
                return m_gamectrl_R1;
            }
            set {
                m_gamectrl_R1 = value;
            }
        }

        /// <summary>
        /// Button index of "L2"
        /// </summary>
        public int GameControlL2 {
            get {
                return m_gamectrl_L2;
            }
            set {
                m_gamectrl_L2 = value;
            }
        }

        /// <summary>
        /// Button index of "L1"
        /// </summary>
        public int GameControlL1 {
            get {
                return m_gamectrl_L1;
            }
            set {
                m_gamectrl_L1 = value;
            }
        }

        /// <summary>
        /// Button index of "SELECT"
        /// </summary>
        public int GameControlSelect {
            get {
                return m_gamectrl_select;
            }
            set {
                m_gamectrl_select = value;
            }
        }

        /// <summary>
        /// Button index of "□"
        /// </summary>
        public int GameControlerRectangle {
            get {
                return m_gamectrl_re;
            }
            set {
                m_gamectrl_re = value;
            }
        }

        /// <summary>
        /// Button index of "×"
        /// </summary>
        public int GameControlerCross {
            get {
                return m_gamectrl_x;
            }
            set {
                m_gamectrl_x = value;
            }
        }

        /// <summary>
        /// Button index of "○"
        /// </summary>
        public int GameControlerCircle {
            get {
                return m_gamectrl_o;
            }
            set {
                m_gamectrl_o = value;
            }
        }

        /// <summary>
        /// Button index of "△"
        /// </summary>
        public int GameControlerTriangle {
            get {
                return m_gamectrl_tr;
            }
            set {
                m_gamectrl_tr = value;
            }
        }

        /// <summary>
        /// ピアノロール上でマウスホバーイベントが発生するまでの時間(millisec)
        /// </summary>
        public int MouseHoverTime {
            get {
                return m_mouse_hover_time;
            }
            set {
                if ( value < 0 ) {
                    m_mouse_hover_time = 0;
                } else if ( 2000 < m_mouse_hover_time ) {
                    m_mouse_hover_time = 2000;
                } else {
                    m_mouse_hover_time = value;
                }
            }
        }

        public QuantizeMode PositionQuantize {
            get {
                return m_position_quantize;
            }
            set {
                if ( m_position_quantize != value ) {
                    m_position_quantize = value;
                    if ( QuantizeModeChanged != null ) {
                        QuantizeModeChanged( typeof( EditorConfig ), new EventArgs() );
                    }
                }
            }
        }

        public bool PositionQuantizeTriplet {
            get {
                return m_position_quantize_triplet;
            }
            set {
                if ( m_position_quantize_triplet != value ) {
                    m_position_quantize_triplet = value;
                    if ( QuantizeModeChanged != null ) {
                        QuantizeModeChanged( typeof( EditorConfig ), new EventArgs() );
                    }
                }
            }
        }

        public QuantizeMode LengthQuantize {
            get {
                return m_length_quantize;
            }
            set {
                if ( m_length_quantize != value ) {
                    m_length_quantize = value;
                    if ( QuantizeModeChanged != null ) {
                        QuantizeModeChanged( typeof( EditorConfig ), new EventArgs() );
                    }
                }
            }
        }

        public bool LengthQuantizeTriplet {
            get {
                return m_length_quantize_triplet;
            }
            set {
                if ( m_length_quantize_triplet != value ) {
                    m_length_quantize_triplet = value;
                    if ( QuantizeModeChanged != null ) {
                        QuantizeModeChanged( typeof( EditorConfig ), new EventArgs() );
                    }
                }
            }
        }

        /// <summary>
        /// 「最近使用したファイル」のリストに、アイテムを追加します
        /// </summary>
        /// <param name="new_file"></param>
        public void PushRecentFiles( string new_file ) {
#if DEBUG
            System.Diagnostics.Debug.WriteLine( "PushRecentFiles" );
#endif
            // NumRecentFilesは0以下かも知れない
            if ( NumRecentFiles <= 0 ) {
                NumRecentFiles = 5;
            }

            // RecentFilesはnullかもしれない．
            if ( RecentFiles == null ) {
                RecentFiles = new List<string>();
            }

            // 重複があれば消す
            List<string> dict = new List<string>();
            foreach ( string s in RecentFiles ) {
                bool found = false;
                for ( int i = 0; i < dict.Count; i++ ) {
                    if ( s == dict[i] ) {
                        found = true;
                    }
                }
                if ( !found ) {
                    dict.Add( s );
                }
            }
            RecentFiles.Clear();
            foreach ( string s in dict ) {
                RecentFiles.Add( s );
            }

            // 現在登録されているRecentFilesのサイズが規定より大きければ，下の方から消す
            if ( RecentFiles.Count > NumRecentFiles ) {
                for ( int i = RecentFiles.Count - 1; i > NumRecentFiles; i-- ) {
                    RecentFiles.RemoveAt( i );
                }
            }

            // 登録しようとしているファイルは，RecentFilesの中に既に登録されているかs？
            int index = -1;
            for ( int i = 0; i < RecentFiles.Count; i++ ) {
                if ( RecentFiles[i] == new_file ) {
                    index = i;
                    break;
                }
            }

            if ( index >= 0 ) {  // 登録されてる場合
                RecentFiles.RemoveAt( index );
            }
            RecentFiles.Insert( 0, new_file );
        }
    }

}
