﻿/*
 * AppManager.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Threading;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.CodeDom.Compiler;
using Microsoft.CSharp;
using System.Text;

using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    public delegate void BSimpleDelegate<T>( T arg );

    public static class AppManager {
        /// <summary>
        /// 鍵盤の表示幅(pixel)
        /// </summary>
        public const int _KEY_LENGTH = 68;
        private const string _CONFIG_FILE_NAME = "config.xml";
        private const string _CONFIG_DIR_NAME = "Cadencii";
        /// <summary>
        /// OSのクリップボードに貼り付ける文字列の接頭辞．これがついていた場合，クリップボードの文字列をCadenciiが使用できると判断する．
        /// </summary>
        private const string _CLIP_PREFIX = "CADENCIIOBJ";

        /// <summary>
        /// エディタの設定
        /// </summary>
        public static EditorConfig EditorConfig = new EditorConfig();
        /// <summary>
        /// AttachedCurve用のシリアライザ
        /// </summary>
        public static XmlSerializer XmlSerializerListBezierCurves = new XmlSerializer( typeof( AttachedCurve ) );
        public static Font BaseFont8 = null;
        public static Font BaseFont9 = null;
        /// <summary>
        /// ピアノロールの歌詞の描画に使用されるフォント。
        /// </summary>
        public static Font BaseFont10 = null;
        /// <summary>
        /// ピアノロールの歌詞の描画に使用されるフォント。（発音記号固定の物の場合）
        /// </summary>
        public static Font BaseFont10Bold = null;
        /// <summary>
        /// 歌詞を音符の（高さ方向の）真ん中に描画するためのオフセット
        /// </summary>
        public static int BaseFont10OffsetHeight = 0;
        public static int BaseFont8OffsetHeight = 0;
        public static int BaseFont9OffsetHeight = 0;

        #region Static Readonly Fields
        public static readonly Color[] s_HILIGHT = new Color[] { 
            Color.FromArgb( 181, 220, 16 ),
            Color.FromArgb( 231, 244, 49 ),
            Color.FromArgb( 252, 230, 29 ),
            Color.FromArgb( 247, 171, 20 ),
            Color.FromArgb( 249, 94, 17 ),
            Color.FromArgb( 234, 27, 47 ),
            Color.FromArgb( 175, 20, 80 ),
            Color.FromArgb( 183, 24, 149 ),
            Color.FromArgb( 105, 22, 158 ),
            Color.FromArgb( 22, 36, 163 ),
            Color.FromArgb( 37, 121, 204 ),
            Color.FromArgb( 29, 179, 219 ),
            Color.FromArgb( 24, 239, 239 ),
            Color.FromArgb( 25, 206, 175 ),
            Color.FromArgb( 23, 160, 134 ),
            Color.FromArgb( 79, 181, 21 ) };
        public static readonly Color[] s_RENDER = new Color[]{
            Color.FromArgb( 19, 143, 52 ),
            Color.FromArgb( 158, 154, 18 ),
            Color.FromArgb( 160, 143, 23 ),
            Color.FromArgb( 145, 98, 15 ),
            Color.FromArgb( 142, 52, 12 ),
            Color.FromArgb( 142, 19, 37 ),
            Color.FromArgb( 96, 13, 47 ),
            Color.FromArgb( 117, 17, 98 ),
            Color.FromArgb( 62, 15, 99 ),
            Color.FromArgb( 13, 23, 84 ),
            Color.FromArgb( 25, 84, 132 ),
            Color.FromArgb( 20, 119, 142 ),
            Color.FromArgb( 19, 142, 139 ),
            Color.FromArgb( 17, 122, 102 ),
            Color.FromArgb( 13, 86, 72 ),
            Color.FromArgb( 43, 91, 12 ) };
        public static readonly string[] _USINGS = new string[] { "using System;",
                                             "using System.IO;",
                                             "using Boare.Lib.Vsq;",
                                             "using Boare.Cadencii;",
                                             "using bocoree;",
                                             "using Boare.Lib.Media;",
                                             "using Boare.Lib.AppUtil;",
                                             "using System.Windows.Forms;",
                                             "using System.Collections.Generic;",
                                             "using System.Drawing;",
                                             "using System.Text;",
                                             "using System.Xml.Serialization;" };
        #endregion

        #region Private Static Fields
        private static int s_base_tempo = 480000;
        private static SolidBrush s_hilight_brush = new SolidBrush( Color.CornflowerBlue );
        private static object s_locker;
        private static XmlSerializer s_serizlizer = null;
        #endregion

        public delegate void MainFormClosedEventHandler( EditorManager manager );
        public delegate void EditorConfigChangedEventHandler( EditorManager manager );

        public static void Init() {
            s_locker = new object();
            SymbolTable.loadDictionary();
            LoadConfig();
            VSTiProxy.CurrentUser = "";

            #region Apply User Dictionary Configuration
            List<ValuePair<string, bool>> current = new List<ValuePair<string, bool>>();
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                current.Add( new ValuePair<string, bool>( SymbolTable.getSymbolTable( i ).getName(), false ) );
            }
            List<ValuePair<string, bool>> config_data = new List<ValuePair<string, bool>>();
            for ( int i = 0; i < EditorConfig.UserDictionaries.Count; i++ ) {
                string[] spl = EditorConfig.UserDictionaries[i].Split( "\t".ToCharArray(), 2 );
                config_data.Add( new ValuePair<string, bool>( spl[0], (spl[1] == "T" ? true : false) ) );
#if DEBUG
                    Common.DebugWriteLine( "    " + spl[0] + "," + spl[1] );
#endif
            }
            List<KeyValuePair<string, bool>> common = new List<KeyValuePair<string, bool>>();
            for ( int i = 0; i < config_data.Count; i++ ) {
                for ( int j = 0; j < current.Count; j++ ) {
                    if ( config_data[i].Key == current[j].Key ) {
                        current[j].Value = true; //こっちのboolは、AppManager.EditorConfigのUserDictionariesにもKeyが含まれているかどうかを表すので注意
                        common.Add( new KeyValuePair<string, bool>( config_data[i].Key, config_data[i].Value ) );
                        break;
                    }
                }
            }
            for ( int i = 0; i < current.Count; i++ ) {
                if ( !current[i].Value ) {
                    common.Add( new KeyValuePair<string, bool>( current[i].Key, false ) );
                }
            }
            SymbolTable.changeOrder( common.ToArray() );
            #endregion

            Boare.Lib.AppUtil.Messaging.LoadMessages();
            Boare.Lib.AppUtil.Messaging.Language = EditorConfig.Language;

            KeySoundPlayer.Init();
            PaletteToolServer.Init();
        }

        /// <summary>
        /// オブジェクトをシリアライズし，クリップボードに格納するための文字列を作成します
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        private static string GetSerializedText( object obj ) {
            string str = "";
            using ( MemoryStream ms = new MemoryStream() ) {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize( ms, obj );
                ms.Seek( 0, SeekOrigin.Begin );
                byte[] arr = new byte[ms.Length];
                ms.Read( arr, 0, arr.Length );
                str = _CLIP_PREFIX + ":" + obj.GetType().FullName + ":" + Convert.ToBase64String( arr );
            }
            return str;
        }

        /// <summary>
        /// クリップボードに格納された文字列を元に，デシリアライズされたオブジェクトを取得します
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        private static object GetDeserializedObjectFromText( string s ) {
            if ( s.StartsWith( _CLIP_PREFIX ) ) {
                int index = s.IndexOf( ":" );
                index = s.IndexOf( ":", index + 1 );
                object ret = null;
                try {
                    using ( MemoryStream ms = new MemoryStream( Convert.FromBase64String( s.Substring( index + 1 ) ) ) ) {
                        BinaryFormatter bf = new BinaryFormatter();
                        ret = bf.Deserialize( ms );
                    }
                } catch {
                    ret = null;
                }
                return ret;
            } else {
                return null;
            }
        }

        public static void ClearClipBoard() {
            if ( Clipboard.ContainsText() ) {
                if ( Clipboard.GetText().StartsWith( _CLIP_PREFIX ) ) {
                    Clipboard.Clear();
                }
            }
        }

        public static void SetClipboard( ClipboardEntry item ) {
            string clip = GetSerializedText( item );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }

        #region VsqEvent用のクリップボード管理
        public static List<VsqEvent> GetCopiedEvent() {
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
#if DEBUG
                    Common.DebugWriteLine( "typename=" + typename );
#endif
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        List<VsqEvent> ret = null;
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            ret = ce.Event;
                        } catch {
                            ret = null;
                        }
                        if ( ret != null ) {
                            return ret;
                        }
                    }
                }
            }
            return new List<VsqEvent>();
        }

        public static void SetCopiedEvent( List<VsqEvent> item ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Event = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region TempoTableEntry用のクリップボード管理
        public static List<TempoTableEntry> GetCopiedTempo( out int copy_started_clock ) {
            List<TempoTableEntry> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Tempo;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new List<TempoTableEntry>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedTempo( List<TempoTableEntry> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Tempo = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region TimeSigTableEntry用のクリップボード管理
        public static List<TimeSigTableEntry> GetCopiedTimesig( out int copy_started_clock ) {
            List<TimeSigTableEntry> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Timesig;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new List<TimeSigTableEntry>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedTimesig( List<TimeSigTableEntry> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Timesig = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        #region CurveType, List<BPPair>用のクリップボード管理
        public static Dictionary<CurveType, List<BPPair>> GetCopiedCurve( out int copy_started_clock ) {
            Dictionary<CurveType, List<BPPair>> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Curve;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new Dictionary<CurveType, List<BPPair>>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedCurve( Dictionary<CurveType, List<BPPair>> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Curve = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        //Dictionary<CurveType, List<BezierChain>>
        #region CurveType, List<BezierChain>用のクリップボード管理
        public static Dictionary<CurveType, List<BezierChain>> GetCopiedBezier( out int copy_started_clock ) {
            Dictionary<CurveType, List<BezierChain>> tempo_table = null;
            copy_started_clock = 0;
            if ( Clipboard.ContainsText() ) {
                string clip = Clipboard.GetText();
                if ( clip.StartsWith( _CLIP_PREFIX ) ) {
                    int index1 = clip.IndexOf( ":" );
                    int index2 = clip.IndexOf( ":", index1 + 1 );
                    string typename = clip.Substring( index1 + 1, index2 - index1 - 1 );
                    if ( typename == typeof( ClipboardEntry ).FullName ) {
                        try {
                            ClipboardEntry ce = (ClipboardEntry)GetDeserializedObjectFromText( clip );
                            tempo_table = ce.Bezier;
                            copy_started_clock = ce.CopyStartedClock;
                        } catch {
                            tempo_table = null;
                        }
                    }
                }
            }
            if ( tempo_table == null ) {
                tempo_table = new Dictionary<CurveType, List<BezierChain>>();
                copy_started_clock = 0;
            }
            return tempo_table;
        }

        public static void SetCopiedBezier( Dictionary<CurveType, List<BezierChain>> item, int copy_started_clock ) {
            ClipboardEntry ce = new ClipboardEntry();
            ce.Bezier = item;
            string clip = GetSerializedText( ce );
            Clipboard.Clear();
            Clipboard.SetText( clip );
        }
        #endregion

        public static Assembly CompileScript( string code, out CompilerResults results ) {
            CSharpCodeProvider provider = new CSharpCodeProvider();
            string path = Application.StartupPath;
            CompilerParameters parameters = new CompilerParameters( new string[] {
                Path.Combine( path, "Boare.Lib.Vsq.dll" ),
                Path.Combine( path, "Cadencii.exe" ),
                Path.Combine( path, "Boare.Lib.Media.dll" ),
                Path.Combine( path, "Boare.Lib.AppUtil.dll" ),
                Path.Combine( path, "bocoree.dll" ) } );
            parameters.ReferencedAssemblies.Add( "System.Windows.Forms.dll" );
            parameters.ReferencedAssemblies.Add( "System.dll" );
            parameters.ReferencedAssemblies.Add( "System.Drawing.dll" );
            parameters.ReferencedAssemblies.Add( "System.Xml.dll" );
            parameters.GenerateInMemory = true;
            parameters.GenerateExecutable = false;
            parameters.IncludeDebugInformation = true;
            results = provider.CompileAssemblyFromSource( parameters, code );
            return results.CompiledAssembly;
        }

        /// <summary>
        /// アプリケーションデータの保存位置を取得します
        /// Gets the path for application data
        /// </summary>
        public static string ApplicationDataPath {
            get {
                string dir = Path.Combine( Environment.GetFolderPath( Environment.SpecialFolder.LocalApplicationData ), "Boare" );
                if ( !Directory.Exists( dir ) ) {
                    Directory.CreateDirectory( dir );
                }
                string dir2 = Path.Combine( dir, _CONFIG_DIR_NAME );
                if ( !Directory.Exists( dir2 ) ) {
                    Directory.CreateDirectory( dir2 );
                }
                return dir2;
            }
        }

        /// <summary>
        /// 位置クオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetPositionQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.PositionQuantize, EditorConfig.PositionQuantizeTriplet );
        }

        /// <summary>
        /// 音符長さクオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetLengthQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.LengthQuantize, EditorConfig.LengthQuantizeTriplet );
        }

        public static void SaveConfig() {
            // ユーザー辞書の情報を取り込む
            EditorConfig.UserDictionaries.Clear();
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                EditorConfig.UserDictionaries.Add( SymbolTable.getSymbolTable( i ).getName() + "\t" + (SymbolTable.getSymbolTable( i ).isEnabled() ? "T" : "F") );
            }

            if ( s_serizlizer == null ) {
                s_serizlizer = new XmlSerializer( typeof( EditorConfig ) );
            }
            string file = Path.Combine( ApplicationDataPath, _CONFIG_FILE_NAME );
            using ( FileStream fs = new FileStream( file, FileMode.Create ) ) {
                try {
                    s_serizlizer.Serialize( fs, EditorConfig );
                } catch {
                }
            }
        }

        public static void LoadConfig() {
            string config_file = Path.Combine( ApplicationDataPath, _CONFIG_FILE_NAME );
            EditorConfig ret = null;
            if ( File.Exists( config_file ) ) {
                FileStream fs = null;
                try {
                    XmlSerializer serizlizer = new XmlSerializer( typeof( EditorConfig ) );
                    fs = new FileStream( config_file, FileMode.Open );
                    ret = (EditorConfig)serizlizer.Deserialize( fs );
                } catch {
                } finally {
                    if ( fs != null ) {
                        fs.Close();
                    }
                }
            } else {
                config_file = Path.Combine( Application.StartupPath, _CONFIG_FILE_NAME );
                if ( File.Exists( config_file ) ) {
                    FileStream fs = null;
                    try {
                        XmlSerializer serizlizer = new XmlSerializer( typeof( EditorConfig ) );
                        fs = new FileStream( config_file, FileMode.Open );
                        ret = (EditorConfig)serizlizer.Deserialize( fs );
                    } catch {
                    } finally {
                        if ( fs != null ) {
                            fs.Close();
                        }
                    }
                }
            }
            if ( ret == null ) {
                ret = new EditorConfig();
            }
            EditorConfig = ret;
            for ( int i = 0; i < SymbolTable.getCount(); i++ ) {
                SymbolTable st = SymbolTable.getSymbolTable( i );
                bool found = false;
                foreach ( string s in EditorConfig.UserDictionaries ) {
                    string[] spl = s.Split( "\t".ToCharArray(), 2 );
                    if ( st.getName() == spl[0] ) {
                        found = true;
                        break;
                    }
                }
                if ( !found ) {
                    EditorConfig.UserDictionaries.Add( st.getName() + "\tT" );
                }
            }
        }

        public static VsqID getSingerIDUtau( string name ) {
            VsqID ret = new VsqID( 0 );
            ret.type = VsqIDType.Singer;
            int index = -1;
            for ( int i = 0; i < EditorConfig.UtauSingers.Count; i++ ) {
                if ( EditorConfig.UtauSingers[i].VOICENAME == name ) {
                    index = i;
                    break;
                }
            }
            if ( index >= 0 ) {
                SingerConfig sc = EditorConfig.UtauSingers[index];
                int lang = 0;//utauは今のところ全部日本語
                ret.IconHandle = new IconHandle();
                ret.IconHandle.IconID = "$0701" + index.ToString( "0000" );
                ret.IconHandle.IDS = sc.VOICENAME;
                ret.IconHandle.Index = 0;
                ret.IconHandle.Language = lang;
                ret.IconHandle.Length = 1;
                ret.IconHandle.Original = sc.Original;
                ret.IconHandle.Program = sc.Program;
                ret.IconHandle.Caption = "";
                return ret;
            } else {
                //VsqID ret = new VsqID( 0 );
                ret.IconHandle = new IconHandle();
                ret.IconHandle.Program = 0;
                ret.IconHandle.Language = 0;
                ret.IconHandle.IconID = "$0701" + 0.ToString( "0000" );
                ret.IconHandle.IDS = "Unknown";
                ret.type = VsqIDType.Singer;
                return ret;
            }
        }

        public static SingerConfig getSingerInfoUtau( int program_change ) {
            if ( 0 <= program_change && program_change < EditorConfig.UtauSingers.Count ) {
                return EditorConfig.UtauSingers[program_change];
            } else {
                return null;
            }
        }

        public static string _VERSION {
            get {
                string prefix = "";
                string rev = "";
                // $Id: AppManager.cs 226 2009-06-04 16:02:28Z kbinani $
                string id = GetAssemblyConfigurationAttribute();
                string[] spl0 = id.Split( " ".ToCharArray(), StringSplitOptions.RemoveEmptyEntries );
                if ( spl0.Length >= 3 ){
                    string s = spl0[2];
#if DEBUG
                    Common.DebugWriteLine( "AppManager.get__VERSION; s=" + s );
#endif
                    string[] spl = s.Split( " ".ToCharArray(), StringSplitOptions.RemoveEmptyEntries );
                    if ( spl.Length > 0 ) {
                        rev = spl[0];
                    }
                }
                if ( rev == "" ) {
                    rev = "?";
                }
#if DEBUG
                prefix = "(rev: " + rev + "; build: debug)";
#else
                prefix = "(rev: " + rev + "; build: release)";
#endif
                return GetAssemblyFileVersion( typeof( AppManager ) ) + " " + prefix;
            }
        }

        public static string GetAssemblyConfigurationAttribute() {
            Assembly a = Assembly.GetAssembly( typeof( AppManager ) );
            AssemblyConfigurationAttribute attr = (AssemblyConfigurationAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyConfigurationAttribute ) );
            return attr.Configuration;
        }

        public static string GetAssemblyFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return afva.Version;
        }

        public static string GetAssemblyNameAndFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return a.GetName().Name + " v" + afva.Version;
        }

        public static SolidBrush HilightBrush {
            get {
                return s_hilight_brush;
            }
        }

        public static Color HilightColor {
            get {
                return s_hilight_brush.Color;
            }
            set {
                s_hilight_brush = new SolidBrush( value );
            }
        }

        /// <summary>
        /// ベースとなるテンポ。
        /// </summary>
        public static int BaseTempo {
            get {
                return s_base_tempo;
            }
            set {
                s_base_tempo = value;
            }
        }
    }

}
