﻿/*
 * VsqFileEx.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Drawing;

using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    [Serializable]
    public class VsqFileEx : VsqFile, ICloneable, ICommandRunnable<CadenciiCommand> {
        private List<AttachedCurve> m_attached_curve;

        public List<AttachedCurve> AttachedCurves {
            get {
                return m_attached_curve;
            }
            set {
                m_attached_curve = value;
            }
        }

        public new object Clone(){
            VsqFileEx ret = new VsqFileEx( 0, 1, 4, 4, 5000 );
            ret.m_tracks = new List<VsqTrack>();
            for ( int i = 0; i < m_tracks.Count; i++ ) {
                ret.m_tracks.Add( (VsqTrack)m_tracks[i].Clone() );
            }
            ret.m_tempo_table = new List<TempoTableEntry>();
            for ( int i = 0; i < m_tempo_table.Count; i++ ) {
                ret.m_tempo_table.Add( (TempoTableEntry)m_tempo_table[i].Clone() );
            }
            ret.m_timesig_table = new List<TimeSigTableEntry>();
            for ( int i = 0; i < m_timesig_table.Count; i++ ) {
                ret.m_timesig_table.Add( (TimeSigTableEntry)m_timesig_table[i].Clone() );
            }
            ret.m_tpq = m_tpq;
            ret.m_total_clocks = m_total_clocks;
            ret.m_base_tempo = m_base_tempo;
            ret.m_master = (VsqMaster)m_master.Clone();
            ret.m_mixer = (VsqMixer)m_mixer.Clone();
            ret.m_premeasure_clocks = m_premeasure_clocks;
            ret.m_attached_curve.Clear();
            for ( int i = 0; i < m_attached_curve.Count; i++ ) {
                ret.m_attached_curve.Add( (AttachedCurve)m_attached_curve[i].Clone() );
            }
            return ret;
        }

        public static CadenciiCommand GCommandAddBezierChain( int track, CurveType curve_type, int chain_id, int clock_resolution, BezierChain chain ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.AddBezierChain;
            ret.Args = new object[5];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = (BezierChain)chain.Clone();
            ret.Args[3] = clock_resolution;
            ret.Args[4] = chain_id;
            return ret;
        }

        public static CadenciiCommand GCommandDeleteBezierChain( int track, CurveType curve_type, int chain_id, int clock_resolution ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.DeleteBezierChain;
            ret.Args = new object[4];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = chain_id;
            ret.Args[3] = clock_resolution;
            return ret;
        }

        public static CadenciiCommand GCommandReplaceBezierChain( int track, CurveType curve_type, int chain_id, BezierChain chain, int clock_resolution ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.ReplaceBezierChain;
            ret.Args = new object[5];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = chain_id;
            ret.Args[3] = chain;
            ret.Args[4] = clock_resolution;
            return ret;
        }

        public static CadenciiCommand GCommandReplace( VsqFileEx vsq ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.Replace;
            ret.Args = new object[1];
            ret.Args[0] = (VsqFileEx)vsq.Clone();
            return ret;
        }

        public static CadenciiCommand GCommandReplaceAttachedCurveRange( int track, Dictionary<CurveType, List<BezierChain>> attached_curves ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.ReplaceAttachedCurveRange;
            ret.Args = new object[2];
            ret.Args[0] = track;
            ret.Args[1] = attached_curves;
            return ret;
        }

        public CadenciiCommand Execute( CadenciiCommand command ) {
#if DEBUG
            Common.DebugWriteLine( "VsqFileEx.Execute" );
#endif
            CadenciiCommand ret = null;
            if ( command.Type == CadenciiCommandType.VsqCommand ) {
                ret = new CadenciiCommand();
                ret.Type = CadenciiCommandType.VsqCommand;
                ret.VsqCommand = base.Execute( command.VsqCommand );
            } else {
                if ( command.Type == CadenciiCommandType.AddBezierChain ) {
                    #region AddBezierChain
#if DEBUG
                    Common.DebugWriteLine( "    AddBezierChain" );
#endif
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    BezierChain chain = (BezierChain)command.Args[2];
                    int clock_resolution = (int)command.Args[3];
                    int added_id = (int)command.Args[4];
                    AttachedCurves[track - 1].AddBezierChain( curve_type, chain, added_id );
                    ret = GCommandDeleteBezierChain( track, curve_type, added_id, clock_resolution );
                    if ( chain.Count > 1 ) {
                        int min = (int)chain.points[0].Base.X;
                        int max = min;
                        for ( int i = 1; i < chain.points.Count; i++ ) {
                            min = Math.Min( min, (int)chain.points[i].Base.X );
                            max = Math.Max( max, (int)chain.points[i].Base.X );
                        }
                        int max_value = curve_type.Maximum;
                        int min_value = curve_type.Minimum;
                        if ( min < max ) {
                            List<BPPair> edit = new List<BPPair>();
                            int last_value = int.MaxValue;
                            for ( int clock = min; clock < max; clock += clock_resolution ) {
                                int value = (int)chain.GetValue( (float)clock );
                                if ( value < min_value ) {
                                    value = min_value;
                                } else if ( max_value < value ) {
                                    value = max_value;
                                }
                                if ( value != last_value ) {
                                    edit.Add( new BPPair( clock, value ) );
                                    last_value = value;
                                }
                            }
                            int value2 = Tracks[track][curve_type.Name][max];
#if DEBUG
                            Common.DebugWriteLine( "        " + max + "," + value2 );
#endif
                            edit.Add( new BPPair( max, value2 ) );
                            command.VsqCommand = VsqCommand.GCommandTrackEditCurve( track, curve_type.Name, edit );
                        }
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.DeleteBezierChain ) {
                    #region DeleteBezierChain
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    int chain_id = (int)command.Args[2];
                    int clock_resolution = (int)command.Args[3];
                    BezierChain chain = (BezierChain)AttachedCurves[track - 1][curve_type][chain_id].Clone();
                    AttachedCurves[track - 1][curve_type].Remove( chain_id );
                    ret = GCommandAddBezierChain( track, curve_type, chain_id, clock_resolution, chain );
                    if ( command.VsqCommand != null && ret != null ) {
                        ret.VsqCommand = base.Execute( command.VsqCommand );
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.ReplaceBezierChain ) {
                    #region ReplaceBezierChain
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    int chain_id = (int)command.Args[2];
                    BezierChain chain = (BezierChain)command.Args[3];
                    int clock_resolution = (int)command.Args[4];
                    BezierChain target = (BezierChain)AttachedCurves[track - 1][curve_type][chain_id].Clone();
                    AttachedCurves[track - 1][curve_type][chain_id] = chain;
                    ret = GCommandReplaceBezierChain( track, curve_type, chain_id, target, clock_resolution );
                    if ( chain.Count == 1 ) {
                        int ex_min = (int)chain.points[0].Base.X;
                        int ex_max = ex_min;
                        if ( target.points.Count > 1 ) {
                            for ( int i = 1; i < target.points.Count; i++ ) {
                                ex_min = Math.Min( ex_min, (int)target.points[i].Base.X );
                                ex_max = Math.Max( ex_max, (int)target.points[i].Base.X );
                            }
                            if ( ex_min < ex_max ) {
                                int default_value = curve_type.Default;
                                List<BPPair> edit = new List<BPPair>();
                                edit.Add( new BPPair( ex_min, default_value ) );
                                edit.Add( new BPPair( ex_max, default_value ) );
                                command.VsqCommand = VsqCommand.GCommandTrackEditCurve( track, curve_type.Name, edit );
                            }
                        }
                    } else if ( chain.Count > 1 ) {
                        int min = (int)chain.points[0].Base.X;
                        int max = min;
                        for ( int i = 1; i < chain.points.Count; i++ ) {
                            min = Math.Min( min, (int)chain.points[i].Base.X );
                            max = Math.Max( max, (int)chain.points[i].Base.X );
                        }
                        int ex_min = min;
                        int ex_max = max;
                        if ( target.points.Count > 0 ) {
                            ex_min = (int)target.points[0].Base.X;
                            ex_max = ex_min;
                            for ( int i = 1; i < target.points.Count; i++ ) {
                                ex_min = Math.Min( ex_min, (int)target.points[i].Base.X );
                                ex_max = Math.Max( ex_max, (int)target.points[i].Base.X );
                            }
                        }
                        int max_value = curve_type.Maximum;
                        int min_value = curve_type.Minimum;
                        int default_value = curve_type.Default;
                        List<BPPair> edit = new List<BPPair>();
                        if ( ex_min < min ) {
                            edit.Add( new BPPair( ex_min, default_value ) );
                            /*if ( !Tracks[track][curve_type].ContainsKey( ex_min ) ) {
                                edit.Add( new BPPair( ex_min, Tracks[track][curve_type][ex_min] ) );
                            }
                            foreach ( int key_clock in Tracks[track][curve_type].GetKeyClockEnumerator() ) {
                                if ( ex_min <= key_clock && key_clock < min ) {
                                    edit.Add( new BPPair( key_clock, Tracks[track][curve_type][key_clock] ) );
                                } else if ( min <= key_clock ) {
                                    break;
                                }
                            }*/
                        }
                        if ( min < max ) {
                            int last_value = int.MaxValue;
                            for ( int clock = min; clock < max; clock += clock_resolution ) {
                                int value = (int)chain.GetValue( (float)clock );
                                if ( value < min_value ) {
                                    value = min_value;
                                } else if ( max_value < value ) {
                                    value = max_value;
                                }
                                if ( last_value != value ) {
                                    edit.Add( new BPPair( clock, value ) );
                                    last_value = value;
                                }
                            }
                            int value2 = 0;
                            if ( max < ex_max ) {
                                value2 = (int)chain.GetValue( max );
                                if ( value2 < min_value ) {
                                    value2 = min_value;
                                } else if ( max_value < value2 ) {
                                    value2 = max_value;
                                }
                            } else {
                                value2 = Tracks[track][curve_type.Name][max];
                            }
                            edit.Add( new BPPair( max, value2 ) );
                        }
                        if ( max < ex_max ) {
                            /*foreach ( int key_clock in Tracks[track][curve_type].GetKeyClockEnumerator() ) {
                                if ( max < key_clock && key_clock < ex_max ) {
                                    edit.Add( new BPPair( key_clock, Tracks[track][curve_type][key_clock] ) );
                                } else if ( ex_max <= key_clock ) {
                                    break;
                                }
                            }
                            edit.Add( new BPPair( ex_max, Tracks[track][curve_type][ex_max] ) );*/
                            if ( max + 1 < ex_max ) {
                                edit.Add( new BPPair( max + 1, default_value ) );
                            }
                            edit.Add( new BPPair( ex_max, default_value ) );
                        }
                        if ( edit.Count > 0 ) {
                            command.VsqCommand = VsqCommand.GCommandTrackEditCurve( track, curve_type.Name, edit );
                        }
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.Replace ) {
                    #region Replace
                    VsqFileEx vsq = (VsqFileEx)command.Args[0];
                    VsqFileEx inv = (VsqFileEx)this.Clone();
                    m_tracks.Clear();
                    for ( int i = 0; i < vsq.m_tracks.Count; i++ ) {
                        m_tracks.Add( (VsqTrack)vsq.m_tracks[i].Clone() );
                    }
                    m_tempo_table.Clear();
                    for ( int i = 0; i < vsq.m_tempo_table.Count; i++ ) {
                        m_tempo_table.Add( (TempoTableEntry)vsq.m_tempo_table[i].Clone() );
                    }
                    m_timesig_table.Clear();
                    for ( int i = 0; i < vsq.m_timesig_table.Count; i++ ) {
                        m_timesig_table.Add( (TimeSigTableEntry)vsq.m_timesig_table[i].Clone() );
                    }
                    m_tpq = vsq.m_tpq;
                    m_total_clocks = vsq.m_total_clocks;
                    m_base_tempo = vsq.m_base_tempo;
                    m_master = (VsqMaster)vsq.m_master.Clone();
                    m_mixer = (VsqMixer)vsq.m_mixer.Clone();
                    m_premeasure_clocks = vsq.m_premeasure_clocks;
                    m_attached_curve.Clear();
                    for ( int i = 0; i < vsq.m_attached_curve.Count; i++ ) {
                        m_attached_curve.Add( (AttachedCurve)vsq.m_attached_curve[i].Clone() );
                    }
                    UpdateTotalClocks();
                    ret = GCommandReplace( inv );
                    #endregion
                } else if ( command.Type == CadenciiCommandType.ReplaceAttachedCurveRange ) {
                    #region ReplaceAttachedCurveRange
                    int track = (int)command.Args[0];
                    Dictionary<CurveType, List<BezierChain>> curves = (Dictionary<CurveType, List<BezierChain>>)command.Args[1];
                    Dictionary<CurveType, List<BezierChain>> inv = new Dictionary<CurveType, List<BezierChain>>();
                    foreach ( CurveType ct in curves.Keys ) {
                        List<BezierChain> chains = new List<BezierChain>();
                        foreach ( int id in this.AttachedCurves[track - 1][ct].Keys ) {
                            chains.Add( (BezierChain)this.AttachedCurves[track - 1][ct][id].Clone() );
                        }
                        inv.Add( ct, chains );

                        this.AttachedCurves[track - 1][ct].Clear();
                        foreach ( BezierChain bc in curves[ct] ) {
                            this.AttachedCurves[track - 1][ct].Add( bc.ID, bc );
                        }
                    }
                    ret = GCommandReplaceAttachedCurveRange( track, inv );
                    #endregion
                }
                if ( command.VsqCommand != null && ret != null ) {
                    ret.VsqCommand = base.Execute( command.VsqCommand );
                }
            }
            return ret;
        }

        public VsqFileEx( int default_singer_program_change, int pre_measure, int numerator, int denominator, int tempo ) :
            base( default_singer_program_change, pre_measure, numerator, denominator, tempo ) {
            m_attached_curve = new List<AttachedCurve>();
            for ( int i = 1; i < Tracks.Count; i++ ) {
                m_attached_curve.Add( new AttachedCurve() );
            }
        }

        public VsqFileEx( string _fpath ) :
            base( _fpath ){
            m_attached_curve = new List<AttachedCurve>();
            for ( int i = 1; i < Tracks.Count; i++ ) {
                m_attached_curve.Add( new AttachedCurve() );
            }
        }
    }

}
