/*
 * $HeadURL: https://svn.apache.org/repos/asf/jakarta/httpcomponents/httpclient/tags/4.0-alpha2/module-client/src/main/java/org/apache/http/cookie/CookieOrigin.java $
 * $Revision: 535132 $
 * $Date: 2007-05-04 11:01:16 +0200 (Fri, 04 May 2007) $
 *
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package org.apache.http.cookie;

import org.apache.http.util.CharArrayBuffer;

/**
 * CookieOrigin class incapsulates details of an origin server that 
 * are relevant when parsing, validating or matching HTTP cookies.
 * 
 * @author <a href="mailto:oleg at ural.ru">Oleg Kalnichevski</a>
 * 
 * @since 4.0
 */
public final class CookieOrigin {

	private final String host;
	private final int port;
	private final String path;
	private final boolean secure;
	
	public CookieOrigin(final String host, int port, final String path, boolean secure) {
		super();
        if (host == null) {
            throw new IllegalArgumentException(
                    "Host of origin may not be null");
        }
        if (host.trim().equals("")) {
            throw new IllegalArgumentException(
                    "Host of origin may not be blank");
        }
        if (port < 0) {
            throw new IllegalArgumentException("Invalid port: " + port);
        }
        if (path == null) {
            throw new IllegalArgumentException(
                    "Path of origin may not be null.");
        }
		this.host = host.toLowerCase();
		this.port = port;
        if (!path.trim().equals("")) {
            this.path = path;
        } else {
            this.path = "/";
        }
		this.secure = secure;
	}

	public String getHost() {
		return this.host;
	}

	public String getPath() {
		return this.path;
	}

	public int getPort() {
		return this.port;
	}

	public boolean isSecure() {
		return this.secure;
	}

    public String toString() {
        CharArrayBuffer buffer = new CharArrayBuffer(32);
        buffer.append("[");
        if (this.secure) {
            buffer.append("(secure)");
        }
        buffer.append(this.host);
        buffer.append(":");
        buffer.append(Integer.toString(this.port));
        buffer.append(this.path);
        buffer.append("]");
        return buffer.toString();
    }
    
}
