/**
 * Copyright (C) 2006 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.inject.binder;

import com.google.inject.Key;
import com.google.inject.Provider;
import com.google.inject.TypeLiteral;

/**
 * Links a binding to another binding or an instance.
 *
 * @author crazybob@google.com (Bob Lee)
 */
public interface LinkedBindingBuilder<T> extends ScopedBindingBuilder {

  /**
   * Binds to another binding with the specified type.
   */
  ScopedBindingBuilder to(Class<? extends T> implementation);

  /**
   * Binds to another binding with the specified type.
   */
  ScopedBindingBuilder to(TypeLiteral<? extends T> implementation);

  /**
   * Binds to another binding with the specified key. 
   */
  ScopedBindingBuilder to(Key<? extends T> targetKey);

  /**
   * Binds to the given instance. The Injector will automatically inject the
   * members of this instance when it is first created. See {@link
   * com.google.inject.Injector#injectMembers(Object)}.
   */
  void toInstance(T instance);

  /**
   * Binds to instances generated by the given Provider. The Injector will
   * automatically inject the members of this provider instance when it is first
   * created. See {@link com.google.inject.Injector#injectMembers(Object)}.
   */
  ScopedBindingBuilder toProvider(Provider<? extends T> provider);

  /**
   * Binds to instances from the provider bound to the given provider type.
   */
  ScopedBindingBuilder toProvider(
      Class<? extends Provider<? extends T>> providerType);

  /**
   * Binds to instances from the provider bound to the given key.
   */
  ScopedBindingBuilder toProvider(
      Key<? extends Provider<? extends T>> providerKey);
}
