/*
//	lockfree_stack.hpp
//
//	Distributed under the Boost Software License, Version 1.0.(See accompanying
//	file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt
*/

#ifndef LOCKFREE_QUEUE_H
#define LOCKFREE_QUEUE_H

#include <boost/noncopyable.hpp>

template<class Tvalue>
class lockfree_queue : protected boost::noncopyable {
protected:
	struct node_type {
		Tvalue*		value;
		node_type*	next;
		node_type() : value( 0 ) , next( NULL ) {}
	};

	mutable volatile size_t		counter;
	node_type*			headloc;
	node_type*			tailloc;
	node_type*			dummy;

public:
	// constractor
	explicit lockfree_queue() : counter(0){
		dummy = new node_type();
		dummy->next = dummy;
		headloc = tailloc = dummy;
	}

	// destractor
	~lockfree_queue(){
		while( counter ){
			pop();
		}
		delete dummy;
	}

	//pusher
	bool push(const Tvalue* value){
		volatile node_type* new_node = new node_type();
		node_type* tail, *next;
		new_node->value = const_cast<Tvalue*>(value);
		new_node->next = dummy;
		//transaction st
		while( true ){
			tail = tailloc;
			if( __sync_bool_compare_and_swap(&tailloc->next,dummy,new_node) ){
				__sync_bool_compare_and_swap(&tailloc,tail,new_node);
				break;
			} else {
				next = tail->next;
				if( next != dummy ){
					__sync_bool_compare_and_swap(&tailloc,tail,next);
				}
			}
		}
		__sync_add_and_fetch( &counter, 1 );
		//transaction ed
		return true;
	}

	//poper
	 Tvalue* pop() {
		node_type *next,*head;
		Tvalue* value;
		//transaction start
		while( true ){
			head = headloc;
			next = headloc->next;
			if( head == dummy ){
				if( next == dummy ) return NULL;
				if( __sync_bool_compare_and_swap(&headloc,head,next->next) ){
					__sync_lock_test_and_set(&dummy->next,dummy);
					value = next->value;
					delete next;
					break;
				}
			} else {
				if( __sync_bool_compare_and_swap(&headloc,head,next) ){
					__sync_bool_compare_and_swap(&tailloc,head,dummy);
					value = head->value;
					delete head;
					break;
				}
			}
		}
		__sync_sub_and_fetch( &counter, 1 );
		return value;
		//transaction ed
	}
	//size
	size_t size() const{ return counter; }
	//empty
	bool empty() const{ return !counter; }
};

#endif	// LOCKFREE_QUEUE_H
