
/**
 * @class シンボルを表示する画像か、文字を格納するspanを提供するパーツ
 * @constructor
 * @extends Bredima.Parts
 * @param {Object} bredima bredimaオブジェクト MathMLの判定に使用
 * @param {String} sid 表示するシンボルのID だいたいはMathML名
 * @param {Number} level 表示サイズのレベル
 * @param {String} fixsize 表示サイズをTokenの高さに揃える場合はfixsizeと渡す
 */
Bredima.Parts.Symbol = function(bredima, sid, level, fixsize) {
    if(!Bredima.Parts.Symbol.all[sid]) {
	// 存在しないコードが渡ってきたら刎ねる
	//Bredima.note(sid); // デバッグ用
	return false;
    }
    this.symbolID = sid;
    this.symbol = Bredima.Parts.Symbol.all[this.symbolID];
    this.level = level;
    this.mode = (bredima.getConfig('isMML')) ? 'text' : 'img';
    if(this.mode == 'text') {
	this.obj = document.createElement('span');
	this.obj.appendChild(document.createTextNode(this.symbol.code));
	this.width = Bredima.Token.width[this.level];
	this.height = Bredima.Token.height[this.level];
    }
    else {
	this.obj = document.createElement('img');
	this.obj.src = Bredima.util.getTexURI('\\' + this.symbol.tex, level);
	var s = Bredima.Token.height[this.level];
	var img = this.symbol.img[level];
	this.width = (fixsize) ? s : img.width;
	this.height = (fixsize) ? s : img.height;
	if(fixsize) {
	    this.obj.style.marginTop = img.margin.top + 'px';
	    this.obj.style.marginRight = img.margin.right + 'px';
	    this.obj.style.marginBottom = img.margin.bottom + 'px';
	    this.obj.style.marginLeft = img.margin.left + 'px';
	}
    }
    this.setClassName('parts');
}

Bredima.util.inherit(Bredima.Parts.Symbol, Bredima.Parts);

/**
 * MathML表記を返す
 * @param {Boolean} expand 実体参照でなく実体を渡すか
 */
Bredima.Parts.Symbol.prototype.toMML = function(expand) {
    return (expand) ? this.symbol.code : 
	'&' + ((this.symbol.mathml) ? this.symbol.mathml : this.symbolID) + ';';
}

/**
 * LaTeX表記を返す
 */
Bredima.Parts.Symbol.prototype.toLatex = function() {
    return '\\' + this.symbol.tex;
}

/**
 * シンボルの種類を返す
 */
Bredima.Parts.Symbol.prototype.getGroup = function() {return this.symbol.group;}


Bredima.Parts.Symbol.list = {
    // 演算子
    operator : {
    sdot: {code: '\u22c5', tex: 'cdot', img: ['',
        {width: 8, height: 2, margin: {top: 10, right: 7, bottom: 10, left: 7}},
        {width: 8, height: 2, margin: {top: 8, right: 5, bottom: 8, left: 5}}]},
    times: {code: '\u00d7', tex: 'times', img: ['',
        {width: 8, height: 9, margin: {top: 7, right: 7, bottom: 6, left: 7}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    divide: {code: '\u00f7', tex: 'div', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    PlusMinus: {code: '\u00b1', tex: 'pm', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    MinusPlus: {code: '\u2213', tex: 'mp', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    NotEqual: {code: '\u2260', tex: 'neq', img: ['',
        {width: 16, height: 23, margin: {top: 0, right: 3, bottom: 0, left: 3}},
        {width: 16, height: 19, margin: {top: 0, right: 1, bottom: 0, left: 1}}]},
    lE: {code: '\u2266', tex: 'leq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    gE: {code: '\u2267', tex: 'geq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    sub: {code: '\u2282', tex: 'subset', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    sup: {code: '\u2283', tex: 'supset', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    sube: {code: '\u2286', tex: 'subseteq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    supe: {code: '\u2287', tex: 'supseteq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    isin: {code: '\u2208', tex: 'in', img: ['',
        {width: 8, height: 11, margin: {top: 6, right: 7, bottom: 5, left: 7}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    ni: {code: '\u220b', tex: 'ni', img: ['',
        {width: 8, height: 11, margin: {top: 6, right: 7, bottom: 5, left: 7}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    cup: {code: '\u222a', tex: 'cup', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    cap: {code: '\u2229', tex: 'cap', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    larr: {code: '\u2190', tex: 'leftarrow', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    rarr: {code: '\u2192', tex: 'rightarrow', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    lArr: {code: '\u21d0', tex: 'Leftarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    rArr: {code: '\u21d2', tex: 'Rightarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    hArr: {code: '\u21d4', tex: 'Leftrightarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    compfn: {code: '\u2218', tex: 'circ', img: ['',
        {width: 8, height: 7, margin: {top: 8, right: 7, bottom: 7, left: 7}},
        {width: 8, height: 6, margin: {top: 6, right: 5, bottom: 6, left: 5}}]},
    ctdot: {code: '\u22ef', tex: 'cdots', img: ['',
        {width: 24, height: 2, margin: {top: 10, right: 0, bottom: 10, left: 0}},
        {width: 24, height: 2, margin: {top: 8, right: 0, bottom: 8, left: 0}}]},
    equiv: {code: '\u2261', tex: 'equiv', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    perp: {code: '\u22a5', tex: 'perp', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]}
    },
    // 識別子
    identifier : {
    alpha: {code: '\u03b1', tex: 'alpha', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 8, height: 7, margin: {top: 6, right: 5, bottom: 5, left: 5}}]},
    beta: {code: '\u03b2', tex: 'beta', img: ['',
        {width: 16, height: 17, margin: {top: 3, right: 3, bottom: 2, left: 3}},
        {width: 16, height: 14, margin: {top: 2, right: 1, bottom: 2, left: 1}}]},
    delta: {code: '\u03b4', tex: 'delta', img: ['',
        {width: 8, height: 14, margin: {top: 4, right: 7, bottom: 4, left: 7}},
        {width: 8, height: 11, margin: {top: 4, right: 5, bottom: 3, left: 5}}]},
    epsiv: {code: '\u025b', tex: 'varepsilon', img: ['',
        {width: 8, height: 8, margin: {top: 7, right: 7, bottom: 7, left: 7}},
        {width: 8, height: 7, margin: {top: 6, right: 5, bottom: 5, left: 5}}]},
    gamma: {code: '\u03b3', tex: 'gamma', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    omega: {code: '\u03c9', tex: 'omega', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    phi: {code: '\u03c6', tex: 'phi', img: ['',
        {width: 16, height: 17, margin: {top: 3, right: 3, bottom: 2, left: 3}},
        {width: 8, height: 14, margin: {top: 2, right: 5, bottom: 2, left: 5}}]},
    phiv: {code: '\u03d5', tex: 'varphi', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    theta: {code: '\u03b8', tex: 'theta', img: ['',
        {width: 8, height: 13, margin: {top: 5, right: 7, bottom: 4, left: 7}},
        {width: 8, height: 11, margin: {top: 4, right: 5, bottom: 3, left: 5}}]},
    Delta: {code: '\u0394', tex: 'Delta', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    pi: {code: '\u03c0', tex: 'pi', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    infin: {code: '\u221e', tex: 'infty', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    bigtriangleup: {code: '\u25b3', tex: 'bigtriangleup', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]}
    },
    // その他パーツ
    other: {
    OverBar: {code: '\u00af', tex: ''},
    sum: {code: '\u2211', tex: 'sum', img: ['',
        {width: 24, height: 25, margin: {top: 0, right: 0, bottom: 0, left: 0}},
        {width: 16, height: 16, margin: {top: 1, right: 1, bottom: 1, left: 1}}]},
	mathint: {code: '\u222b', tex: 'int', mathml: 'int', img: ['',
        {width: 16, height: 40, margin: {top: 0, right: 6, bottom: 0, left: 7}},
        {width: 16, height: 18, margin: {top: 0, right: 1, bottom: 0, left: 1}}]}
	// intが予約語なので避ける
    }
};

Bredima.Parts.Symbol.all = {}; // 初期化時に設定

Bredima.Parts.Symbol.initializeOnLoad = function() {
    // all配列作成
    for(var group in Bredima.Parts.Symbol.list) {
	for(var id in Bredima.Parts.Symbol.list[group]) {
	    Bredima.Parts.Symbol.all[id] = Bredima.Parts.Symbol.list[group][id];
	    Bredima.Parts.Symbol.all[id].group = group;
	}
    }
    Bredima.Init.ackFinish();
}
Bredima.Init.attachInit(Bredima.Parts.Symbol);
