
/**
 * @class シンボルを表示する画像か、文字を格納するspanを提供するパーツ
 * @constructor
 * @extends Bredima.Parts
 * @param {Object} bredima bredimaオブジェクト MathMLの判定に使用
 * @param {String} sid 表示するシンボルのID だいたいはMathML名
 * @param {Number} level 表示サイズのレベル
 * @param {String} fixsize 表示サイズをTokenの高さに揃える場合はfixsizeと渡す
 */
Bredima.Parts.Symbol = function(bredima, sid, level, fixsize) {
    this.symbolID = sid;
    this.symbol = Bredima.Parts.Symbol.all[this.symbolID];
    this.level = level;
    this.mode = (bredima.getConfig('isMML')) ? 'text' : 'img';
    if(this.mode == 'text') {
	this.obj = document.createElement('span');
	this.obj.appendChild(document.createTextNode(this.symbol.code));
	this.width = Bredima.Token.width[this.level];
	this.height = Bredima.Token.height[this.level];
    }
    else {
	this.obj = document.createElement('img');
	this.obj.src = this.symbol.img[this.level].obj.src;
	this.width = this.symbol.img[this.level].width;
	this.height = this.symbol.img[this.level].height;
	if(fixsize) {
	    var s = Bredima.Token.height[this.level] - 2;
	    this.setSize((s < this.width) ? s : this.width, (s < this.height) ? s : this.height);
	    if(this.width < s) {
		this.obj.style.marginLeft = Math.ceil((s - this.width) / 2) + 'px';
		this.obj.style.marginRight = Math.floor((s - this.width) / 2) + 'px';
	    }
	    if(this.height < s) {
		this.obj.style.marginTop = Math.ceil((s - this.height) / 2) + 'px';
		this.obj.style.marginBottom = Math.floor((s - this.height) / 2) + 'px';
	    }
	    this.width = s;
	    this.height = s;
	}
    }
}

Bredima.util.inherit(Bredima.Parts.Symbol, Bredima.Parts);

/**
 * MathML表記を返す
 * @param {Boolean} expand 実体参照でなく実体を渡すか
 */
Bredima.Parts.Symbol.prototype.toMML = function(expand) {
    return (expand) ? this.symbol.code : 
	'&' + ((this.symbol.mathml) ? this.symbol.mathml : this.symbolID) + ';';
}

/**
 * LaTeX表記を返す
 */
Bredima.Parts.Symbol.prototype.toLatex = function() {
    return '\\' + this.symbol.tex;
}

/**
 * シンボルの種類を返す
 */
Bredima.Parts.Symbol.prototype.getGroup = function() {return this.symbol.group;}

Bredima.Parts.Symbol.observer = new Bredima.Observer();
Bredima.Parts.Symbol.loadedNumber = 0;
Bredima.Parts.Symbol.initialized = false;

/**
 * Bredima.Parts.Symbolクラス初期化
 */
Bredima.Parts.Symbol.initializeOnLoad = function() {
    // all配列作成
    for(var group in Bredima.Parts.Symbol.list) {
	for(var id in Bredima.Parts.Symbol.list[group]) {
	    Bredima.Parts.Symbol.all[id] = Bredima.Parts.Symbol.list[group][id];
	    Bredima.Parts.Symbol.all[id].group = group;
	    Bredima.Parts.Symbol.all_length++;
	}
    }

    // 画像の先読み
    for(var id in Bredima.Parts.Symbol.all) {
	var s = Bredima.Parts.Symbol.all[id];
	s.img = new Array;
	for(var size = 1; size <= 2; size++) {
	    var img = new Image();
	    s.img[size] = {obj: img};
	    img.onload = (function(a1, a2) {
			      return function(){Bredima.Parts.Symbol._imgOnload(a1, a2);};
			  })(id, size);
	    img.src = Bredima.util.getTexURI('\\' + s.tex, size);
	}
    }
}
Bredima.Init.attachInit(Bredima.Parts.Symbol);

/**
 * 先読み画像が読み込まれた時の処理
 * @private
 */
Bredima.Parts.Symbol._imgOnload = function(id, size) {
    var s = Bredima.Parts.Symbol.all[id].img[size];
    s.width = s.obj.width;
    s.height = s.obj.height;
    s.obj.onload = '';

    // 全ての画像がロードされたら
    if(++Bredima.Parts.Symbol.loadedNumber == Bredima.Parts.Symbol.all_length * 2) {
	Bredima.Init.ackFinish();
    }
}

Bredima.Parts.Symbol.all = {}; // 初期化時に設定
Bredima.Parts.Symbol.all_length = 0;
Bredima.Parts.Symbol.list = {
    // 演算子
    operator : {
	sdot: {code: '\u22c5', tex: 'cdot'},
	times: {code: '\u00d7', tex: 'times'},
	divide: {code: '\u00f7', tex: 'div'},
	PlusMinus: {code: '\u00b1', tex: 'pm'},
	MinusPlus: {code: '\u2213', tex: 'mp'},
	NotEqual: {code: '\u2260', tex: 'neq'},
	//efDot: {code: '\u2252', tex: ''},
	lE: {code: '\u2266', tex: 'leq'},
	gE: {code: '\u2267', tex: 'geq'},
	sub: {code: '\u2282', tex: 'subset'},
	sup: {code: '\u2283', tex: 'supset'},
	sube: {code: '\u2286', tex: 'subseteq'},
	supe: {code: '\u2287', tex: 'supseteq'},
	isin: {code: '\u2208', tex: 'in'},
	ni: {code: '\u220b', tex: 'ni'},
	cup: {code: '\u222a', tex: 'cup'},
	cap: {code: '\u2229', tex: 'cap'},
	larr: {code: '\u2190', tex: 'leftarrow'},
	rarr: {code: '\u2192', tex: 'rightarrow'},
	lArr: {code: '\u21d0', tex: 'Leftarrow'},
	rArr: {code: '\u21d2', tex: 'Rightarrow'},
	hArr: {code: '\u21d4', tex: 'Leftrightarrow'},
	compfn: {code: '\u2218', tex: 'circ'},
	ctdot: {code: '\u22ef', tex: 'cdots'},
	equiv: {code: '\u2261', tex: 'equiv'},
	//bsim: {code: '\u223d', tex: ''},
	perp: {code: '\u22a5', tex: 'perp'}
    },
    // 識別子
    identifier : {
	alpha: {code: '\u03b1', tex: 'alpha'},
	beta: {code: '\u03b2', tex: 'beta'},
	delta: {code: '\u03b4', tex: 'delta'},
	epsiv: {code: '\u025b', tex: 'varepsilon'},
	gamma: {code: '\u03b3', tex: 'gamma'},
	omega: {code: '\u03c9', tex: 'omega'},
	phi: {code: '\u03c6', tex: 'phi'},
	phiv: {code: '\u03d5', tex: 'varphi'},
	theta: {code: '\u03b8', tex: 'theta'},
	Delta: {code: '\u0394', tex: 'Delta'},
	pi: {code: '\u03c0', tex: 'pi'},
	infin: {code: '\u221e', tex: 'infty'},
	bigtriangleup: {code: '\u25b3', tex: 'bigtriangleup'}
	//angle: {code: '\u2220', tex: ''},
	//deg: {code: '\u00b0', tex: ''}
    },
    // その他パーツ
    other: {
	OverBar: {code: '\u00af', tex: ''},
	sum: {code: '\u2211', tex: 'sum'},
	mathint: {code: '\u222b', tex: 'int', mathml: 'int'}
    }
};
