/**
 * @file lcdimg.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 * @brief LCD image model.
 * @details This LCD image model provide a image model for a LCD.
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <stdlib.h>
#include "lcdimg.h"
#include "lcdprof.h"
#include "lcdfont.h"
#include "bmpimg.h"

static const uint8_t font5x8_master[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(1)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(2)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(3)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(4)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(5)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(6)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(7)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(8)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(1)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(2)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(3)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(4)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(5)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(6)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(7)
    0x00, 0x00, 0x00, 0x00, 0x00, // CGRAM(8)
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, // (white space)
    0x00, 0x00, 0x5F, 0x00, 0x00, // !
    0x00, 0x07, 0x00, 0x07, 0x00, // "
    0x14, 0x7F, 0x14, 0x7F, 0x14, // #
    0x24, 0x2A, 0x7F, 0x2A, 0x12, // $
    0x23, 0x13, 0x08, 0x64, 0x62, // %
    0x36, 0x49, 0x55, 0x22, 0x50, // &
    0x00, 0x05, 0x03, 0x00, 0x00, // '
    0x00, 0x1C, 0x22, 0x41, 0x00, // (
    0x00, 0x41, 0x22, 0x1C, 0x00, // )
    0x08, 0x2A, 0x1C, 0x2A, 0x08, // *
    0x08, 0x08, 0x3E, 0x08, 0x08, // +
    0x00, 0x50, 0x30, 0x00, 0x00, // ,
    0x08, 0x08, 0x08, 0x08, 0x08, // -
    0x00, 0x60, 0x60, 0x00, 0x00, // .
    0x20, 0x10, 0x08, 0x04, 0x02, // /
    0x3E, 0x51, 0x49, 0x45, 0x3E, // 0
    0x00, 0x42, 0x7F, 0x40, 0x00, // 1
    0x42, 0x61, 0x51, 0x49, 0x46, // 2
    0x21, 0x41, 0x45, 0x4B, 0x31, // 3
    0x18, 0x14, 0x12, 0x7F, 0x10, // 4
    0x27, 0x45, 0x45, 0x45, 0x39, // 5
    0x3C, 0x4A, 0x49, 0x49, 0x30, // 6
    0x01, 0x71, 0x09, 0x05, 0x03, // 7
    0x36, 0x49, 0x49, 0x49, 0x36, // 8
    0x06, 0x49, 0x49, 0x29, 0x1E, // 9
    0x00, 0x36, 0x36, 0x00, 0x00, // :
    0x00, 0x56, 0x36, 0x00, 0x00, // ;
    0x00, 0x08, 0x14, 0x22, 0x41, // <
    0x14, 0x14, 0x14, 0x14, 0x14, // =
    0x41, 0x22, 0x14, 0x08, 0x00, // >
    0x02, 0x01, 0x51, 0x09, 0x06, // ?
    0x32, 0x49, 0x79, 0x41, 0x3E, // @
    0x7E, 0x11, 0x11, 0x11, 0x7E, // A
    0x7F, 0x49, 0x49, 0x49, 0x36, // B
    0x3E, 0x41, 0x41, 0x41, 0x22, // C
    0x7F, 0x41, 0x41, 0x22, 0x1C, // D
    0x7F, 0x49, 0x49, 0x49, 0x41, // E
    0x7F, 0x09, 0x09, 0x01, 0x01, // F
    0x3E, 0x41, 0x41, 0x51, 0x32, // G
    0x7F, 0x08, 0x08, 0x08, 0x7F, // H
    0x00, 0x41, 0x7F, 0x41, 0x00, // I
    0x20, 0x40, 0x41, 0x3F, 0x01, // J
    0x7F, 0x08, 0x14, 0x22, 0x41, // K
    0x7F, 0x40, 0x40, 0x40, 0x40, // L
    0x7F, 0x02, 0x04, 0x02, 0x7F, // M
    0x7F, 0x04, 0x08, 0x10, 0x7F, // N
    0x3E, 0x41, 0x41, 0x41, 0x3E, // O
    0x7F, 0x09, 0x09, 0x09, 0x06, // P
    0x3E, 0x41, 0x51, 0x21, 0x5E, // Q
    0x7F, 0x09, 0x19, 0x29, 0x46, // R
    0x46, 0x49, 0x49, 0x49, 0x31, // S
    0x01, 0x01, 0x7F, 0x01, 0x01, // T
    0x3F, 0x40, 0x40, 0x40, 0x3F, // U
    0x1F, 0x20, 0x40, 0x20, 0x1F, // V
    0x7F, 0x20, 0x18, 0x20, 0x7F, // W
    0x63, 0x14, 0x08, 0x14, 0x63, // X
    0x03, 0x04, 0x78, 0x04, 0x03, // Y
    0x61, 0x51, 0x49, 0x45, 0x43, // Z
    0x00, 0x00, 0x7F, 0x41, 0x41, // [
    0x02, 0x04, 0x08, 0x10, 0x20, // /
    0x41, 0x41, 0x7F, 0x00, 0x00, // ]
    0x04, 0x02, 0x01, 0x02, 0x04, // ^
    0x40, 0x40, 0x40, 0x40, 0x40, // _
    0x00, 0x01, 0x02, 0x04, 0x00, // `
    0x20, 0x54, 0x54, 0x54, 0x78, // a
    0x7F, 0x48, 0x44, 0x44, 0x38, // b
    0x38, 0x44, 0x44, 0x44, 0x20, // c
    0x38, 0x44, 0x44, 0x48, 0x7F, // d
    0x38, 0x54, 0x54, 0x54, 0x18, // e
    0x08, 0x7E, 0x09, 0x01, 0x02, // f
    0x08, 0x14, 0x54, 0x54, 0x3C, // g
    0x7F, 0x08, 0x04, 0x04, 0x78, // h
    0x00, 0x44, 0x7D, 0x40, 0x00, // i
    0x20, 0x40, 0x44, 0x3D, 0x00, // j
    0x00, 0x7F, 0x10, 0x28, 0x44, // k
    0x00, 0x41, 0x7F, 0x40, 0x00, // l
    0x7C, 0x04, 0x18, 0x04, 0x78, // m
    0x7C, 0x08, 0x04, 0x04, 0x78, // n
    0x38, 0x44, 0x44, 0x44, 0x38, // o
    0x7C, 0x14, 0x14, 0x14, 0x08, // p
    0x08, 0x14, 0x14, 0x18, 0x7C, // q
    0x7C, 0x08, 0x04, 0x04, 0x08, // r
    0x48, 0x54, 0x54, 0x54, 0x20, // s
    0x04, 0x3F, 0x44, 0x40, 0x20, // t
    0x3C, 0x40, 0x40, 0x20, 0x7C, // u
    0x1C, 0x20, 0x40, 0x20, 0x1C, // v
    0x3C, 0x40, 0x30, 0x40, 0x3C, // w
    0x44, 0x28, 0x10, 0x28, 0x44, // x
    0x0C, 0x50, 0x50, 0x50, 0x3C, // y
    0x44, 0x64, 0x54, 0x4C, 0x44, // z
    0x00, 0x08, 0x36, 0x41, 0x00, // {
    0x00, 0x00, 0x7F, 0x00, 0x00, // |
    0x00, 0x41, 0x36, 0x08, 0x00, // }
    0x08, 0x08, 0x2A, 0x1C, 0x08, // ->
    0x08, 0x1C, 0x2A, 0x08, 0x08, // <-
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0x00, //
};

typedef void (*LCDIMG_DRAWER)(const int px, const int py, int on, void *extobj);

/**
 * @brief A simple canvas buffer structure for bmpio.
 */
typedef struct {
    int w;              /**< The image width. */
    int h;              /**< The image height. */
    bmpcol_t *buffer;   /**< The pointer to the color data buffer. */
} canvas_t;

/**
 * @brief Canvas initializer.
 */
#define CANVAS_INIT(P, W, H) \
    do { \
        (P)->w = (W); \
        (P)->h = (H); \
        (P)->buffer = (bmpcol_t *)malloc(sizeof(bmpcol_t) * (W) * (H)); \
        if ((P)->buffer == NULL) { \
            printf("Memory Allocation Error.\n"); \
        } \
    } while (0)

/**
 * @brief Canvas finisher.
 */
#define CANVAS_FINI(P) \
    do { \
        free((P)->buffer); \
    } while (0)

struct LCDIMG {
    LCDPROF prof;           /**< Profile */
    unsigned char *txtbuf;  /**< Text Buffer */
    canvas_t canvas;        /**< Canvas */
    bmpimg_t bmpimg;
    uint8_t font5x8[sizeof(font5x8_master) / sizeof(font5x8_master[0])];
};

static void calc_canvas_size(LCDIMG *p, int *cx, int *cy)
{
    *cx =
        p->prof.chofs_x +
        ((p->prof.pixsiz_x * FONT_X) * p->prof.chcnt_x) +
        ((p->prof.pixgap_x * (FONT_X - 1)) * p->prof.chcnt_x) +
        (p->prof.chgap_x * (p->prof.chcnt_x - 1)) +
        p->prof.chofs_x;
    *cy =
        p->prof.chofs_y +
        ((p->prof.pixsiz_y * FONT_Y) * p->prof.chcnt_y) +
        ((p->prof.pixgap_y * (FONT_Y - 1)) * p->prof.chcnt_y) +
        (p->prof.chgap_y * (p->prof.chcnt_y - 1)) +
        p->prof.chofs_y;
}

/**
 * @brief Pixel writer for bmpimg module.
 *
 * @param x X point.
 * @param y Y point.
 * @param r Red.
 * @param g Green.
 * @param b Blue.
 * @param extobj User external object.
 */
static void canvas_pixel_writer(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj)
{
    LCDIMG *p = (LCDIMG *)extobj;
    canvas_t *canvas = (canvas_t *)&(p->canvas);
    bmpcol_t *buffer = canvas->buffer + (canvas->w * y) + x;
    buffer->r = r;
    buffer->g = g;
    buffer->b = b;
}

/**
 * @brief Pixel reader for bmpimg module.
 *
 * @param x X point.
 * @param y Y point.
 * @param r Red.
 * @param g Green.
 * @param b Blue.
 * @param extobj User external object.
 */
static void canvas_pixel_reader(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj)
{
    LCDIMG *p = (LCDIMG *)extobj;
    canvas_t *canvas = (canvas_t *)&(p->canvas);
    bmpcol_t *buffer = canvas->buffer + (canvas->w * y) + x;
    *r = buffer->r;
    *g = buffer->g;
    *b = buffer->b;
}

#define CLIP_MIN(N, MIN)        (((N) < (MIN)) ? (MIN) : (N))
#define CLIP_MAX(N, MAX)        (((MAX) < (N)) ? (MAX) : (N))
#define CLIPPING(N, MIN, MAX)   CLIP_MAX(CLIP_MIN(N, MIN), MAX)

/**
 * @brief Canvas drawer for lcdimg module.
 *
 * @param px Pixel X point.
 * @param py Pixel Y point.
 * @param on Pixel statement.
 * @param extobj User external object.
 */
static void canvas_drawer(const int px, const int py, int on, void *extobj)
{
    LCDIMG *p = (LCDIMG *)extobj;
    canvas_t *canvas = (canvas_t *)&(p->canvas);
    bmpcol_t *buffer = canvas->buffer + (canvas->w * py) + px;
    buffer->r = on ? p->prof.fgcol.r : (unsigned char)CLIPPING(
            (p->prof.fgcol.r * (100 - p->prof.contrast) / 100.0) + (p->prof.bgcol.r * (p->prof.contrast / 100.0)), 0, 255);
    buffer->g = on ? p->prof.fgcol.g : (unsigned char)CLIPPING(
            (p->prof.fgcol.g * (100 - p->prof.contrast) / 100.0) + (p->prof.bgcol.g * (p->prof.contrast / 100.0)), 0, 255);
    buffer->b = on ? p->prof.fgcol.b : (unsigned char)CLIPPING(
            (p->prof.fgcol.b * (100 - p->prof.contrast) / 100.0) + (p->prof.bgcol.b * (p->prof.contrast / 100.0)), 0, 255);
}

static void lcdfont_read_callback(const int code, const int x, const int y, int on, void *extobj)
{
    LCDIMG *p = (LCDIMG *)extobj;
    const int FNTDATSIZ = sizeof(p->font5x8) / sizeof(p->font5x8[0]);

    if (code < FNTDATSIZ / FONT_X) {
        if (on) {
            p->font5x8[(code * FONT_X) + x] |= (1 << y);
        } else {
            p->font5x8[(code * FONT_X) + x] &= ~(1 << y);
        }
    }
}

LCDIMG *lcdimg_open(const char *profile, const char *fontfile)
{
    const int FNTDATSIZ = sizeof(font5x8_master) / sizeof(font5x8_master[0]);
    int verbose = 0;
    int i;
    int cx, cy;
    LCDIMG *p = (LCDIMG *)malloc(sizeof(LCDIMG));
    if (p == NULL) {
        return NULL;
    }

    /*
     * Setup the LCD profile.
     */
    LCDPROF_INIT(&(p->prof));
    if (lcdprof_read(profile, &(p->prof), verbose) != 0) {
        printf("LCD profile read failed.\n");
        printf("Use default profile.\n");
    }

    /*
     * Setup the internal text buffer.
     */
    p->txtbuf = (unsigned char *)malloc(sizeof(unsigned char) * p->prof.chcnt_x * p->prof.chcnt_y);
    if (p->txtbuf == NULL) {
        free(p);
        return NULL;
    }
    memset(p->txtbuf, ' ', p->prof.chcnt_x * p->prof.chcnt_y);

    /*
     * Setup the internal BMP canvas object.
     */
    calc_canvas_size(p, &cx, &cy);
    CANVAS_INIT(&(p->canvas), cx, cy);
    if (p->canvas.buffer == NULL) {
        free(p->txtbuf);
        free(p);
        return NULL;
    }
    bmpimg_open(&(p->bmpimg), cx, cy, canvas_pixel_writer, p, canvas_pixel_reader, p);

    /*
     * Copy the font data from the master.
     */
    for (i = 0; i < FNTDATSIZ; i++) {
        p->font5x8[i] = font5x8_master[i];
    }

    /*
     * Read the user font data from a file.
     */
    if (lcdfont_read(fontfile, lcdfont_read_callback, p) != 0) {
        printf("LCD font data read failed.\n");
        printf("Use default font.\n");
    }

    return p;
}

int lcdimg_text(LCDIMG *p, const int x, const int y, const unsigned char c)
{
    if (x < 0) {
        return -1;
    }
    if (y < 0) {
        return -1;
    }
    if (p->prof.chcnt_x <= x) {
        return -1;
    }
    if (p->prof.chcnt_y <= y) {
        return -1;
    }
    p->txtbuf[(y * p->prof.chcnt_x) + x] = c;
    return 0;
}

static void draw_char(LCDIMG *p, LCDIMG_DRAWER drawer, void *extobj, const int px, const int py, const unsigned char c)
{
    const int FNTDATSIZ = sizeof(p->font5x8) / sizeof(p->font5x8[0]);
    int x, y;
    int nx, ny;

    for (y = 0; y < FONT_Y; y++) {
        for (x = 0; x < FONT_X; x++) {
            for (ny = 0; ny < p->prof.pixsiz_y; ny++) {
                for (nx = 0; nx < p->prof.pixsiz_x; nx++) {
                    uint8_t pat = 0;
                    if (c < FNTDATSIZ / FONT_X) {
                        pat = p->font5x8[(c * FONT_X) + x];
                    }
                    drawer(
                            px + (p->prof.pixsiz_x * x) + (p->prof.pixgap_x * (x - 1)) + nx,
                            py + (p->prof.pixsiz_y * y) + (p->prof.pixgap_y * (y - 1)) + ny,
                            (pat & (1 << y)),
                            extobj);
                }
            }
        }
    }
}

static int func_fwrite(const void *buf, const unsigned int size, void *extobj)
{
    FILE *fp = (FILE *)extobj;
    return fwrite(buf, size, 1, fp);
}

int lcdimg_write(LCDIMG *p, const char *filename)
{
    FILE *fp;
    int x, y;
    int cx, cy;
    bmpcol_t col;

    /*
     * Setup the canvas.
     */
    calc_canvas_size(p, &cx, &cy);
    BMPIMG_SET_COLOR(col, p->prof.bgcol.r, p->prof.bgcol.g, p->prof.bgcol.b);
    bmpimg_fill_box(&(p->bmpimg), 0, 0, cx - 1, cy - 1, &col);

    /*
     * Draw the text data.
     */
    for (y = 0; y < p->prof.chcnt_y; y++) {
        for (x = 0; x < p->prof.chcnt_x; x++) {
            draw_char(
                    p,
                    canvas_drawer,
                    p,
                    p->prof.chofs_x + ((p->prof.pixsiz_x * FONT_X) * x) + ((p->prof.pixgap_x * (FONT_X - 1)) * x) + (p->prof.chgap_x * x),
                    p->prof.chofs_y + ((p->prof.pixsiz_y * FONT_Y) * y) + ((p->prof.pixgap_y * (FONT_Y - 1)) * y) + (p->prof.chgap_y * y),
                    p->txtbuf[(y * p->prof.chcnt_x) + x]);
        }
    }

    /*
     * Write to the target file.
     */
    fp = fopen(filename, "wb");
    if (fp != NULL) {
        bmpimg_bmp_write(&(p->bmpimg), func_fwrite, fp);
        fclose(fp);
    }

    return 0;
}

int lcdimg_close(LCDIMG *p)
{
    if (p == NULL) {
        return 0;
    }
    bmpimg_close(&(p->bmpimg));
    CANVAS_FINI(&(p->canvas));
    free(p->txtbuf);
    free(p);
    return 0;
}

