/**
 * @file bmplowio.h
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef BMPLOWIO_H
#define BMPLOWIO_H

#include <stdint.h>

/**
 * @brief ビットマップファイルを示すマジックテキスト。
 */
#define BMP_FILE_MAGIC_TEXT                 (('B' << 0) | ('M' << 8))

#define BMP_FILE_GET_TYPE(P)                ((P)->bfType)
#define BMP_FILE_GET_SIZE(P)                ((P)->bfSize)
#define BMP_FILE_GET_RESERVED1(P)           ((P)->bfReserved1)
#define BMP_FILE_GET_RESERVED2(P)           ((P)->bfReserved2)
#define BMP_FILE_GET_OFFBITS(P)             ((P)->bfOffBits)

#define BMP_FILE_SET_TYPE(P, N)             do { (P)->bfType = (N); } while (0)
#define BMP_FILE_SET_SIZE(P, N)             do { (P)->bfSize = (N); } while (0)
#define BMP_FILE_SET_RESERVED1(P, N)        do { (P)->bfReserved1 = (N); } while (0)
#define BMP_FILE_SET_RESERVED2(P, N)        do { (P)->bfReserved2 = (N); } while (0)
#define BMP_FILE_SET_OFFBITS(P, N)          do { (P)->bfOffBits = (N); } while (0)

#define BMP_INFO_GET_SIZE(P)                ((P)->biSize)
#define BMP_INFO_GET_WIDTH(P)               ((P)->biWidth)
#define BMP_INFO_GET_HEIGHT(P)              ((P)->biHeight)
#define BMP_INFO_GET_PLANES(P)              ((P)->biPlanes)
#define BMP_INFO_GET_BIT_COUNT(P)           ((P)->biBitCount)
#define BMP_INFO_GET_COMPRESSION(P)         ((P)->biCompression)
#define BMP_INFO_GET_SIZE_IMAGE(P)          ((P)->biSizeImage)
#define BMP_INFO_GET_X_PELS_PER_METER(P)    ((P)->biXPelsPerMeter)
#define BMP_INFO_GET_Y_PELS_PER_METER(P)    ((P)->biYPelsPerMeter)
#define BMP_INFO_GET_CLR_USED(P)            ((P)->biClrUsed)
#define BMP_INFO_GET_CLR_IMPORTANT(P)       ((P)->biClrImportant)

#define BMP_INFO_SET_SIZE(P, N)             do { (P)->biSize = (N); } while (0)
#define BMP_INFO_SET_WIDTH(P, N)            do { (P)->biWidth = (N); } while (0)
#define BMP_INFO_SET_HEIGHT(P, N)           do { (P)->biHeight = (N); } while (0)
#define BMP_INFO_SET_PLANES(P, N)           do { (P)->biPlanes = (N); } while (0)
#define BMP_INFO_SET_BIT_COUNT(P, N)        do { (P)->biBitCount = (N); } while (0)
#define BMP_INFO_SET_COMPRESSION(P, N)      do { (P)->biCompression = (N); } while (0)
#define BMP_INFO_SET_SIZE_IMAGE(P, N)       do { (P)->biSizeImage = (N); } while (0)
#define BMP_INFO_SET_X_PELS_PER_METER(P, N) do { (P)->biXPelsPerMeter = (N); } while (0)
#define BMP_INFO_SET_Y_PELS_PER_METER(P, N) do { (P)->biYPelsPerMeter = (N); } while (0)
#define BMP_INFO_SET_CLR_USED(P, N)         do { (P)->biClrUsed = (N); } while (0)
#define BMP_INFO_SET_CLR_IMPORTANT(P, N)    do { (P)->biClrImportant = (N); } while (0)

#define BMP_FILE_PRINT(P) \
    fprintf(stderr, "BMP FILE HEADER : (Type=0x%04x, Size=0x%08x, Reserved1=0x%04x, Reserved2=0x%04x, OffBits=0x%08x)\n", \
            BMP_FILE_GET_TYPE(P), \
            BMP_FILE_GET_SIZE(P), \
            BMP_FILE_GET_RESERVED1(P), \
            BMP_FILE_GET_RESERVED2(P), \
            BMP_FILE_GET_OFFBITS(P))

#define BMP_INFO_PRINT(P) \
    fprintf(stderr, "BMP INFO HEADER : (Size=0x%08x, Width=0x%08x, Height=0x%08x, Planes=0x%04x, BitCount=0x%04x, Compression=0x%08x, SizeImage=0x%08x, XPelsPerMeter=0x%08x, YPelsPerMeter=0x%08x, ClrUsed=0x%08x, ClrImportant=0x%08x)\n", \
            BMP_INFO_GET_SIZE(P), \
            BMP_INFO_GET_WIDTH(P), \
            BMP_INFO_GET_HEIGHT(P), \
            BMP_INFO_GET_PLANES(P), \
            BMP_INFO_GET_BIT_COUNT(P), \
            BMP_INFO_GET_COMPRESSION(P), \
            BMP_INFO_GET_SIZE_IMAGE(P), \
            BMP_INFO_GET_X_PELS_PER_METER(P), \
            BMP_INFO_GET_Y_PELS_PER_METER(P), \
            BMP_INFO_GET_CLR_USED(P), \
            BMP_INFO_GET_CLR_IMPORTANT(P))

typedef struct bmp_file {
    uint16_t bfType;
    uint32_t bfSize;
    uint16_t bfReserved1;
    uint16_t bfReserved2;
    uint32_t bfOffBits;
} bmp_file_t;

typedef struct bmp_info {
    uint32_t biSize;
    uint32_t biWidth;
    uint32_t biHeight;
    uint16_t biPlanes;
    uint16_t biBitCount;
    uint32_t biCompression;
    uint32_t biSizeImage;
    uint32_t biXPelsPerMeter;
    uint32_t biYPelsPerMeter;
    uint32_t biClrUsed;
    uint32_t biClrImportant;
} bmp_info_t;

typedef struct bmp_rgbquad {
    uint8_t blue;
    uint8_t green;
    uint8_t red;
    uint8_t reserved;
} bmp_rgbquad_t;

#ifdef __cplusplus
extern "C" {
#endif

int bmplowio_header_init(
    bmp_file_t* filehead,
    bmp_info_t *infohead);

int bmplowio_header_write(
    int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
    void *extobj_func_fwrite,
    bmp_file_t* filehead,
    bmp_info_t *infohead);

int bmplowio_palette_write(
    int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
    void *extobj_func_fwrite,
    bmp_rgbquad_t *rgbquad,
    uint32_t n);

int bmplowio_image_write(
    int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
    void *extobj_func_fwrite,
    bmp_file_t *filehead,
    bmp_info_t *infohead,
    void (*func_pixel_read)(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj_pixel_read),
    void *extobj_pixel_read);

int bmplowio_image_read(
    int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
    void *extobj_func_fread,
    bmp_file_t *filehead,
    bmp_info_t *infohead,
    void (*func_pixel_write)(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj_pixel_write),
    void *extobj_pixel_write);

int bmplowio_header_read(
    int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
    void *extobj_func_fread,
    bmp_file_t *filehead,
    bmp_info_t *infohead);

int bmplowio_palette_read(
    int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
    void *extobj_func_fread,
    bmp_rgbquad_t *rgbquad,
    uint32_t n);

int bmplowio_have_palette(const int biBitCount);

int bmplowio_calc_framebytesize(
        const int biBitCount,
        const int biWidth,
        const int biHeight);

#ifdef __cplusplus
}
#endif

#endif

