/**
 * @file bmplowio.c
 * @author Shinichiro Nakamura
 * @brief 小規模組み込みシステム向けBMP I/Oの実装。
 */

/*
 * ===============================================================
 *  Tiny BMP I/O Module
 *  Version 0.0.1
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "bmplowio.h"

static int fread_uint32_t(
        uint32_t *data,
        int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
        void *extobj_func_fread)
{
    unsigned char temp[4];
    if (!func_fread(temp, 4, extobj_func_fread)) {
        return -1;
    }
    *data =
        ((((uint32_t)temp[0]) <<  0) & 0x000000ff) |
        ((((uint32_t)temp[1]) <<  8) & 0x0000ff00) |
        ((((uint32_t)temp[2]) << 16) & 0x00ff0000) |
        ((((uint32_t)temp[3]) << 24) & 0xff000000);
    return 0;

}

static int fread_uint16_t(
        uint16_t *data,
        int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
        void *extobj_func_fread)
{
    unsigned char temp[2];
    if (!func_fread(temp, 2, extobj_func_fread)) {
        return -1;
    }
    *data =
        ((((uint16_t)temp[0]) << 0) & 0x00ff) |
        ((((uint16_t)temp[1]) << 8) & 0xff00);
    return 0;
}

static int fwrite_uint32_t(
        uint32_t data,
        int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
        void *extobj_func_fwrite)
{
    unsigned char temp[4];
    temp[0] = ((data) >>  0) & 0x000000ff;
    temp[1] = ((data) >>  8) & 0x000000ff;
    temp[2] = ((data) >> 16) & 0x000000ff;
    temp[3] = ((data) >> 24) & 0x000000ff;
    if (!func_fwrite(temp, 4, extobj_func_fwrite)) {
        return -1;
    }
    return 0;
}

static int fwrite_uint16_t(
        uint16_t data,
        int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
        void *extobj_func_fwrite)
{
    unsigned char temp[2];
    temp[0] = ((data) >> 0) & 0x00ff;
    temp[1] = ((data) >> 8) & 0x00ff;
    if (!func_fwrite(temp, 2, extobj_func_fwrite)) {
        return -1;
    }
    return 0;
}

int bmplowio_header_init(bmp_file_t* filehead, bmp_info_t *infohead)
{
    BMP_FILE_SET_TYPE(filehead, 0);
    BMP_FILE_SET_SIZE(filehead, 0);
    BMP_FILE_SET_RESERVED1(filehead, 0);
    BMP_FILE_SET_RESERVED2(filehead, 0);
    BMP_FILE_SET_OFFBITS(filehead, 0);

    BMP_INFO_SET_SIZE(infohead, 0);
    BMP_INFO_SET_WIDTH(infohead, 0);
    BMP_INFO_SET_HEIGHT(infohead, 0);
    BMP_INFO_SET_PLANES(infohead, 0);
    BMP_INFO_SET_BIT_COUNT(infohead, 0);
    BMP_INFO_SET_COMPRESSION(infohead, 0);
    BMP_INFO_SET_SIZE_IMAGE(infohead, 0);
    BMP_INFO_SET_X_PELS_PER_METER(infohead, 0);
    BMP_INFO_SET_Y_PELS_PER_METER(infohead, 0);
    BMP_INFO_SET_CLR_USED(infohead, 0);
    BMP_INFO_SET_CLR_IMPORTANT(infohead, 0);

    return 0;
}

int bmplowio_header_write(
        int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
        void *extobj_func_fwrite,
        bmp_file_t *filehead,
        bmp_info_t *infohead)
{
    fwrite_uint16_t(filehead->bfType, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(filehead->bfSize, func_fwrite, extobj_func_fwrite);
    fwrite_uint16_t(filehead->bfReserved1, func_fwrite, extobj_func_fwrite);
    fwrite_uint16_t(filehead->bfReserved2, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(filehead->bfOffBits, func_fwrite, extobj_func_fwrite);

    fwrite_uint32_t(infohead->biSize, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biWidth, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biHeight, func_fwrite, extobj_func_fwrite);
    fwrite_uint16_t(infohead->biPlanes, func_fwrite, extobj_func_fwrite);
    fwrite_uint16_t(infohead->biBitCount, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biCompression, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biSizeImage, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biXPelsPerMeter, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biYPelsPerMeter, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biClrUsed, func_fwrite, extobj_func_fwrite);
    fwrite_uint32_t(infohead->biClrImportant, func_fwrite, extobj_func_fwrite);

    return 0;
}

int bmplowio_palette_write(
        int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
        void *extobj_func_fwrite,
        bmp_rgbquad_t *rgbquad,
        uint32_t n)
{
    bmp_rgbquad_t *p = rgbquad;
    int i;
    for (i = 0; i < (int)n; i++) {
        func_fwrite(&(p->red), 1, extobj_func_fwrite);
        func_fwrite(&(p->green), 1, extobj_func_fwrite);
        func_fwrite(&(p->blue), 1, extobj_func_fwrite);
        func_fwrite(&(p->reserved), 1, extobj_func_fwrite);
        p++;
    }
    return 0;
}

int bmplowio_image_write(
        int (*func_fwrite)(const void *buf, const uint32_t size, void *extobj_func_fwrite),
        void *extobj_func_fwrite,
        bmp_file_t *filehead,
        bmp_info_t *infohead,
        void (*func_pixel_read)(const int x, const int y, uint8_t *r, uint8_t *g, uint8_t *b, void *extobj_pixel_read),
        void *extobj_pixel_read)
{
    const int w = infohead->biWidth;
    const int h = infohead->biHeight;
    const int pad = (w * infohead->biBitCount + 31) / 32 * 4 - (w * infohead->biBitCount + 7 ) / 8;
    int x, y;
    int i;
    for (y = 0; y < (int)infohead->biHeight; y++) {
        for (x = 0; x < (int)infohead->biWidth; x++) {
            if (infohead->biBitCount == 24) {
                uint8_t a, b, c;
                func_pixel_read(x, h - y - 1, &c, &b, &a, extobj_pixel_read);
                func_fwrite(&a, 1, extobj_func_fwrite);
                func_fwrite(&b, 1, extobj_func_fwrite);
                func_fwrite(&c, 1, extobj_func_fwrite);
            } else {
                /*
                 * @todo 24ビット深度以外は必要になったら対応する。
                 */
            }
        }
        for (i = 0; i < pad; i++) {
            uint8_t a = 0x00;
            func_fwrite(&a, 1, extobj_func_fwrite);
        }
    }
    return 0;
}

int bmplowio_image_read(
        int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
        void *extobj_func_fread,
        bmp_file_t *filehead,
        bmp_info_t *infohead,
        void (*func_pixel_write)(const int x, const int y, const uint8_t r, const uint8_t g, const uint8_t b, void *extobj_pixel_write),
        void *extobj_pixel_write)
{
    const int w = infohead->biWidth;
    const int h = infohead->biHeight;
    const int pad = (w * infohead->biBitCount + 31) / 32 * 4 - (w * infohead->biBitCount + 7 ) / 8;
    int x, y;
    int i;
    for (y = 0; y < (int)infohead->biHeight; y++) {
        for (x = 0; x < (int)infohead->biWidth; x++) {
            if (infohead->biBitCount == 24) {
                uint8_t a, b, c;
                func_fread(&a, 1, extobj_func_fread);
                func_fread(&b, 1, extobj_func_fread);
                func_fread(&c, 1, extobj_func_fread);
                func_pixel_write(x, h - y - 1, c, b, a, extobj_pixel_write);
            } else {
                /*
                 * @todo 24ビット深度以外は必要になったら対応する。
                 */
            }
        }
        for (i = 0; i < pad; i++) {
            uint8_t a;
            func_fread(&a, 1, extobj_func_fread);
        }
    }
    return 0;
}

int bmplowio_header_read(
        int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
        void *extobj_func_fread,
        bmp_file_t *filehead,
        bmp_info_t *infohead)
{
    fread_uint16_t(&filehead->bfType, func_fread, extobj_func_fread);
    fread_uint32_t(&filehead->bfSize, func_fread, extobj_func_fread);
    fread_uint16_t(&filehead->bfReserved1, func_fread, extobj_func_fread);
    fread_uint16_t(&filehead->bfReserved2, func_fread, extobj_func_fread);
    fread_uint32_t(&filehead->bfOffBits, func_fread, extobj_func_fread);

    fread_uint32_t(&infohead->biSize, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biWidth, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biHeight, func_fread, extobj_func_fread);
    fread_uint16_t(&infohead->biPlanes, func_fread, extobj_func_fread);
    fread_uint16_t(&infohead->biBitCount, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biCompression, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biSizeImage, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biXPelsPerMeter, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biYPelsPerMeter, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biClrUsed, func_fread, extobj_func_fread);
    fread_uint32_t(&infohead->biClrImportant, func_fread, extobj_func_fread);

    return 0;
}

int bmplowio_palette_read(
        int (*func_fread)(void *buf, const uint32_t size, void *extobj_func_fread),
        void *extobj_func_fread,
        bmp_rgbquad_t *rgbquad,
        uint32_t n)
{
    bmp_rgbquad_t *p = rgbquad;
    int i;
    for (i = 0; i < (int)n; i++) {
        func_fread(&(p->red), 1, extobj_func_fread);
        func_fread(&(p->green), 1, extobj_func_fread);
        func_fread(&(p->blue), 1, extobj_func_fread);
        func_fread(&(p->reserved), 1, extobj_func_fread);
        p++;
    }
    return 0;
}

int bmplowio_have_palette(const int biBitCount)
{
    if ((biBitCount == 8)
            || (biBitCount == 4)
            || (biBitCount == 2)
            || (biBitCount == 1)) {
        return 1;
    }
    return 0;
}

int bmplowio_calc_framebytesize(
        const int biBitCount,
        const int biWidth,
        const int biHeight)
{
    int bmp_frame_bytesize = -1;

    if (biBitCount == 32) {
        bmp_frame_bytesize = biWidth * 4 * biHeight;
    }

    if (biBitCount == 24) {
        bmp_frame_bytesize =
            ((biWidth * 3 + 3) / 4) * 4 * biHeight;
    }

    if (biBitCount == 8) {
        bmp_frame_bytesize =
            ((biWidth     + 3) / 4) * 4 * biHeight;
    }

    if (biBitCount == 4) {
        bmp_frame_bytesize =
            ((biWidth / 2 + 3) / 4) * 4 * biHeight;
    }

    if (biBitCount == 2) {
        bmp_frame_bytesize =
            ((biWidth / 4 + 3) / 4) * 4 * biHeight;
    }

    if (biBitCount == 1) {
        bmp_frame_bytesize =
            ((biWidth / 8 + 3) / 4) * 4 * biHeight;
    }

    return bmp_frame_bytesize;
}

