/*
 * blancoDb
 * Copyright (C) 2004-2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.db.generator;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;

import org.xml.sax.SAXException;

import sun.nio.cs.Surrogate.Generator;
import blanco.commons.util.BlancoNameAdjuster;
import blanco.db.conf.BlancoDbSetting;
import blanco.db.definition.BlancoDbDefinition;
import blanco.db.definition.QueryCaller;
import blanco.db.definition.QueryField;
import blanco.db.definition.QueryInvoker;
import blanco.db.definition.QueryIterator;
import blanco.db.definition.TableField;
import blanco.db.exception.IntegrityConstraintException;
import blanco.db.exception.NoRowFoundException;
import blanco.db.exception.NoRowModifiedException;
import blanco.db.exception.NotSingleRowException;
import blanco.db.exception.TooManyRowsFoundException;
import blanco.db.exception.TooManyRowsModifiedException;
import blanco.db.expander.query.caller.QueryCallerClass;
import blanco.db.expander.query.invoker.QueryInvokerClass;
import blanco.db.expander.query.iterator.QueryIteratorClass;
import blanco.db.util.BlancoDbObjectStorage;
import blanco.db.util.BlancoDbUtil;
import blanco.ig.expander.Type;
import blanco.ig.generator.ImplementGenerator;
import blanco.ig.generator.RuntimeGenerator;
import blanco.valueobject.BlancoValueObjectXml2JavaClass;
import blanco.valueobject.concretesax.BlancoValueObjectSerializer;

/**
 * @author Yasuo Nakanishi
 */
public class BlancoDbGenerator extends Generator {
    private final BlancoDbObjectStorage storage = new BlancoDbObjectStorage();

    private ImplementGenerator _generator = null;

    public BlancoDbGenerator(final BlancoDbSetting setting) {
        storage.setSetting(setting);
    }

    /**
     * TableGatewayȊO܂B
     * 
     * @param definition
     * @throws IOException
     * @throws SAXException
     * @throws TransformerException
     */
    public void setup(final BlancoDbDefinition definition,
            final String rootPackage, final File outputDirectory)
            throws SAXException, IOException, TransformerException {
        _generator = new ImplementGenerator(storage.getSetting());

        setupQueryIteratorExpander(definition, rootPackage, outputDirectory);
        setupQueryInvokerExpander(definition);
        setupQueryCallerExpander(definition);
    }

    /**
     * TableGatewaŷݍ쐬܂B
     * 
     * @param definition
     * @throws IOException
     * @throws SAXException
     * @throws TransformerConfigurationException
     */
    public void setupTableGateway(final BlancoDbDefinition definition,
            final String rootPackage, final File outputDirectory)
            throws TransformerConfigurationException, SAXException, IOException {
        _generator = new ImplementGenerator(storage.getSetting());
    }

    public void generateRuntime() throws IOException {
        final RuntimeGenerator generator = new RuntimeGenerator(storage
                .getSetting());
        generator.addSourceDirectory("src/main");
        generator.addRuntime(BlancoDbUtil.class);
        generator.addRuntime(IntegrityConstraintException.class);
        generator.addRuntime(NoRowFoundException.class);
        generator.addRuntime(NoRowModifiedException.class);
        generator.addRuntime(NotSingleRowException.class);
        generator.addRuntime(TooManyRowsFoundException.class);
        generator.addRuntime(TooManyRowsModifiedException.class);
        generator.addPackagePair("blanco.db.util", storage.getSetting()
                .getRootNameSpace()
                + ".util");
        generator.addPackagePair("blanco.db.exception", storage.getSetting()
                .getRootNameSpace()
                + ".exception");
        generator.generate();
    }

    private void setupQueryIteratorExpander(
            final BlancoDbDefinition definition, final String rootPackage,
            final File outputDirectory) throws SAXException, IOException,
            TransformerException {
        final Iterator iterator = definition.getQueryIteratorIterator();
        while (iterator.hasNext()) {
            final QueryIterator queryIterator = (QueryIterator) iterator.next();
            createRowObjectClass(queryIterator, rootPackage, outputDirectory);
            _generator.addMain(new QueryIteratorClass(storage, new Type(storage
                    .getSetting().getRootNameSpace()
                    + ".query", queryIterator.getName() + "Iterator"),
                    queryIterator));

        }
    }

    private void setupQueryInvokerExpander(final BlancoDbDefinition definition) {
        final Iterator iterator = definition.getQueryInvokerIterator();
        while (iterator.hasNext()) {
            final QueryInvoker invoker = (QueryInvoker) iterator.next();
            _generator.addMain(new QueryInvokerClass(storage, new Type(storage
                    .getSetting().getRootNameSpace()
                    + ".query", invoker.getName() + "Invoker"), invoker));
        }
    }

    private void setupQueryCallerExpander(BlancoDbDefinition definition)
            throws TransformerConfigurationException, SAXException, IOException {
        final Iterator iterator = definition.getQueryCallerIterator();
        while (iterator.hasNext()) {
            final QueryCaller queryCaller = (QueryCaller) iterator.next();
            _generator
                    .addMain(new QueryCallerClass(storage, new Type(storage
                            .getSetting().getRootNameSpace()
                            + ".query", queryCaller.getName() + "Caller"),
                            queryCaller));
        }
    }

    private void createRowObjectClass(final QueryIterator queryIterator,
            final String rootPackage, final File outputDirectory)
            throws SAXException, IOException, TransformerException {
        final String typeName = queryIterator.getName() + "Row";
        // Query̏ꍇɂ́ANXւ̕ό`{ĂKv܂B
        createRowObjectClass(BlancoNameAdjuster.toClassName(typeName),
                rootPackage + ".row", queryIterator.getFieldIterator(),
                outputDirectory);
    }

    private void createRowObjectClass(final String className,
            final String packageName, final Iterator iterator,
            final File outputDirectory) throws SAXException, IOException,
            TransformerException {
        ArrayList listFieldTypes = new ArrayList();
        for (; iterator.hasNext();) {
            String fieldName = null;
            String fieldType = null;
            Object objField = iterator.next();
            if (objField instanceof TableField) {
                TableField field = (TableField) objField;
                fieldName = field.getName();
                fieldType = field.getJavaType().getFullName();
            } else if (objField instanceof QueryField) {
                QueryField field = (QueryField) objField;
                fieldName = field.getName();
                fieldType = field.getJavaType().getFullName();
            }

            if (fieldType.equals("java.sql.Date")) {
                // java.sql.Date java.util.Dateɓǂݑւ܂B
                fieldType = "java.util.Date";
            }

            listFieldTypes.add(new String[] { fieldName, fieldType });
        }

        OutputStream outStream = null;
        final File fileWorkXml = new File(outputDirectory.getAbsolutePath()
                + "/" + className + ".blancovalueobject");
        try {
            outStream = new BufferedOutputStream(new FileOutputStream(
                    fileWorkXml));
            BlancoValueObjectSerializer serializer = new BlancoValueObjectSerializer(
                    outStream);
            serializer.startDocument();
            serializer.startElementBlanco("0.1.0");
            serializer.characters("\n");
            serializer.characters("  ");
            serializer.startElementTarget("blancoValueObject");
            serializer.characters("\n");
            serializer.characters("    ");
            serializer.startElementBlancovalueobject(className, packageName);
            serializer.characters("\n");
            serializer.characters("      ");
            serializer.startElementFile();
            serializer.characters("\n");
            serializer.characters("        ");
            serializer.startElementComment();
            serializer.characters("'" + className + "'s\sNXB\n");
            serializer.endElementComment();
            serializer.characters("\n");
            serializer.characters("      ");
            serializer.endElementFile();
            serializer.characters("\n");
            serializer.characters("      ");
            serializer.startElementClass();
            serializer.characters("\n");
            serializer.characters("        ");
            serializer.startElementJavadoc();
            serializer.characters("'" + className + "'s\sNXB\n");
            for (int index = 0; index < listFieldTypes.size(); index++) {
                String[] fieldTypes = (String[]) listFieldTypes.get(index);
                String fieldName = fieldTypes[0];
                String fieldType = fieldTypes[1];

                serializer.characters("(" + String.valueOf(index + 1) + ") '"
                        + fieldName + "' ^:" + fieldType + "\n");
            }
            serializer.endElementJavadoc();
            serializer.characters("\n");
            serializer.characters("      ");
            serializer.endElementClass();
            serializer.characters("\n");
            serializer.characters("\n");

            for (int index = 0; index < listFieldTypes.size(); index++) {
                String[] fieldTypes = (String[]) listFieldTypes.get(index);
                String fieldName = fieldTypes[0];
                String fieldType = fieldTypes[1];

                serializer.characters("      ");
                serializer.startElementField(fieldName, fieldType);
                serializer.characters("\n");
                serializer.characters("        ");
                serializer.startElementJavadoc();
                serializer.characters("tB[h[" + fieldName + "]łB");
                serializer.endElementJavadoc();
                serializer.characters("\n");

                serializer.characters("      ");
                serializer.endElementField();
                serializer.characters("\n");
            }

            serializer.characters("    ");
            serializer.endElementBlancovalueobject();
            serializer.characters("\n");
            serializer.characters("  ");
            serializer.endElementTarget();
            serializer.characters("\n");
            serializer.endElementBlanco();
            serializer.endDocument();
            try {
                outStream.flush();
            } catch (IOException e) {
                new SAXException(e);
            }
        } finally {
            if (outStream != null) {
                outStream.close();
            }
        }

        BlancoValueObjectXml2JavaClass xml2javaclass = new BlancoValueObjectXml2JavaClass();
        xml2javaclass.process(fileWorkXml, new File(storage.getSetting()
                .getWorkDirectory()));
    }

    public void generate() throws IOException {
        generateRuntime();
        generateMain();
    }

    public void generateMain() throws IOException {
        _generator.generateMain();
    }

    public void generateTest() throws IOException {
        // ܂B
    }

    public void generateDataStruct() throws IOException {
        // ܂BY鏈݂͑܂B
    }
}