﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CookComputing.XmlRpc;

namespace BTS.Backlog
{
    public class Project
    {
        /// <summary>
        /// ユーザー属性
        /// </summary>
        class AttributeNames
        {
            public const string PROJECT_ID = "projectId";   // プロジェクトID
            public const string ID = "id";                  // プロジェクトID
            public const string NAME = "name";              // プロジェクト名
            public const string KEY = "key";                // プロジェクトキー
            public const string URL = "url";                // プロジェクトURL
        }

        /// <summary>
        /// プロジェクトID
        /// </summary>
        public int Id
        {
            get;
            private set;
        }

        /// <summary>
        /// プロジェクト名
        /// </summary>
        public string Name
        {
            get;
            private set;
        }

        /// <summary>
        /// プロジェクト識別キー
        /// </summary>
        public string Key
        {
            get;
            private set;
        }

        /// <summary>
        /// プロジェクトURL
        /// </summary>
        public string Url
        {
            get;
            private set;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="projectId">取得するプロジェクトのID</param>
        public Project( int projectId )
        {
            Get( projectId );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="projectKey">取得するプロジェクトのキー</param>
        public Project( string projectKey )
        {
            Get( projectKey );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="project">プロジェクト情報</param>
        private Project( XmlRpcStruct project )
        {
            Set( project );
        }

        /// <summary>
        /// プロジェクト情報を設定する
        /// </summary>
        /// <param name="project"></param>
        private void Set( XmlRpcStruct project )
        {
            if ( project.Count == 0 ) {
                throw new Exception( "指定されたプロジェクトが見つかりません" );
            }

            Id = int.Parse( project[AttributeNames.ID].ToString() );
            Name = project[AttributeNames.NAME].ToString();
            Key = project[AttributeNames.KEY].ToString();
            Url = project[AttributeNames.URL].ToString();
        }

        /// <summary>
        /// プロジェクトの取得
        /// </summary>
        /// <param name="projectId">プロジェクトID</param>
        public void Get( int projectId )
        {
            XmlRpcStruct project = Backlog.Rpc.GetProject( projectId );
            Set( project );
        }

        /// <summary>
        /// プロジェクトの取得
        /// </summary>
        /// <param name="projectKey">プロジェクトキー</param>
        public void Get( string projectKey )
        {
            XmlRpcStruct project = Backlog.Rpc.GetProject( projectKey );
            Set( project );
        }

        #region チケットパラメータ
        /// <summary>
        /// 全てのユーザを取得する
        /// </summary>
        /// <param name="project"></param>
        /// <returns></returns>
        public List<User> GetUsers()
        {
            XmlRpcStruct[] result = Backlog.Rpc.GetUsers( Id );
            List<User> list = new List<User>( result.Length );
            foreach ( XmlRpcStruct param in result ) {
                list.Add( new User( param ) );
            }

            return list;
        }

        /// <summary>
        /// すべてのコンポーネントを取得する
        /// </summary>
        /// <param name="project"></param>
        /// <returns></returns>
        public List<Component> GetComponents()
        {
            XmlRpcStruct[] result = Backlog.Rpc.GetComponents( Id );
            List<Component> list = new List<Component>( result.Length );
            foreach ( XmlRpcStruct param in result ) {
                list.Add( new Component( param ) );
            }

            return list;
        }

        /// <summary>
        /// すべてのバージョン/マイルストーンを取得する
        /// </summary>
        /// <param name="project"></param>
        /// <returns></returns>
        public List<Version> GetVersions()
        {
            XmlRpcStruct[] result = Backlog.Rpc.GetVersions( Id );
            List<Version> list = new List<Version>( result.Length );
            foreach ( XmlRpcStruct param in result ) {
                list.Add( new Version( param ) );
            }

            return list;
        }

        /// <summary>
        /// すべての課題種別を取得する
        /// </summary>
        /// <param name="project"></param>
        /// <returns></returns>
        public List<IssueType> GetIssueTypes()
        {
            XmlRpcStruct[] result = Backlog.Rpc.GetIssueTypes( Id );
            List<IssueType> list = new List<IssueType>( result.Length );
            foreach ( XmlRpcStruct param in result ) {
                list.Add( new IssueType( param ) );
            }

            return list;
        }
        #endregion

        #region チケット
        /// <summary>
        /// 課題の作成
        /// </summary>
        /// <param name="issue"></param>
        public Issue CreateIssue( Issue issue )
        {
            issue.Attributes.Set<int>( AttributeNames.PROJECT_ID, Id );
            XmlRpcStruct result = Backlog.Rpc.CreateIssue( issue.Attributes.Value );

            return new Issue( result );
        }

        /// <summary>
        /// 全ての課題を取得
        /// </summary>
        /// <returns></returns>
        public List<Issue> GetIssues()
        {
            return FindIssue( new FindIssueData() );
        }

        /// <summary>
        /// 課題を検索
        /// </summary>
        /// <param name="findIssue"></param>
        /// <returns></returns>
        public List<Issue> FindIssue( FindIssueData findIssue )
        {
            findIssue.ProjetcId = Id;

            XmlRpcStruct[] result = Backlog.Rpc.FindIssue( findIssue.ToXmlRpcStruct() );
            List<Issue> issues = new List<Issue>();
            foreach ( XmlRpcStruct issue in result ) {
                issues.Add( new Issue( issue ) );
            }

            return issues;
        }

        /// <summary>
        /// 全ての課題を取得
        /// </summary>
        /// <returns></returns>
        public int CountIssue()
        {
            return CountIssue( new FindIssueData() );
        }

        /// <summary>
        /// 全ての課題を取得
        /// </summary>
        /// <returns></returns>
        public int CountIssue( FindIssueData findIssue )
        {
            findIssue.ProjetcId = Id;

            return Backlog.Rpc.CountIssue( findIssue.ToXmlRpcStruct() );
        }
        #endregion

        /// <summary>
        /// プロジェクト一覧の取得
        /// </summary>
        /// <returns>プロジェクトのリスト</returns>
        public static List<Project> GetAll()
        {
            XmlRpcStruct[] projects = Backlog.Rpc.GetProjects();
            List<Project> projectList = new List<Project>( projects.Length );
            foreach ( XmlRpcStruct project in projects ) {
                projectList.Add( new Project( project ) );
            }

            return projectList;
        }
    }
}
