﻿using System;
using System.Globalization;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CookComputing.XmlRpc;

namespace BTS.Backlog
{
    public class Issue
    {
        /// <summary>
        /// チケット属性
        /// </summary>
        public class AttributeNames
        {
            public const string PROJECT_ID = "projectId";       // プロジェクトID
            public const string ID = "id";                      // 課題ID
            public const string KEY = "key";                    // 課題キー
            public const string SUMMARY = "summary";            // 概要
            public const string DESCRIPTION = "description";    // 説明
            public const string URL = "url";                    // 課題のURL
            public const string CREATED_USER = "created_user";  // 登録者
            public const string ASSIGNER = "assigner";          // 担当者
            public const string DUE_DATE = "due_date";          // 期限日
            public const string START_DATE = "start_date";      // 開始日
            public const string CREATED_ON = "created_on";      // 登録日時
            public const string UPDATED_ON = "updated_on";      // 最終更新日時
            public const string COMPONENT_ID = "componentId";   // コンポーネントID（登録用）
            public const string COMPONENT = "component";        // コンポーネント名称（登録用）
            public const string COMPONENTS = "components";      // コンポーネント
            public const string VERSION_ID = "versionId";       // バージョンID（登録用）
            public const string VERSION = "version";            // バージョン名称（登録用）
            public const string VERSIONS = "versions";          // バージョン
            public const string MILESTONE_ID = "milestoneId";   // マイルストーンID（登録用）
            public const string MILESTONE = "milestone";        // マイルストーン名称（登録用）
            public const string MILESTONES = "milestones";      // マイルストーン
            public const string STATUS = "status";              // 状態
            public const string RESOLUTION_ID = "resolutionId"; // 完了理由（登録用）
            public const string RESOLUTION = "resolution";      // 完了理由
            public const string PRIORITY_ID = "priorityId";     // 優先度（登録用）
            public const string PRIORITY = "priority";          // 優先度
            public const string ISSUE_TYPE_ID = "issueTypeId";  // 稼働種別ID（登録用）
            public const string ISSUE_TYPE = "issueType";       // 稼働種別（登録用）
            public const string COMMENT = "comment";            // コメント
        }

        /// <summary>
        /// DateTime のフォーマット（年月日）
        /// </summary>
        private static string DateFormat = "yyyyMMdd";

        /// <summary>
        /// DateTime のフォーマット（年月日自分秒）
        /// </summary>
        private static string DateTimeFormat = "yyyyMMddHHmmss";

        /// <summary>
        /// 属性
        /// </summary>
        public XmlRpcAttributes Attributes
        {
            get;
            private set;
        }

        #region プロパティ
        /// <summary>
        /// 課題ID
        /// </summary>
        public int Id
        {
            get
            {
                return Attributes.Get<int>( AttributeNames.ID );
            }

            set
            {
                Attributes.Set<int>( AttributeNames.ID, value );
            }
        }

        /// <summary>
        /// 課題キー
        /// </summary>
        public string Key
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.KEY );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.KEY, value );
            }
        }

        /// <summary>
        /// 概要
        /// </summary>
        public string Summary
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.SUMMARY );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.SUMMARY, value );
            }
        }

        /// <summary>
        /// 概要
        /// </summary>
        public string Description
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.DESCRIPTION );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.DESCRIPTION, value );
            }
        }

        /// <summary>
        /// URL
        /// </summary>
        public string Url
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.URL );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.URL, value );
            }
        }

        /// <summary>
        /// DateTime のパース
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        private DateTime ParseDateTime( string key, string format )
        {
            string str = Attributes.Get<string>( key );
            if ( string.IsNullOrEmpty( str ) ) {
                return new DateTime();
            }
            else {
                return DateTime.ParseExact( str, format, DateTimeFormatInfo.InvariantInfo );
            }
        }

        /// <summary>
        /// User のパース
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        private User ParseUser( string key )
        {
            if ( Attributes.IsContains( key ) ) {
                return new User( Attributes.Get<XmlRpcStruct>( key ) );
            }
            else {
                return new User();
            }
        }

        /// <summary>
        /// 開始日
        /// </summary>
        public DateTime StartDate
        {
            get
            {
                return ParseDateTime( AttributeNames.START_DATE, DateFormat );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.START_DATE, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 終了日
        /// </summary>
        public DateTime DueDate
        {
            get
            {
                return ParseDateTime( AttributeNames.DUE_DATE, DateFormat );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.DUE_DATE, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 作成日時
        /// </summary>
        public DateTime CreateOn
        {
            get
            {
                return ParseDateTime( AttributeNames.CREATED_ON, DateTimeFormat );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.CREATED_ON, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 最終更新日時
        /// </summary>
        public DateTime UpdatedOn
        {
            get
            {
                return ParseDateTime( AttributeNames.UPDATED_ON, DateTimeFormat );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.UPDATED_ON, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 報告者
        /// </summary>
        public User CreatedUser
        {
            get
            {
                return ParseUser( AttributeNames.CREATED_USER );
            }

            set
            {
                Attributes.Set<XmlRpcStruct>( AttributeNames.CREATED_USER, value.ToXmlRpcStruct() );
            }
        }

        /// <summary>
        /// 担当者
        /// </summary>
        public User Assigner
        {
            get
            {
                return ParseUser( AttributeNames.ASSIGNER );
            }

            set
            {
                Attributes.Set<XmlRpcStruct>( AttributeNames.ASSIGNER, value.ToXmlRpcStruct() );
            }
        }

        /// <summary>
        /// 解決方法
        /// </summary>
        public int ResolutionId
        {
            get
            {
                return Attributes.Get<int>( AttributeNames.RESOLUTION_ID );
            }

            set
            {
                Attributes.Set<int>( AttributeNames.RESOLUTION_ID, value );
            }
        }

        /// <summary>
        /// 解決方法
        /// </summary>
        public Resolution Resolution
        {
            get
            {
                if ( Attributes.IsContains( AttributeNames.RESOLUTION ) ) {
                    return new Resolution( Attributes.Get<XmlRpcStruct>( AttributeNames.RESOLUTION ) );
                }
                else {
                    return new Resolution();
                }
            }

            set
            {
                Attributes.Set<XmlRpcStruct>( AttributeNames.RESOLUTION, value.ToXmlRpcStruct() );
            }
        }

        /// <summary>
        /// 状態
        /// </summary>
        public StatusType Status
        {
            get
            {
                if ( Attributes.IsContains( AttributeNames.STATUS ) ) {
                    return new StatusType( Attributes.Get<XmlRpcStruct>( AttributeNames.STATUS ) );
                }
                else {
                    return new StatusType();
                }
            }

            set
            {
                Attributes.Set<XmlRpcStruct>( AttributeNames.STATUS, value.ToXmlRpcStruct() );
            }
        }

        /// <summary>
        /// 優先度
        /// </summary>
        public string Priority
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.PRIORITY );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.PRIORITY, value );
            }
        }

        /// <summary>
        /// カテゴリー
        /// </summary>
        public string[] Component
        {
            get
            {
                return Attributes.Get<string[]>( AttributeNames.COMPONENT );
            }

            set
            {
                Attributes.Set<string[]>( AttributeNames.COMPONENT, value );
            }
        }

        /// <summary>
        /// カテゴリー
        /// </summary>
        public List<Component> Components
        {
            get
            {
                if ( Attributes.IsContains( AttributeNames.COMPONENTS ) ) {
                    List<Component> components = new List<Component>();
                    foreach ( object version in Attributes.Get<object[]>( AttributeNames.COMPONENTS ) ) {
                        components.Add( new Component( (XmlRpcStruct)version ) );
                    }

                    return components;
                }
                else {
                    return new List<Component>();
                }
            }
        }

        /// <summary>
        /// バージョンのパース
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public List<Version> ParseVersions( string key )
        {
            if ( Attributes.IsContains( key ) ) {
                List<Version> versions = new List<Version>();
                foreach ( object version in Attributes.Get<object[]>( key ) ) {
                    versions.Add( new Version( (XmlRpcStruct)version ) );
                }

                return versions;
            }
            else {
                return new List<Version>();
            }
        }

        /// <summary>
        /// バージョン
        /// </summary>
        public string[] Version
        {
            get
            {
                return Attributes.Get<string[]>( AttributeNames.VERSION );
            }

            set
            {
                Attributes.Set<string[]>( AttributeNames.VERSION, value );
            }
        }

        /// <summary>
        /// バージョン
        /// </summary>
        public List<Version> Versions
        {
            get
            {
                return ParseVersions( AttributeNames.VERSIONS );
            }
        }

        /// <summary>
        /// マイルストーン
        /// </summary>
        public string[] Milestone
        {
            get
            {
                return Attributes.Get<string[]>( AttributeNames.MILESTONE );
            }

            set
            {
                Attributes.Set<string[]>( AttributeNames.MILESTONE, value );
            }
        }

        /// <summary>
        /// マイルストーン
        /// </summary>
        public List<Version> Milestones
        {
            get
            {
                return ParseVersions( AttributeNames.MILESTONES );
            }
        }
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Issue()
        {
            Attributes = new XmlRpcAttributes();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="param"></param>
        public Issue( XmlRpcStruct param )
            : this()
        {
            Set( param );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="key"></param>
        public Issue( string key )
        {
            Set( Backlog.Rpc.GetIssue( key ) );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="id"></param>
        public Issue( int id )
        {
            Set( Backlog.Rpc.GetIssue( id ) );
        }

        /// <summary>
        /// XmlRpcStruct の設定
        /// </summary>
        /// <param name="param"></param>
        private void Set( XmlRpcStruct param )
        {
            Attributes = new XmlRpcAttributes( param );

            // 本来ないはずのものが来るようなので削除する
            Attributes.Value.Remove( AttributeNames.MILESTONE );
            Attributes.Value.Remove( AttributeNames.VERSION );
            Attributes.Value.Remove( AttributeNames.COMPONENT );

            // フォーマットが違うので直す
            if ( Attributes.IsContains( AttributeNames.START_DATE ) ) {
                string time = Attributes.Get<string>( AttributeNames.START_DATE );
                if ( !string.IsNullOrEmpty( time ) ) {
                    Attributes.Set<string>( AttributeNames.START_DATE, time.Substring( 0, DateFormat.Length ) );
                }
            }

            if ( Attributes.IsContains( AttributeNames.DUE_DATE ) ) {
                string time = Attributes.Get<string>( AttributeNames.DUE_DATE );
                if ( !string.IsNullOrEmpty( time ) ) {
                    Attributes.Set<string>( AttributeNames.DUE_DATE, time.Substring( 0, DateFormat.Length ) );
                }
            }
        }

        /// <summary>
        /// 課題の取得
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public void Get( string key )
        {
            Set( Backlog.Rpc.GetIssue( key ) );
        }

        /// <summary>
        /// 課題の取得
        /// </summary>
        /// <param id="key"></param>
        /// <returns></returns>
        public void Get( int id )
        {
            Set( Backlog.Rpc.GetIssue( id ) );
        }

        /// <summary>
        /// 課題の状態を変更する
        /// </summary>
        /// <param name="status"></param>
        public void SwitchStatus( Status status, string comment )
        {
            status.Key = Key;
            status.Comment = comment;
            Attributes.Value = Backlog.Rpc.SwitchStatus( status.ToXmlRpcStruct() );
        }

        /// <summary>
        /// 課題の更新
        /// </summary>
        /// <param name="update"></param>
        /// <returns></returns>
        public void Update( string comment )
        {
            Attributes.Value.Add( AttributeNames.COMMENT, comment );
            Set( Backlog.Rpc.UpdateIssue( Attributes.Value ) );
        }

        /// <summary>
        /// 課題のコメントを取得
        /// </summary>
        /// <returns></returns>
        public List<Comment> GetComment()
        {
            XmlRpcStruct[] result = Backlog.Rpc.GetComments( Id );

            List<Comment> comments = new List<Comment>( result.Length );
            foreach ( XmlRpcStruct comment in result ) {
                comments.Add( new Comment( comment ) );
            }

            return comments;
        }

        /// <summary>
        /// 等価チェック
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals( object obj )
        {
            Issue rhs = obj as Issue;
            if ( rhs == null ) {
                return false;
            }

            // 属性に違いがあれば異なるデータ
            foreach ( string key in Attributes.Value.Keys ) {
                if ( !this.Attributes.Value[key].Equals( rhs.Attributes.Value[key] ) ) {
                    return false;
                }
            }

            // 全て一致で同一データ
            return true;
        }
    }
}
