package cerot.tools.taskmine.converter;

import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cerot.tools.redmine.timelog.entity.TimeEntry;
import cerot.tools.taskcoach.entity.Effort;
import cerot.tools.taskcoach.entity.Task;
import cerot.tools.taskcoach.entity.Tasks;
import cerot.tools.taskcoach.util.TaskUtils;

/**
 * TaskCoachのモデルをRedmineの経過時間モデルに変換します。
 * @author cero-t
 * 
 */
public final class TaskConverter {
	/** チケット番号を含むタイトルの正規表現 */
	private static final Pattern PATTERN_SUBJECT = Pattern.compile("#(\\d+).*");

	/** チケット番号を含む本文の正規表現 */
	private static final Pattern PATTERN_DESCRIPTION = Pattern
			.compile(".*refs #(\\d+).*");

	/** 日付のフォーマット */
	private static final String FORMAT_SPENTON = "yyyy-MM-dd";

	/** 経過時間のフォーマット */
	private static final String FORMAT_HOURS = "{0,number,0.00}";

	/**
	 * タスクのルートモデルを経過時間のリストに変換します。
	 * @param tasks タスクのルートモデル
	 * @return 経過時間のリスト
	 */
	public static List<TimeEntry> toTimeEntry(Tasks tasks) {
		List<TimeEntry> result = new ArrayList<TimeEntry>();
		List<Task> extractTask = TaskUtils.extractTask(tasks);

		for (Task task : extractTask) {
			Integer issueId = extractIssueId(task);

			Map<String, Long> durationMap = createDurationMap(task);
			for (Map.Entry<String, Long> durationEntry : durationMap.entrySet()) {
				TimeEntry entry = new TimeEntry();
				entry.setIssueId(issueId);
				entry.setSpentOn(durationEntry.getKey());

				String hours = toNormalHours(durationEntry.getValue()
						.longValue());
				entry.setHours(hours);

				// TODO: アクティビティIDの指定方法の検討
				entry.setActivityId(Integer.valueOf(6));

				result.add(entry);
			}
		}

		return result;
	}

	/**
	 * ミリ秒を、xx.xx(h)の形式に変換する。
	 * @param time ミリ秒で示した時間
	 * @return xx.xx形式の時間
	 */
	protected static String toNormalHours(long time) {
		long hours = time / (60 * 60 * 1000);
		long minutes = (time / (60 * 1000)) % 60;

		long quarter = minutes / 15;
		// 15分単位は7捨8入
		if (minutes % 15 > 7) {
			quarter++;

			if (quarter == 4) {
				quarter = 0;
				hours++;
			}
		}

		double total = hours + (double) quarter * 0.25;

		String result = MessageFormat.format(FORMAT_HOURS, Double
				.valueOf(total));
		return result;
	}

	/**
	 * タスクの日付ごとの経過時間のマップを作成します。
	 * @param task タスク
	 * @return 日付ごとの経過時間
	 */
	protected static Map<String, Long> createDurationMap(Task task) {
		Map<String, Long> durationMap = new HashMap<String, Long>();

		if (task.effortList == null) {
			return durationMap;
		}

		for (Effort effort : task.effortList) {
			if (effort.start == null || effort.stop == null) {
				continue;
			}

			long duration = effort.stop.getTime() - effort.start.getTime();
			String spentOn = new SimpleDateFormat(FORMAT_SPENTON)
					.format(effort.start);

			Long sum = durationMap.get(spentOn);
			if (sum == null) {
				sum = Long.valueOf(duration);
			} else {
				sum = Long.valueOf(sum.longValue() + duration);
			}

			durationMap.put(spentOn, sum);
		}

		return durationMap;
	}

	/**
	 * タスクからチケット番号を取得します。
	 * @param task タスク
	 * @return チケット番号
	 */
	public static Integer extractIssueId(Task task) {
		String id = findFirst(task.subject, PATTERN_SUBJECT);

		if (id == null) {
			id = findFirst(task.description, PATTERN_DESCRIPTION);
		}

		if (id == null) {
			return null;
		}

		return Integer.valueOf(id);
	}

	/**
	 * 正規表現パターンに一致した値を返却します。
	 * @param target 対象の文字列
	 * @param pattern 正規表現パターン
	 * @return 一致部分の文字列。一致しなかった場合にはnullを返却します。
	 */
	private static String findFirst(String target, Pattern pattern) {
		if (target == null) {
			return null;
		}

		Matcher matcher = pattern.matcher(target);
		if (matcher.matches() == false) {
			return null;
		}

		return matcher.group(1);
	}

	/**
	 * プライベートコンストラクタ。呼び出し禁止。
	 */
	private TaskConverter() {
		// Do Nothing.
	}
}
