package cerot.tools.redmine.timelog.accessor;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;

import cerot.tools.redmine.timelog.entity.TimeEntry;

/**
 * Redmineへのアクセッサクラス。
 * @author cero-t
 * 
 */
public class RedmineAccessor {
	/** RedmineのURL */
	private String url;

	/** Redmineのユーザ名 */
	private String username;

	/** Redmineのパスワード */
	private String password;

	/** HTTPクライアント */
	private DefaultHttpClient client = new DefaultHttpClient();

	/**
	 * コンストラクタ。プロパティファイルを読み込みます。
	 * @throws IOException プロパティファイルの読み込みに失敗した場合
	 */
	public RedmineAccessor() throws IOException {
		Properties properties = new Properties();
		InputStream in = RedmineAccessor.class
				.getResourceAsStream("/redmine.properties");
		try {
			properties.load(in);
		} finally {
			if (in != null) {
				in.close();
			}
		}

		String urlProp = properties.getProperty("redmine.url");
		if (urlProp.endsWith("/")) {
			urlProp = urlProp.substring(0, urlProp.length() - 1);
		}
		this.url = urlProp;
		this.username = properties.getProperty("redmine.username");
		this.password = properties.getProperty("redmine.password");
	}

	/**
	 * 経過時間を記録します。
	 * @param timeEntryList 経過時間のリスト
	 * @throws IOException 通信に失敗した場合
	 */
	public synchronized void trackTime(List<TimeEntry> timeEntryList)
			throws IOException {
		boolean loginSuccess = login();
		if (loginSuccess == false) {
			System.out.println("ログインに失敗しました。");
			return;
		}

		for (TimeEntry timeEntry : timeEntryList) {
			trackSingleTime(timeEntry);
		}
	}

	/**
	 * Redmineへのログインを行います。
	 * @return ログインに成功した場合はtrue、失敗した場合はfalseを返します。
	 * @throws IOException 通信に失敗した場合
	 */
	private boolean login() throws IOException {
		HttpPost post = new HttpPost(this.url + "/login");

		List<NameValuePair> nvps = new ArrayList<NameValuePair>();
		nvps.add(new BasicNameValuePair("username", this.username));
		nvps.add(new BasicNameValuePair("password", this.password));
		post.setEntity(new UrlEncodedFormEntity(nvps, HTTP.UTF_8));

		Boolean result = client.execute(post, new LoginResponseHandler());
		return result.booleanValue();
	}

	/**
	 * 経過時間を記録します。
	 * @param timeEntry 経過時間
	 * @throws IOException 通信に失敗した場合
	 */
	private void trackSingleTime(TimeEntry timeEntry) throws IOException {
		HttpGet get = new HttpGet(this.url + "/timelog/edit?issue_id="
				+ timeEntry.getIssueId().toString());
		String path = client.execute(get, new TimelogFormResponseHandler());

		HttpPost post = new HttpPost(this.url + path);

		List<NameValuePair> nvps = new ArrayList<NameValuePair>(5);
		nvps.add(new BasicNameValuePair("time_entry[issue_id]", timeEntry
				.getIssueId().toString()));
		nvps.add(new BasicNameValuePair("time_entry[spent_on]", timeEntry
				.getSpentOn()));
		nvps.add(new BasicNameValuePair("time_entry[hours]", timeEntry
				.getHours()));
		nvps.add(new BasicNameValuePair("time_entry[comments]", timeEntry
				.getComments()));
		nvps.add(new BasicNameValuePair("time_entry[activity_id]", timeEntry
				.getActivityId().toString()));
		post.setEntity(new UrlEncodedFormEntity(nvps, HTTP.UTF_8));

		HttpResponse response = client.execute(post);
		response.getEntity().consumeContent();
	}

	/**
	 * HTTPクライアントを設定します。
	 * @param client HTTPクライアント
	 */
	public synchronized void setClient(DefaultHttpClient client) {
		this.client = client;
	}
}
