package cerot.blight.btrace;

import java.util.ArrayList;
import java.util.List;

import com.sun.btrace.comm.MessageCommand;

/**
 * BTraceのメッセージコマンドを変換するコンバータクラス。
 * @author cero-t
 * 
 */
public final class BTraceConverter {
	/** メソッド呼び出しを示す文字列 */
	public static final String CALL = "CALL:";

	/** メソッドのリターンを示す文字列 */
	public static final String RETURN = "RETURN:";

	/** 時間とメッセージのセパレータ */
	public static final String SEPARATOR_TIME = " - ";

	/** クラスとメソッドのセパレータ */
	public static final String SEPARATOR = "#";

	/**
	 * メッセージコマンドのリストを、メソッド表現のリストに変換します。
	 * @param commandList メッセージコマンドのリスト
	 * @return メソッド表現のリスト
	 */
	public static List<MethodExpression> toMethodExpressionList(
			List<MessageCommand> commandList) {
		List<MethodExpression> result = new ArrayList<MethodExpression>(
				commandList.size());
		for (MessageCommand command : commandList) {
			MethodExpression methodCall = toMethodExpression(command);
			result.add(methodCall);
		}

		return result;
	}

	/**
	 * メッセージコマンドをメソッド表現に変換します。
	 * @param command メソッドを示すメッセージコマンド
	 * @return メソッド表現
	 */
	public static MethodExpression toMethodExpression(MessageCommand command) {
		String message = command.getMessage();
		message = message.replaceAll("\n$", "");
		message = message.replaceAll("\r$", "");

		return toMethodExpression(message);
	}

	/**
	 * メソッド表現を作成します。
	 * @param message メソッドを示す文字列
	 * @return メソッド表現
	 */
	public static MethodExpression toMethodExpression(String message) {
		int index = message.indexOf(SEPARATOR_TIME);
		if (index < 0) {
			throw new IllegalArgumentException("Message has no timestamp.");
		}

		Long timestamp = Long.valueOf(message.substring(0, index));
		message = message.substring(index + SEPARATOR_TIME.length());

		MethodExpression result = null;
		if (message.startsWith(CALL)) {
			message = message.substring(CALL.length());
			result = new MethodCall();
		} else if (message.startsWith(RETURN)) {
			message = message.substring(RETURN.length());
			result = new MethodReturn();
		} else {
			throw new IllegalArgumentException(
					"Message is not method call or return.");
		}

		index = message.indexOf(SEPARATOR);
		String className = message.substring(0, index);
		String methodName = message.substring(index + SEPARATOR.length());

		result.setTimestamp(timestamp);
		result.setClassName(className);
		result.setMethodName(methodName);

		return result;
	}

	/**
	 * プライベートコンストラクタ。呼び出し禁止。
	 */
	private BTraceConverter() {
		// Do Nothing.
	}
}
