/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.value;

import java.math.BigInteger;
import java.util.Collection;
import java.util.regex.Pattern;

import net.morilib.awk.AwkArithmeticException;
import net.morilib.awk.AwkCastException;
import net.morilib.awk.namespace.AwkNamespace;

public abstract class AwkValue {

	private static final BigInteger MINB =
		BigInteger.valueOf(Integer.MIN_VALUE);
	private static final BigInteger MAXB =
		BigInteger.valueOf(Integer.MAX_VALUE);

	/**
	 * 
	 */
	public static final AwkValue TRUE = AwkInteger.ONE;

	/**
	 * 
	 */
	public static final AwkValue FALSE = AwkInteger.ZERO;

	/**
	 * 
	 */
	public static final AwkValue IOERROR = AwkInteger.valueOf(-1);

	/**
	 * 
	 * @return
	 */
	public abstract BigInteger toInteger();

	/**
	 * 
	 * @return
	 */
	public int toIntExact() {
		if(!isIntValue()) {
			throw new AwkArithmeticException(
					"the integer is not small");
		}
		return toInteger().intValue();
	}

	/**
	 * 
	 * @return
	 */
	public abstract double toFloat();

	/**
	 * 
	 * @return
	 */
	public abstract String toString(AwkNamespace ns);

	/**
	 * 
	 * @return
	 */
	public String toString() {
		return toString(null);
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean toBoolean();

	/**
	 * 
	 * @return
	 */
	public abstract Number toReal();

	/**
	 * 
	 * @return
	 */
	public abstract Pattern toRegex();

	/**
	 * 
	 * @return
	 */
	public abstract Object toObject();

	/**
	 * 
	 * @return
	 */
	public AwkValue referArray(String v) {
		throw new AwkCastException();
	}

	/**
	 * 
	 * @return
	 */
	public AwkValue putArray(String k, AwkValue v) {
		throw new AwkCastException();
	}

	/**
	 * 
	 * @param k
	 * @return
	 */
	public boolean contains(String k) {
		throw new AwkCastException();
	}

	/**
	 * 
	 * @return
	 */
	public AwkValue deleteArray(String v) {
		throw new AwkCastException();
	}

	/**
	 * 
	 * @return
	 */
	public abstract Collection<AwkValue> values();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isInteger();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFloat();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isReal();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isIntegerValue();

	/**
	 * 
	 * @return
	 */
	public boolean isIntValue() {
		return (isIntegerValue() &&
				toInteger().compareTo(MINB) > 0 &&
				toInteger().compareTo(MAXB) < 0);
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFloatValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isRealValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isPositiveValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isZeroValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isNegativeValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isString();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isRegex();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFunction();

	/**
	 * 
	 * @return
	 */
	public AwkNamespace getNamespace() {
		return null;
	}

	/**
	 * 
	 * @param v
	 * @param ns
	 * @return
	 */
	public abstract int compareTo(AwkValue v, AwkNamespace ns);

}
