/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Map;

import net.morilib.awk.builtin.AwkBuiltInLoader;
import net.morilib.awk.code.AwkProgram;
import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.namespace.AwkRootNamespace;
import net.morilib.awk.parser.AwkLexer;
import net.morilib.awk.parser.AwkParser;
import net.morilib.awk.value.AwkArray;
import net.morilib.awk.value.AwkInteger;
import net.morilib.awk.value.AwkString;
import net.morilib.awk.value.AwkValue;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/03/09
 */
public final class Awk {

	/**
	 * 
	 * @param source
	 * @param stdin
	 * @param stdout
	 * @param stderr
	 * @param args
	 * @return
	 * @throws IOException
	 */
	public static AwkValue execute(Reader source, String filename,
			Reader stdin, Writer stdout, Writer stderr,
			Map<Object, Object> vars,
			String... args) throws IOException {
		BufferedReader rd = new BufferedReader(stdin);
		AwkNamespace r = new AwkRootNamespace();
		AwkProgram p;
		AwkFiles f;
		AwkLexer l;
		String s;

		AwkBuiltInLoader.load(r);
		r.assign("FILENAME", AwkString.valueOf(filename));
		r.assign("ARGC", AwkInteger.valueOf(args.length));
		r.assign("ARGV", new AwkArray(0, args));
		if(vars != null) {
			for(Map.Entry<Object, Object> o : vars.entrySet()) {
				r.assign(o.getKey().toString(),
						AwkString.valueOf(o.getValue().toString()));
			}
		}

		l = new AwkLexer(source);
		p = AwkParser.parse(r, l);
		f = new AwkFiles(rd, stdout, stderr);
		p.executeBegin(r, f);
		while((s = rd.readLine()) != null) {
			try {
				p.execute(r, f, s);
			} catch(AwkNextException e) {
				// ignore
			} catch(AwkExitException e) {
				p.executeEnd(r, f);
				return e.getValue();
			}
		}
		p.executeEnd(r, f);
		return AwkValue.TRUE;
	}

	/**
	 * 
	 * @param source
	 * @param inputFile
	 * @param outputFile
	 * @param stderr
	 * @param inputEncoding
	 * @param outputEncoding
	 * @param args
	 * @return
	 * @throws IOException
	 */
	public static AwkValue execute(Reader source,
			File inputFile,  File outputFile, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		Writer stdout = null;
		Reader stdin = null;

		try {
			if(inputEncoding != null) {
				stdin = new InputStreamReader(
						new FileInputStream(inputFile),
						inputEncoding);
			} else {
				stdin = new InputStreamReader(
						new FileInputStream(inputFile));
			}

			if(outputEncoding != null) {
				stdout = new OutputStreamWriter(
						new FileOutputStream(outputFile),
						outputEncoding);
			} else {
				stdout = new OutputStreamWriter(
						new FileOutputStream(outputFile));
			}
			return execute(source, inputFile.toString(),
					stdin, stdout, stderr, vars, args);
		} finally {
			if(stdin  != null)  stdin.close();
			if(stdout != null) {
				stdout.flush();
				stdout.close();
			}
		}
	}

	/**
	 * 
	 * @param source
	 * @param inputFile
	 * @param outputFile
	 * @param stderr
	 * @param inputEncoding
	 * @param outputEncoding
	 * @param args
	 * @return
	 * @throws IOException
	 */
	public static AwkValue execute(String source,
			File inputFile, File outputFile, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return execute(srd, inputFile, outputFile, stderr,
				inputEncoding, outputEncoding, vars, args);
	}

	/**
	 * 
	 * @param source
	 * @param inputFile
	 * @param outputFile
	 * @param stderr
	 * @param inputEncoding
	 * @param outputEncoding
	 * @param args
	 * @return
	 * @throws IOException
	 */
	public static AwkValue overwrite(Reader source,
			File file, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		File tf = File.createTempFile("awkium", ".tmp");
		BufferedOutputStream ous = null;
		BufferedInputStream ins = null;
		byte[] b = new byte[4096];
		Writer stdout = null;
		Reader stdin = null;
		AwkValue v;
		int l;

		try {
			if(inputEncoding != null) {
				stdin = new InputStreamReader(
						new FileInputStream(file), inputEncoding);
			} else {
				stdin = new InputStreamReader(
						new FileInputStream(file));
			}

			if(outputEncoding != null) {
				stdout = new OutputStreamWriter(
						new FileOutputStream(tf), outputEncoding);
			} else {
				stdout = new OutputStreamWriter(
						new FileOutputStream(tf));
			}
			v = execute(source, file.toString(), stdin,
					stdout, stderr, vars, args);
		} finally {
			if(stdin  != null)  stdin.close();
			if(stdout != null) {
				stdout.flush();
				stdout.close();
			}
		}

		try {
			ins = new BufferedInputStream(new FileInputStream(tf));
			ous = new BufferedOutputStream(
					new FileOutputStream(file));
			while((l = ins.read(b)) >= 0)  ous.write(b, 0, l);
			return v;
		} finally {
			if(ins != null)  ins.close();
			if(ous != null) {
				ous.flush();
				ous.close();
			}
			tf.delete();
		}
	}

	/**
	 * 
	 * @param source
	 * @param file
	 * @param stderr
	 * @param inputEncoding
	 * @param outputEncoding
	 * @param args
	 * @return
	 * @throws IOException
	 */
	public static AwkValue overwrite(String source,
			File file, Writer stderr,
			Charset inputEncoding, Charset outputEncoding,
			Map<Object, Object> vars,
			String... args) throws IOException {
		StringReader srd = new StringReader(source);

		return overwrite(srd, file, stderr,
				inputEncoding, outputEncoding, vars, args);
	}

}
