/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.bc;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

import junit.framework.TestCase;

public class BcExprParserTest extends TestCase {

	void tsp(String s) {
		try {
			new BcExprParser(s).parseExpression(
					new BcRootNamespace(),
					new PrintWriter(new StringWriter()));
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	void stk(String s, String t) {
		BcRootNamespace n = new BcRootNamespace();
		StringWriter b = new StringWriter();
		PrintWriter  p = new PrintWriter(b);

		try {
			new BcExprParser(s).parseExpression(n, p);
			assertEquals(t, b.toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public void testSyntax001() {
		tsp("1");
	}

	public void testSyntax002() {
		tsp("1+2");
		tsp("1+2+3");
		tsp("1+2-3");
		tsp("1^1*1/1%1+1-1&&1||1");
		tsp("!1^!1*!1/!1%!1+!1-!1&&!1||!1");
		tsp("-1^-1*-1/-1%-1+-1- -1&&-1||-1");
		tsp("a++ + b++ * c-- + d--");
		tsp("++a + ++b * --c + --d");
		tsp("-a++ + -b++ * -c-- + -d--");
		tsp("-++a + -++b * - --c + - --d");
		tsp("!a++ + !b++ * !c-- + !d--");
		tsp("!++a + !++b * !--c + !--d");
		tsp("1^1*1/1%(1+1-1)&&1||1");
		tsp("1<1>1<=1>=1==1!=1");
		tsp("a=a^=a*=a/=a%=a+=a^=1");
	}

	public void testParse001() {
		stk("1||2||3",  "1 2 ｜3 ｜");
		stk("1&&2||3",  "1 2 ＆3 ｜");
		stk("1< 2||3",  "1 2 ＜3 ｜");
		stk("1<=2||3",  "1 2 ≦3 ｜");
		stk("1> 2||3",  "1 2 ＞3 ｜");
		stk("1>=2||3",  "1 2 ≧3 ｜");
		stk("1==2||3",  "1 2 ＝3 ｜");
		stk("1!=2||3",  "1 2 ≠3 ｜");
		stk("1-2||3",  "1 2 -3 ｜");
		stk("1*2||3",  "1 2 *3 ｜");
		stk("1/2||3",  "1 2 /3 ｜");
		stk("1%2||3",  "1 2 %3 ｜");
		stk("1^2||3",  "1 2 ^3 ｜");
		stk("1 ||!3",  "1 3 N｜");
		stk("1 ||-3",  "1 3 0r-｜");

		stk("1||2||3",  "1 2 ｜3 ｜");
		stk("1||2&&3",  "1 2 3 ＆｜");
		stk("1||2< 3",  "1 2 3 ＜｜");
		stk("1||2<=3",  "1 2 3 ≦｜");
		stk("1||2> 3",  "1 2 3 ＞｜");
		stk("1||2>=3",  "1 2 3 ≧｜");
		stk("1||2==3",  "1 2 3 ＝｜");
		stk("1||2!=3",  "1 2 3 ≠｜");
		stk("1||2+3",  "1 2 3 +｜");
		stk("1||2*3",  "1 2 3 *｜");
		stk("1||2/3",  "1 2 3 /｜");
		stk("1||2%3",  "1 2 3 %｜");
		stk("1||2^3",  "1 2 3 ^｜");
		stk("!1|| 3",  "1 N3 ｜");
		stk("-1|| 3",  "1 0r-3 ｜");
	}

	public void testParse002() {
		stk("1||2&&3",  "1 2 3 ＆｜");
		stk("1&&2&&3",  "1 2 ＆3 ＆");
		stk("1< 2&&3",  "1 2 ＜3 ＆");
		stk("1<=2&&3",  "1 2 ≦3 ＆");
		stk("1> 2&&3",  "1 2 ＞3 ＆");
		stk("1>=2&&3",  "1 2 ≧3 ＆");
		stk("1==2&&3",  "1 2 ＝3 ＆");
		stk("1!=2&&3",  "1 2 ≠3 ＆");
		stk("1-2&&3",  "1 2 -3 ＆");
		stk("1*2&&3",  "1 2 *3 ＆");
		stk("1/2&&3",  "1 2 /3 ＆");
		stk("1%2&&3",  "1 2 %3 ＆");
		stk("1^2&&3",  "1 2 ^3 ＆");
		stk("1 &&!3",  "1 3 N＆");
		stk("1 &&-3",  "1 3 0r-＆");

		stk("1&&2||3",  "1 2 ＆3 ｜");
		stk("1&&2&&3",  "1 2 ＆3 ＆");
		stk("1&&2< 3",  "1 2 3 ＜＆");
		stk("1&&2<=3",  "1 2 3 ≦＆");
		stk("1&&2> 3",  "1 2 3 ＞＆");
		stk("1&&2>=3",  "1 2 3 ≧＆");
		stk("1&&2==3",  "1 2 3 ＝＆");
		stk("1&&2!=3",  "1 2 3 ≠＆");
		stk("1&&2+3",  "1 2 3 +＆");
		stk("1&&2*3",  "1 2 3 *＆");
		stk("1&&2/3",  "1 2 3 /＆");
		stk("1&&2%3",  "1 2 3 %＆");
		stk("1&&2^3",  "1 2 3 ^＆");
		stk("!1&& 3",  "1 N3 ＆");
		stk("-1&& 3",  "1 0r-3 ＆");
	}

	public void testParse004_1() {
		stk("1||2< 3",  "1 2 3 ＜｜");
		stk("1&&2< 3",  "1 2 3 ＜＆");
		stk("1< 2< 3",  "1 2 ＜3 ＜");
		stk("1<=2< 3",  "1 2 ≦3 ＜");
		stk("1> 2< 3",  "1 2 ＞3 ＜");
		stk("1>=2< 3",  "1 2 ≧3 ＜");
		stk("1==2< 3",  "1 2 ＝3 ＜");
		stk("1!=2< 3",  "1 2 ≠3 ＜");
		stk("1-2< 3",  "1 2 -3 ＜");
		stk("1*2< 3",  "1 2 *3 ＜");
		stk("1/2< 3",  "1 2 /3 ＜");
		stk("1%2< 3",  "1 2 %3 ＜");
		stk("1^2< 3",  "1 2 ^3 ＜");
		stk("1 < !3",  "1 3 N＜");
		stk("1 < -3",  "1 3 0r-＜");

		stk("1< 2||3",  "1 2 ＜3 ｜");
		stk("1< 2&&3",  "1 2 ＜3 ＆");
		stk("1< 2< 3",  "1 2 ＜3 ＜");
		stk("1< 2<=3",  "1 2 ＜3 ≦");
		stk("1< 2> 3",  "1 2 ＜3 ＞");
		stk("1< 2>=3",  "1 2 ＜3 ≧");
		stk("1< 2==3",  "1 2 ＜3 ＝");
		stk("1< 2!=3",  "1 2 ＜3 ≠");
		stk("1< 2+3",  "1 2 3 +＜");
		stk("1< 2*3",  "1 2 3 *＜");
		stk("1< 2/3",  "1 2 3 /＜");
		stk("1< 2%3",  "1 2 3 %＜");
		stk("1< 2^3",  "1 2 3 ^＜");
		stk("!1<  3",  "1 3 ＜N");
		stk("-1<  3",  "1 0r-3 ＜");
	}

	public void testParse004_2() {
		stk("1||2<=3",  "1 2 3 ≦｜");
		stk("1&&2<=3",  "1 2 3 ≦＆");
		stk("1< 2<=3",  "1 2 ＜3 ≦");
		stk("1<=2<=3",  "1 2 ≦3 ≦");
		stk("1> 2<=3",  "1 2 ＞3 ≦");
		stk("1>=2<=3",  "1 2 ≧3 ≦");
		stk("1==2<=3",  "1 2 ＝3 ≦");
		stk("1!=2<=3",  "1 2 ≠3 ≦");
		stk("1-2<=3",  "1 2 -3 ≦");
		stk("1*2<=3",  "1 2 *3 ≦");
		stk("1/2<=3",  "1 2 /3 ≦");
		stk("1%2<=3",  "1 2 %3 ≦");
		stk("1^2<=3",  "1 2 ^3 ≦");
		stk("1 <=!3",  "1 3 N≦");
		stk("1 <=-3",  "1 3 0r-≦");

		stk("1<=2||3",  "1 2 ≦3 ｜");
		stk("1<=2&&3",  "1 2 ≦3 ＆");
		stk("1<=2< 3",  "1 2 ≦3 ＜");
		stk("1<=2<=3",  "1 2 ≦3 ≦");
		stk("1<=2> 3",  "1 2 ≦3 ＞");
		stk("1<=2>=3",  "1 2 ≦3 ≧");
		stk("1<=2==3",  "1 2 ≦3 ＝");
		stk("1<=2!=3",  "1 2 ≦3 ≠");
		stk("1<=2+3",  "1 2 3 +≦");
		stk("1<=2*3",  "1 2 3 *≦");
		stk("1<=2/3",  "1 2 3 /≦");
		stk("1<=2%3",  "1 2 3 %≦");
		stk("1<=2^3",  "1 2 3 ^≦");
		stk("!1<= 3",  "1 3 ≦N");
		stk("-1<= 3",  "1 0r-3 ≦");
	}

	public void testParse004_3() {
		stk("1||2> 3",  "1 2 3 ＞｜");
		stk("1&&2> 3",  "1 2 3 ＞＆");
		stk("1< 2> 3",  "1 2 ＜3 ＞");
		stk("1<=2> 3",  "1 2 ≦3 ＞");
		stk("1> 2> 3",  "1 2 ＞3 ＞");
		stk("1>=2> 3",  "1 2 ≧3 ＞");
		stk("1==2> 3",  "1 2 ＝3 ＞");
		stk("1!=2> 3",  "1 2 ≠3 ＞");
		stk("1-2> 3",  "1 2 -3 ＞");
		stk("1*2> 3",  "1 2 *3 ＞");
		stk("1/2> 3",  "1 2 /3 ＞");
		stk("1%2> 3",  "1 2 %3 ＞");
		stk("1^2> 3",  "1 2 ^3 ＞");
		stk("1 > !3",  "1 3 N＞");
		stk("1 > -3",  "1 3 0r-＞");

		stk("1> 2||3",  "1 2 ＞3 ｜");
		stk("1> 2&&3",  "1 2 ＞3 ＆");
		stk("1> 2< 3",  "1 2 ＞3 ＜");
		stk("1> 2<=3",  "1 2 ＞3 ≦");
		stk("1> 2> 3",  "1 2 ＞3 ＞");
		stk("1> 2>=3",  "1 2 ＞3 ≧");
		stk("1> 2==3",  "1 2 ＞3 ＝");
		stk("1> 2!=3",  "1 2 ＞3 ≠");
		stk("1> 2+3",  "1 2 3 +＞");
		stk("1> 2*3",  "1 2 3 *＞");
		stk("1> 2/3",  "1 2 3 /＞");
		stk("1> 2%3",  "1 2 3 %＞");
		stk("1> 2^3",  "1 2 3 ^＞");
		stk("!1>  3",  "1 3 ＞N");
		stk("-1>  3",  "1 0r-3 ＞");
	}

	public void testParse004_4() {
		stk("1||2>=3",  "1 2 3 ≧｜");
		stk("1&&2>=3",  "1 2 3 ≧＆");
		stk("1< 2>=3",  "1 2 ＜3 ≧");
		stk("1<=2>=3",  "1 2 ≦3 ≧");
		stk("1> 2>=3",  "1 2 ＞3 ≧");
		stk("1>=2>=3",  "1 2 ≧3 ≧");
		stk("1==2>=3",  "1 2 ＝3 ≧");
		stk("1!=2>=3",  "1 2 ≠3 ≧");
		stk("1-2>=3",  "1 2 -3 ≧");
		stk("1*2>=3",  "1 2 *3 ≧");
		stk("1/2>=3",  "1 2 /3 ≧");
		stk("1%2>=3",  "1 2 %3 ≧");
		stk("1^2>=3",  "1 2 ^3 ≧");
		stk("1 >=!3",  "1 3 N≧");
		stk("1 >=-3",  "1 3 0r-≧");

		stk("1>=2||3",  "1 2 ≧3 ｜");
		stk("1>=2&&3",  "1 2 ≧3 ＆");
		stk("1>=2< 3",  "1 2 ≧3 ＜");
		stk("1>=2<=3",  "1 2 ≧3 ≦");
		stk("1>=2> 3",  "1 2 ≧3 ＞");
		stk("1>=2>=3",  "1 2 ≧3 ≧");
		stk("1>=2==3",  "1 2 ≧3 ＝");
		stk("1>=2!=3",  "1 2 ≧3 ≠");
		stk("1>=2+3",  "1 2 3 +≧");
		stk("1>=2*3",  "1 2 3 *≧");
		stk("1>=2/3",  "1 2 3 /≧");
		stk("1>=2%3",  "1 2 3 %≧");
		stk("1>=2^3",  "1 2 3 ^≧");
		stk("!1>= 3",  "1 3 ≧N");
		stk("-1>= 3",  "1 0r-3 ≧");
	}

	public void testParse004_5() {
		stk("1||2==3",  "1 2 3 ＝｜");
		stk("1&&2==3",  "1 2 3 ＝＆");
		stk("1< 2==3",  "1 2 ＜3 ＝");
		stk("1<=2==3",  "1 2 ≦3 ＝");
		stk("1> 2==3",  "1 2 ＞3 ＝");
		stk("1>=2==3",  "1 2 ≧3 ＝");
		stk("1==2==3",  "1 2 ＝3 ＝");
		stk("1!=2==3",  "1 2 ≠3 ＝");
		stk("1-2==3",  "1 2 -3 ＝");
		stk("1*2==3",  "1 2 *3 ＝");
		stk("1/2==3",  "1 2 /3 ＝");
		stk("1%2==3",  "1 2 %3 ＝");
		stk("1^2==3",  "1 2 ^3 ＝");
		stk("1 ==!3",  "1 3 N＝");
		stk("1 ==-3",  "1 3 0r-＝");

		stk("1==2||3",  "1 2 ＝3 ｜");
		stk("1==2&&3",  "1 2 ＝3 ＆");
		stk("1==2< 3",  "1 2 ＝3 ＜");
		stk("1==2<=3",  "1 2 ＝3 ≦");
		stk("1==2> 3",  "1 2 ＝3 ＞");
		stk("1==2>=3",  "1 2 ＝3 ≧");
		stk("1==2==3",  "1 2 ＝3 ＝");
		stk("1==2!=3",  "1 2 ＝3 ≠");
		stk("1==2+3",  "1 2 3 +＝");
		stk("1==2*3",  "1 2 3 *＝");
		stk("1==2/3",  "1 2 3 /＝");
		stk("1==2%3",  "1 2 3 %＝");
		stk("1==2^3",  "1 2 3 ^＝");
		stk("!1== 3",  "1 3 ＝N");
		stk("-1== 3",  "1 0r-3 ＝");
	}

	public void testParse004_6() {
		stk("1||2!=3",  "1 2 3 ≠｜");
		stk("1&&2!=3",  "1 2 3 ≠＆");
		stk("1< 2!=3",  "1 2 ＜3 ≠");
		stk("1<=2!=3",  "1 2 ≦3 ≠");
		stk("1> 2!=3",  "1 2 ＞3 ≠");
		stk("1>=2!=3",  "1 2 ≧3 ≠");
		stk("1==2!=3",  "1 2 ＝3 ≠");
		stk("1!=2!=3",  "1 2 ≠3 ≠");
		stk("1-2!=3",  "1 2 -3 ≠");
		stk("1*2!=3",  "1 2 *3 ≠");
		stk("1/2!=3",  "1 2 /3 ≠");
		stk("1%2!=3",  "1 2 %3 ≠");
		stk("1^2!=3",  "1 2 ^3 ≠");
		stk("1 !=!3",  "1 3 N≠");
		stk("1 !=-3",  "1 3 0r-≠");

		stk("1!=2||3",  "1 2 ≠3 ｜");
		stk("1!=2&&3",  "1 2 ≠3 ＆");
		stk("1!=2< 3",  "1 2 ≠3 ＜");
		stk("1!=2<=3",  "1 2 ≠3 ≦");
		stk("1!=2> 3",  "1 2 ≠3 ＞");
		stk("1!=2>=3",  "1 2 ≠3 ≧");
		stk("1!=2==3",  "1 2 ≠3 ＝");
		stk("1!=2!=3",  "1 2 ≠3 ≠");
		stk("1!=2+3",  "1 2 3 +≠");
		stk("1!=2*3",  "1 2 3 *≠");
		stk("1!=2/3",  "1 2 3 /≠");
		stk("1!=2%3",  "1 2 3 %≠");
		stk("1!=2^3",  "1 2 3 ^≠");
		stk("!1!= 3",  "1 3 ≠N");
		stk("-1!= 3",  "1 0r-3 ≠");
	}

	public void testParse005() {
		stk("1||2+3",  "1 2 3 +｜");
		stk("1&&2+3",  "1 2 3 +＆");
		stk("1< 2+3",  "1 2 3 +＜");
		stk("1<=2+3",  "1 2 3 +≦");
		stk("1> 2+3",  "1 2 3 +＞");
		stk("1>=2+3",  "1 2 3 +≧");
		stk("1==2+3",  "1 2 3 +＝");
		stk("1!=2+3",  "1 2 3 +≠");
		stk("1-2+3",  "1 2 -3 +");
		stk("1*2+3",  "1 2 *3 +");
		stk("1/2+3",  "1 2 /3 +");
		stk("1%2+3",  "1 2 %3 +");
		stk("1^2+3",  "1 2 ^3 +");
		stk("1 +!3",  "1 3 N+");
		stk("!1+ 3",  "1 3 +N");
		stk("1 +-3",  "1 3 0r-+");

		stk("1+2||3",  "1 2 +3 ｜");
		stk("1+2&&3",  "1 2 +3 ＆");
		stk("1+2< 3",  "1 2 +3 ＜");
		stk("1+2<=3",  "1 2 +3 ≦");
		stk("1+2> 3",  "1 2 +3 ＞");
		stk("1+2>=3",  "1 2 +3 ≧");
		stk("1+2==3",  "1 2 +3 ＝");
		stk("1+2!=3",  "1 2 +3 ≠");
		stk("1+2+3",  "1 2 +3 +");
		stk("1+2*3",  "1 2 3 *+");
		stk("1+2/3",  "1 2 3 /+");
		stk("1+2%3",  "1 2 3 %+");
		stk("1+2^3",  "1 2 3 ^+");
		stk("-1+ 3",  "1 0r-3 +");
	}

	public void testParse006() {
		stk("1||2-3",  "1 2 3 -｜");
		stk("1&&2-3",  "1 2 3 -＆");
		stk("1&&2-3",  "1 2 3 -＆");
		stk("1< 2-3",  "1 2 3 -＜");
		stk("1<=2-3",  "1 2 3 -≦");
		stk("1> 2-3",  "1 2 3 -＞");
		stk("1>=2-3",  "1 2 3 -≧");
		stk("1==2-3",  "1 2 3 -＝");
		stk("1!=2-3",  "1 2 3 -≠");
		stk("1-2-3",  "1 2 -3 -");
		stk("1*2-3",  "1 2 *3 -");
		stk("1/2-3",  "1 2 /3 -");
		stk("1%2-3",  "1 2 %3 -");
		stk("1^2-3",  "1 2 ^3 -");
		stk("1 -!3",  "1 3 N-");
		stk("1- -3",  "1 3 0r--");

		stk("1-2||3",  "1 2 -3 ｜");
		stk("1-2&&3",  "1 2 -3 ＆");
		stk("1-2< 3",  "1 2 -3 ＜");
		stk("1-2<=3",  "1 2 -3 ≦");
		stk("1-2> 3",  "1 2 -3 ＞");
		stk("1-2>=3",  "1 2 -3 ≧");
		stk("1-2==3",  "1 2 -3 ＝");
		stk("1-2!=3",  "1 2 -3 ≠");
		stk("1-2+3",  "1 2 -3 +");
		stk("1-2*3",  "1 2 3 *-");
		stk("1-2/3",  "1 2 3 /-");
		stk("1-2%3",  "1 2 3 %-");
		stk("1-2^3",  "1 2 3 ^-");
		stk("!1- 3",  "1 3 -N");
		stk("-1- 3",  "1 0r-3 -");
	}

	public void testParse007() {
		stk("1||2*3",  "1 2 3 *｜");
		stk("1&&2*3",  "1 2 3 *＆");
		stk("1< 2*3",  "1 2 3 *＜");
		stk("1<=2*3",  "1 2 3 *≦");
		stk("1> 2*3",  "1 2 3 *＞");
		stk("1>=2*3",  "1 2 3 *≧");
		stk("1==2*3",  "1 2 3 *＝");
		stk("1!=2*3",  "1 2 3 *≠");
		stk("1+2*3",  "1 2 3 *+");
		stk("1*2*3",  "1 2 *3 *");
		stk("1/2*3",  "1 2 /3 *");
		stk("1%2*3",  "1 2 %3 *");
		stk("1^2*3",  "1 2 ^3 *");
		stk("1 *!3",  "1 3 N*");
		stk("1* -3",  "1 3 0r-*");

		stk("1*2||3",  "1 2 *3 ｜");
		stk("1*2&&3",  "1 2 *3 ＆");
		stk("1*2< 3",  "1 2 *3 ＜");
		stk("1*2<=3",  "1 2 *3 ≦");
		stk("1*2> 3",  "1 2 *3 ＞");
		stk("1*2>=3",  "1 2 *3 ≧");
		stk("1*2==3",  "1 2 *3 ＝");
		stk("1*2!=3",  "1 2 *3 ≠");
		stk("1*2+3",  "1 2 *3 +");
		stk("1*2*3",  "1 2 *3 *");
		stk("1*2/3",  "1 2 *3 /");
		stk("1*2%3",  "1 2 *3 %");
		stk("1*2^3",  "1 2 3 ^*");
		stk("!1* 3",  "1 3 *N");
		stk("-1* 3",  "1 0r-3 *");
	}

	public void testParse008() {
		stk("1||2/3",  "1 2 3 /｜");
		stk("1&&2/3",  "1 2 3 /＆");
		stk("1< 2/3",  "1 2 3 /＜");
		stk("1<=2/3",  "1 2 3 /≦");
		stk("1> 2/3",  "1 2 3 /＞");
		stk("1>=2/3",  "1 2 3 /≧");
		stk("1==2/3",  "1 2 3 /＝");
		stk("1!=2/3",  "1 2 3 /≠");
		stk("1+2/3",  "1 2 3 /+");
		stk("1*2/3",  "1 2 *3 /");
		stk("1/2/3",  "1 2 /3 /");
		stk("1%2/3",  "1 2 %3 /");
		stk("1^2/3",  "1 2 ^3 /");
		stk("1 /!3",  "1 3 N/");
		stk("1/ -3",  "1 3 0r-/");

		stk("1/2||3",  "1 2 /3 ｜");
		stk("1/2&&3",  "1 2 /3 ＆");
		stk("1/2< 3",  "1 2 /3 ＜");
		stk("1/2<=3",  "1 2 /3 ≦");
		stk("1/2> 3",  "1 2 /3 ＞");
		stk("1/2>=3",  "1 2 /3 ≧");
		stk("1/2==3",  "1 2 /3 ＝");
		stk("1/2!=3",  "1 2 /3 ≠");
		stk("1/2+3",  "1 2 /3 +");
		stk("1/2*3",  "1 2 /3 *");
		stk("1/2/3",  "1 2 /3 /");
		stk("1/2%3",  "1 2 /3 %");
		stk("1/2^3",  "1 2 3 ^/");
		stk("!1/ 3",  "1 3 /N");
		stk("-1/ 3",  "1 0r-3 /");
	}

	public void testParse009() {
		stk("1||2%3",  "1 2 3 %｜");
		stk("1&&2%3",  "1 2 3 %＆");
		stk("1< 2%3",  "1 2 3 %＜");
		stk("1<=2%3",  "1 2 3 %≦");
		stk("1> 2%3",  "1 2 3 %＞");
		stk("1>=2%3",  "1 2 3 %≧");
		stk("1==2%3",  "1 2 3 %＝");
		stk("1!=2%3",  "1 2 3 %≠");
		stk("1+2%3",  "1 2 3 %+");
		stk("1*2%3",  "1 2 *3 %");
		stk("1/2%3",  "1 2 /3 %");
		stk("1%2%3",  "1 2 %3 %");
		stk("1^2%3",  "1 2 ^3 %");
		stk("1 %!3",  "1 3 N%");
		stk("1% -3",  "1 3 0r-%");

		stk("1%2||3",  "1 2 %3 ｜");
		stk("1%2&&3",  "1 2 %3 ＆");
		stk("1%2< 3",  "1 2 %3 ＜");
		stk("1%2<=3",  "1 2 %3 ≦");
		stk("1%2> 3",  "1 2 %3 ＞");
		stk("1%2>=3",  "1 2 %3 ≧");
		stk("1%2==3",  "1 2 %3 ＝");
		stk("1%2!=3",  "1 2 %3 ≠");
		stk("1%2+3",  "1 2 %3 +");
		stk("1%2*3",  "1 2 %3 *");
		stk("1%2/3",  "1 2 %3 /");
		stk("1%2%3",  "1 2 %3 %");
		stk("1%2^3",  "1 2 3 ^%");
		stk("!1% 3",  "1 3 %N");
		stk("-1% 3",  "1 0r-3 %");
	}

	public void testParse010() {
		stk("1||2^3",  "1 2 3 ^｜");
		stk("1&&2^3",  "1 2 3 ^＆");
		stk("1< 2^3",  "1 2 3 ^＜");
		stk("1<=2^3",  "1 2 3 ^≦");
		stk("1> 2^3",  "1 2 3 ^＞");
		stk("1>=2^3",  "1 2 3 ^≧");
		stk("1==2^3",  "1 2 3 ^＝");
		stk("1!=2^3",  "1 2 3 ^≠");
		stk("1+2^3",  "1 2 3 ^+");
		stk("1*2^3",  "1 2 3 ^*");
		stk("1/2^3",  "1 2 3 ^/");
		stk("1%2^3",  "1 2 3 ^%");
		stk("1^2^3",  "1 2 3 ^^");
		stk("1 ^!3",  "1 3 N^");
		stk("1^ -3",  "1 3 0r-^");

		stk("1^2||3",  "1 2 ^3 ｜");
		stk("1^2&&3",  "1 2 ^3 ＆");
		stk("1^2< 3",  "1 2 ^3 ＜");
		stk("1^2<=3",  "1 2 ^3 ≦");
		stk("1^2> 3",  "1 2 ^3 ＞");
		stk("1^2>=3",  "1 2 ^3 ≧");
		stk("1^2==3",  "1 2 ^3 ＝");
		stk("1^2!=3",  "1 2 ^3 ≠");
		stk("1^2+3",  "1 2 ^3 +");
		stk("1^2*3",  "1 2 ^3 *");
		stk("1^2/3",  "1 2 ^3 /");
		stk("1^2%3",  "1 2 ^3 %");
		stk("1^2^3",  "1 2 3 ^^");
		stk("!1^ 3",  "1 3 ^N");
		stk("-1^ 3",  "1 0r-3 ^");
	}

	public void testParse021() {
		stk("a", "l\u4e00");
	}

	public void testParse022() {
		stk("a =1", "1 ds\u4e00");
		stk("a+=1", "1 l\u4e00r+ds\u4e00");
		stk("a-=1", "1 l\u4e00r-ds\u4e00");
		stk("a*=1", "1 l\u4e00r*ds\u4e00");
		stk("a/=1", "1 l\u4e00r/ds\u4e00");
		stk("a%=1", "1 l\u4e00r%ds\u4e00");
		stk("a^=1", "1 l\u4e00r^ds\u4e00");
	}

	public void testParse023() {
		stk("a =b =1", "1 ds\u4e01ds\u4e00");
		stk("a+=b-=1", "1 l\u4e01r-ds\u4e01l\u4e00r+ds\u4e00");
	}

	public void testParse024() {
		stk("a++", "l\u4e00d1+ds\u4e00捨");
		stk("a++ && 2", "l\u4e00d1+ds\u4e00捨2 ＆");
		stk("a++ || 2", "l\u4e00d1+ds\u4e00捨2 ｜");
		stk("a++ <  2", "l\u4e00d1+ds\u4e00捨2 ＜");
		stk("a++ <= 2", "l\u4e00d1+ds\u4e00捨2 ≦");
		stk("a++ >  2", "l\u4e00d1+ds\u4e00捨2 ＞");
		stk("a++ >= 2", "l\u4e00d1+ds\u4e00捨2 ≧");
		stk("a++ == 2", "l\u4e00d1+ds\u4e00捨2 ＝");
		stk("a++ != 2", "l\u4e00d1+ds\u4e00捨2 ≠");
		stk("a++ +  2", "l\u4e00d1+ds\u4e00捨2 +");
		stk("a++ -  2", "l\u4e00d1+ds\u4e00捨2 -");
		stk("a++ *  2", "l\u4e00d1+ds\u4e00捨2 *");
		stk("a++ /  2", "l\u4e00d1+ds\u4e00捨2 /");
		stk("a++ %  2", "l\u4e00d1+ds\u4e00捨2 %");
		stk("a++ ^  2", "l\u4e00d1+ds\u4e00捨2 ^");

		stk("2 && a++", "2 l\u4e00d1+ds\u4e00捨＆");
		stk("2 || a++", "2 l\u4e00d1+ds\u4e00捨｜");
		stk("2 <  a++", "2 l\u4e00d1+ds\u4e00捨＜");
		stk("2 <= a++", "2 l\u4e00d1+ds\u4e00捨≦");
		stk("2 >  a++", "2 l\u4e00d1+ds\u4e00捨＞");
		stk("2 >= a++", "2 l\u4e00d1+ds\u4e00捨≧");
		stk("2 == a++", "2 l\u4e00d1+ds\u4e00捨＝");
		stk("2 != a++", "2 l\u4e00d1+ds\u4e00捨≠");
		stk("2 +  a++", "2 l\u4e00d1+ds\u4e00捨+");
		stk("2 -  a++", "2 l\u4e00d1+ds\u4e00捨-");
		stk("2 *  a++", "2 l\u4e00d1+ds\u4e00捨*");
		stk("2 /  a++", "2 l\u4e00d1+ds\u4e00捨/");
		stk("2 %  a++", "2 l\u4e00d1+ds\u4e00捨%");
		stk("2 ^  a++", "2 l\u4e00d1+ds\u4e00捨^");

		stk("!a++", "l\u4e00d1+ds\u4e00捨N");
		stk("-a++", "l\u4e00d1+ds\u4e00捨0r-");
	}

	public void testParse025() {
		stk("a--", "l\u4e00d1-ds\u4e00捨");
		stk("a-- && 2", "l\u4e00d1-ds\u4e00捨2 ＆");
		stk("a-- || 2", "l\u4e00d1-ds\u4e00捨2 ｜");
		stk("a-- <  2", "l\u4e00d1-ds\u4e00捨2 ＜");
		stk("a-- <= 2", "l\u4e00d1-ds\u4e00捨2 ≦");
		stk("a-- >  2", "l\u4e00d1-ds\u4e00捨2 ＞");
		stk("a-- >= 2", "l\u4e00d1-ds\u4e00捨2 ≧");
		stk("a-- == 2", "l\u4e00d1-ds\u4e00捨2 ＝");
		stk("a-- != 2", "l\u4e00d1-ds\u4e00捨2 ≠");
		stk("a-- +  2", "l\u4e00d1-ds\u4e00捨2 +");
		stk("a-- -  2", "l\u4e00d1-ds\u4e00捨2 -");
		stk("a-- *  2", "l\u4e00d1-ds\u4e00捨2 *");
		stk("a-- /  2", "l\u4e00d1-ds\u4e00捨2 /");
		stk("a-- %  2", "l\u4e00d1-ds\u4e00捨2 %");
		stk("a-- ^  2", "l\u4e00d1-ds\u4e00捨2 ^");

		stk("2 && a--", "2 l\u4e00d1-ds\u4e00捨＆");
		stk("2 || a--", "2 l\u4e00d1-ds\u4e00捨｜");
		stk("2 <  a--", "2 l\u4e00d1-ds\u4e00捨＜");
		stk("2 <= a--", "2 l\u4e00d1-ds\u4e00捨≦");
		stk("2 >  a--", "2 l\u4e00d1-ds\u4e00捨＞");
		stk("2 >= a--", "2 l\u4e00d1-ds\u4e00捨≧");
		stk("2 == a--", "2 l\u4e00d1-ds\u4e00捨＝");
		stk("2 != a--", "2 l\u4e00d1-ds\u4e00捨≠");
		stk("2 +  a--", "2 l\u4e00d1-ds\u4e00捨+");
		stk("2 -  a--", "2 l\u4e00d1-ds\u4e00捨-");
		stk("2 *  a--", "2 l\u4e00d1-ds\u4e00捨*");
		stk("2 /  a--", "2 l\u4e00d1-ds\u4e00捨/");
		stk("2 %  a--", "2 l\u4e00d1-ds\u4e00捨%");
		stk("2 ^  a--", "2 l\u4e00d1-ds\u4e00捨^");

		stk("!a--", "l\u4e00d1-ds\u4e00捨N");
		stk("-a--", "l\u4e00d1-ds\u4e00捨0r-");
	}

	public void testParse026() {
		stk("++a", "l\u4e001+ds\u4e00");
		stk("++a && 2", "l\u4e001+ds\u4e002 ＆");

		stk("2 && ++a", "2 l\u4e001+ds\u4e00＆");
	}

	public void testParse027() {
		stk("a[1]", "1 ;\u4e00");
		stk("a[1] + a[2]", "1 ;\u4e002 ;\u4e00+");
		stk("a[3 * 4]", "3 4 *;\u4e00");
	}

	public void testParse028() {
		stk("a[1] =2", "1 2 s\uff01r:l\uff01");
		stk("a[1]+=2", "1 2 rs\uff02;\u4e00r+dl\uff02:\u4e00");
		stk("a[1]-=2", "1 2 rs\uff02;\u4e00r-dl\uff02:\u4e00");
		stk("a[1]*=2", "1 2 rs\uff02;\u4e00r*dl\uff02:\u4e00");
		stk("a[1]/=2", "1 2 rs\uff02;\u4e00r/dl\uff02:\u4e00");
		stk("a[1]%=2", "1 2 rs\uff02;\u4e00r%dl\uff02:\u4e00");
		stk("a[1]^=2", "1 2 rs\uff02;\u4e00r^dl\uff02:\u4e00");
	}

	public void testParse029() {
		stk("a[1]+=a[2]-=3",
				"1 2 3 rs\uff02;\u4e00r-dl\uff02:\u4e00" +
				"rs\uff02;\u4e00r+dl\uff02:\u4e00");
	}

	public void testParse030() {
		stk("sqrt(841)", "841 v");
		stk("length(841)", "841 Z");
		stk("scale(841)", "841 X");
	}

}
