/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import net.morilib.sh.builtin.ShBuiltInCommands;
import net.morilib.sh.misc.XtraceStream;

public class ShTreeIf implements ShTree {

	private List<ShTree> conditions;
	private List<ShTree> block;
	private ShTree elseblock;

	public ShTreeIf(List<ShTree> conditions, List<ShTree> block,
			ShTree elseblock) {
		this.conditions = new ArrayList<ShTree>(conditions);
		this.block      = new ArrayList<ShTree>(block);
		this.elseblock  = elseblock;
	}

	public int eval(ShEnvironment env,
			ShFileSystem fs,
			ShBuiltInCommands cmds,
			ShRuntime run,
			InputStream in,
			PrintStream out,
			PrintStream err,
			XtraceStream p) throws IOException, ShSyntaxException {
		int a;

		for(int i = 0; i < conditions.size(); i++) {
			a = conditions.get(i).eval(env, fs, cmds, run, in,
					out, err, p);
			if(a == 0) {
				return block.get(i).eval(env, fs, cmds, run, in,
						out, err, p);
			}
		}

		if(elseblock != null) {
			return elseblock.eval(env, fs, cmds, run, in, out, err, p);
		} else {
			return 0;
		}
	}

	public void compileInternally(ShTreeExpressionMachine.Builder b,
			Object brk, Object cnt) {
		Object l1 = null, l2 = new Object();

		for(int i = 0; i < conditions.size(); i++) {
			if(i > 0)  b.addLabel(l1);
			conditions.get(i).compileInternally(b, brk, cnt);
			b.addJmpZ(l1 = new Object());
			block.get(i).compileInternally(b, brk, cnt);
		}

		if(elseblock != null) {
			b.addJmp(l2);
			b.addLabel(l1);
			elseblock.compileInternally(b, brk, cnt);
			b.addLabel(l2);
		} else {
			b.addLabel(l1);
		}
		b.addRes();
	}

	public String toString() {
		StringBuffer b = new StringBuffer("if ");
		String d = "";

		for(int i = 0; i < block.size(); i++) {
			b.append(d).append(conditions.get(i));
			b.append("\nthen\n").append(block.get(i)).append('\n');
			d = "elif ";
		}

		if(elseblock != null) {
			b.append("else\n").append(elseblock).append('\n');
		}
		return b.append("fi").toString();
	}

}
