package net.morilib.sh;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PushbackReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.morilib.sh.misc.IOs;
import net.morilib.sh.misc.XtraceStream;

import junit.framework.TestCase;

public class ShReplaceTest extends TestCase {

	static final PrintStream
	ERR_OUT = new PrintStream(new OutputStream() {

		@Override
		public void write(int b) throws IOException {
			throw new RuntimeException();
		}

	});

	static final ShRuntime T1 = new ShRuntime() {

		public int eval(ShEnvironment env,
				ShFileSystem fs,
				InputStream stdin,
				PrintStream stdout,
				PrintStream stderr,
				XtraceStream p,
				String script) {
			if(script.equals("whoami")) {
				stdout.print("haruka\n");
				stdout.flush();
			}
			return 0;
		}

	};

	public static PushbackReader grd(String s) {
		return new PushbackReader(new StringReader(s));
	}

	public void eqt(ShEnvironment env, String t, String arg) {
		ShString r = new ShString(t);

		assertEquals(arg, r.replaceTilde(env).toString());
	}

	public void eqp(ShEnvironment env, String t, String... args) {
		ShString r = new ShString(t);
		List<String> l = new ArrayList<String>();

		for(ShToken x : r.replaceParameter(env, ERR_OUT)) {
			l.add(x.toString());
		}
		assertEquals(Arrays.asList(args), l);
	}

	public void msp(ShEnvironment env, String t, String msg) {
		ByteArrayOutputStream b = new ByteArrayOutputStream();
		PrintStream p = new PrintStream(b);
		ShString r = new ShString(t);
		String s;

		try {
			r.replaceParameter(env, p);
			fail();
		} catch(ShRuntimeException e) {
			s = new String(b.toByteArray());
			assertEquals(msg, s.replaceAll("\r\n", "\n"));
		}
	}

	public void eqc(String t, String s) {
		ShString r = new ShString(t);
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		try {
			assertEquals(s, r.replaceCommand(
					env,
					new ShFileSystemStub(),
					T1,
					new XtraceStream(IOs.NULL_OUTPUT)).toString());
		} catch (IOException e) {
			throw new RuntimeException();
		} catch (ShSyntaxException e) {
			throw new RuntimeException();
		}
	}

	public void eql(String t, String ifs, String... args) {
		ShString r = new ShString(t);

		assertEquals(Arrays.asList(args), r.splitWord(ifs));
	}

	public void eqa(ShEnvironment env, String t, String s) {
		ShString r = new ShString(t);

		try {
			assertEquals(s, r.replaceArithmetic(env).toString());
		} catch (ShSyntaxException e) {
			throw new RuntimeException(e);
		}
	}

	public void okw(String t) {
		assertTrue(ShToken.isWildcard(t));
	}

	public void ngw(String t) {
		assertFalse(ShToken.isWildcard(t));
	}

	public void testParameter001() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "aaaa", "aaaa");
	}

	public void testParameter002() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "${NAME_1}", "name");
	}

	public void testParameter003() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "${NAME_1}aaa", "nameaaa");
	}

	public void testParameter004() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "aaa${NAME_1}", "aaaname");
	}

	public void testParameter005() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${NAME_1}aaa", "aaa");
	}

	public void testParameter006() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "'${NAME_1}'", "'${NAME_1}'");
	}

	public void testParameter007() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\"${NAME_1}\"", "\"name\"");
	}

	public void testParameter008() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "$(whoami)", "$(whoami)");
	}

	public void testParameter009() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\"$(whoami)\"", "\"$(whoami)\"");
	}

	public void testParameter010() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "'$(whoami)'", "'$(whoami)'");
	}

	public void testParameter011() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${whoami", "${whoami");
	}

	public void testParameter012() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "$(whoami", "$(whoami");
	}

	public void testParameter013() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\\${NAME_1}", "\\${NAME_1}");
	}

	public void testParameter014() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "aa\\${NAME_1}", "aa\\${NAME_1}");
	}

	public void testParameter015() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\"\\${NAME_1}\"", "\"\\${NAME_1}\"");
	}

	public void testParameter016() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\\", "\\");
	}

	public void testParameter017() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("NAME_1", "name");
		eqp(env, "\\'${NAME_1}", "\\'name");
	}

	public void testParameter018() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "${*}", "arg1,arg2");
	}

	public void testParameter019() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "\"${*}\"", "\"arg1,arg2\"");
	}

	public void testParameter020() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "${@}", "arg1", "arg2");
	}

	public void testParameter021() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "\"${@}\"", "\"arg1\"", "\"arg2\"");
	}

	public void testParameter022() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "aaa\"${@}\"aaa",
				"aaa\"arg1\"", "\"arg2\"aaa");
	}

	public void testParameter023() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.put("IFS", ",");
		env.put("0", "arg0");
		env.put("1", "arg1");
		env.put("2", "arg2");
		env.put("#", "3");
		eqp(env, "\"aaa${@}aaa\"",
				"\"aaaarg1\"", "\"arg2aaa\"");
	}

	public void testParameter101() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "bbb");
		eqp(env, "${aaa:-zzz}", "bbb");
	}

	public void testParameter102() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa:-zzz}", "zzz");
	}

	public void testParameter103() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${aaa:-zzz}", "zzz");
	}

	public void testParameter104() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${aaa:-}");
	}

	public void testParameter105() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "bbb");
		eqp(env, "${aaa:=zzz}", "bbb");
		assertEquals("bbb", env.find("aaa"));
	}

	public void testParameter106() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa:=zzz}", "zzz");
		assertEquals("zzz", env.find("aaa"));
	}

	public void testParameter107() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${aaa:=zzz}", "zzz");
		assertEquals("zzz", env.find("aaa"));
	}

	public void testParameter108() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${aaa:=}");
		assertEquals("", env.find("aaa"));
	}

	public void testParameter109() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "bbb");
		eqp(env, "${aaa:?not set}", "bbb");
	}

	public void testParameter110() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		msp(env, "${aaa:?not set}", "aaa: not set\n");
	}

	public void testParameter111() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		msp(env, "${aaa:?not set}", "aaa: not set\n");
	}

	public void testParameter112() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		msp(env, "${aaa:?}", "aaa: parameter null or not set\n");
	}

	public void testParameter113() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "bbb");
		eqp(env, "${aaa:+zzz}", "zzz");
	}

	public void testParameter114() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa:+zzz}");
	}

	public void testParameter115() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqp(env, "${aaa:+zzz}");
	}

	public void testParameter116() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "bbb");
		eqp(env, "${aaa:+}");
	}

	public void testParameter117() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:0:5}", "12345");
	}

	public void testParameter118() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:0:2}", "12");
	}

	public void testParameter119() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:2:2}", "34");
	}

	public void testParameter120() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:0:6}", "12345");
	}

	public void testParameter121() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:0:0}");
	}

	public void testParameter122() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:5:3}");
	}

	public void testParameter123() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "12345");
		eqp(env, "${aaa:4:6}", "5");
	}

	public void testParameter124() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka/bin/idolmaster");
		eqp(env, "${aaa##*/}", "idolmaster");
	}

	public void testParameter125() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka/bin/idolmaster");
		eqp(env, "${aaa##*/idol}", "master");
	}

	public void testParameter126() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa##/*/haruka/*/}", "idolmaster");
	}

	public void testParameter127() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa##/*/haruka/*/}", "idolmaster");
	}

	public void testParameter128() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa##/*/haruka/*}");
	}

	public void testParameter129() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/chihaya/home/bin/idolmaster");
		eqp(env, "${aaa##/*/haruka/*}", "/home/chihaya/home/bin/idolmaster");
	}

	public void testParameter130() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/bin/idolmaster");
		eqp(env, "${aaa#*/}", "haruka/bin/idolmaster");
	}

	public void testParameter131() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka/bin/idolmaster");
		eqp(env, "${aaa#*/}", "home/haruka/bin/idolmaster");
	}

	public void testParameter132() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa#/*/haruka/*/}", "bin/idolmaster");
	}

	public void testParameter133() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa#/*/haruka/*/}", "bin/idolmaster");
	}

	public void testParameter134() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/home/haruka/home/bin/idolmaster");
		eqp(env, "${aaa#/*/haruka/*}", "home/bin/idolmaster");
	}

	public void testParameter135() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa##/*/haruka/*}");
	}

	public void testParameter136() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa#/*/haruka/*}");
	}

	public void testParameter137() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka");
		eqp(env, "${aaa##}", "/home/haruka");
	}

	public void testParameter138() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka");
		eqp(env, "${aaa#}", "/home/haruka");
	}

	public void testParameter139() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/sh.bin/idolmaster.sh");
		eqp(env, "${aaa%.*}", "home/haruka/sh.bin/idolmaster");
	}

	public void testParameter140() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/bin/idolmaster");
		eqp(env, "${aaa%.*}", "home/haruka/bin/idolmaster");
	}

	public void testParameter141() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa%.*}");
	}

	public void testParameter142() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/bin/idolmaster");
		eqp(env, "${aaa%*}", "home/haruka/bin/idolmaster");
	}

	public void testParameter143() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/sh.bin/idolmaster.sh");
		eqp(env, "${aaa%%.*}", "home/haruka/sh");
	}

	public void testParameter144() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/bin/idolmaster");
		eqp(env, "${aaa%%.*}", "home/haruka/bin/idolmaster");
	}

	public void testParameter145() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "");
		eqp(env, "${aaa%%.*}");
	}

	public void testParameter146() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "home/haruka/bin/idolmaster");
		eqp(env, "${aaa%%*}");
	}

	public void testParameter147() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka");
		eqp(env, "${aaa%}", "/home/haruka");
	}

	public void testParameter148() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("aaa", "/home/haruka");
		eqp(env, "${aaa%%}", "/home/haruka");
	}

	public void testCommand001() {
		eqc("aaaa", "aaaa");
	}

	public void testCommand002() {
		eqc("$(whoami)", "haruka");
	}

	public void testCommand003() {
		eqc("$(whoami)aaa", "harukaaaa");
	}

	public void testCommand004() {
		eqc("aaa$(whoami)", "aaaharuka");
	}

	public void testCommand005() {
		eqc("'$(whoami)'", "'$(whoami)'");
	}

	public void testCommand006() {
		eqc("\"$(whoami)aaa\"", "\"harukaaaa\"");
	}

	public void testCommand007() {
		eqc("\"$whoamiaaa\"", "\"$whoamiaaa\"");
	}

	public void testCommand008() {
		eqc("$(whoami", "$(whoami");
	}

	public void testCommand009() {
		eqc("${whoami", "${whoami");
	}

	public void testCommand010() {
		eqc("${whoami}", "${whoami}");
	}

	public void testCommand011() {
		eqc("\\$(whoami)", "\\$(whoami)");
	}

	public void testCommand012() {
		eqc("\"\\$(whoami)\"", "\"\\$(whoami)\"");
	}

	public void testCommand013() {
		eqc("aa\\$(whoami)", "aa\\$(whoami)");
	}

	public void testCommand014() {
		eqc("\\", "\\");
	}

	public void testCommand015() {
		eqc("\\'$(whoami)", "\\'haruka");
	}

	public void testCommand016() {
		eqc("$((1 + 2))", "$((1 + 2))");
	}

	public void testCommand017() {
		eqc("$(($(whoami) + 2))", "$((haruka + 2))");
	}

	public void testCommand018() {
		eqc("$(($(whoami) + $))", "$((haruka + $))");
	}

	public void testCommand019() {
		eqc("$(($ + $(whoami)))", "$(($ + haruka))");
	}

	public void testSplit001() {
		eql("aaaa", " ", "aaaa");
	}

	public void testSplit002() {
		eql("aaa  aaa", " ", "aaa", "aaa");
	}

	public void testSplit003() {
		eql("  aaa  aaa", " ", "aaa", "aaa");
	}

	public void testSplit004() {
		eql("aaa  aaa  ", " ", "aaa", "aaa");
	}

	public void testSplit005() {
		eql("'\"aaa  aaa\"'", " ", "'\"aaa  aaa\"'");
	}

	public void testSplit006() {
		eql("\"'aaa  aaa'\"", " ", "\"'aaa  aaa'\"");
	}

	public void testSplit007() {
		eql("'\"aaa  aaa\"'aa aa", " ", "'\"aaa  aaa\"'aa", "aa");
	}

	public void testSplit008() {
		eql("\"'aaa  aaa'\"aa aa", " ", "\"'aaa  aaa'\"aa", "aa");
	}

	public void testSplit009() {
		eql("aa'\"aaa  aaa\"'", " ", "aa'\"aaa  aaa\"'");
	}

	public void testSplit010() {
		eql("aa\"'aaa  aaa'\"", " ", "aa\"'aaa  aaa'\"");
	}

	public void testSplit011() {
		eql("aa\"\\\"\"", " ", "aa\"\\\"\"");
	}

	public void testSplit012() {
		eql("aa\\\"aa", " ", "aa\\\"aa");
	}

	public void testSplit013() {
		eql("aa\\\'aa", " ", "aa\\\'aa");
	}

	public void testIsWildcard001() {
		okw("*");
		okw("aaa*aaa");
		okw("???.txt");
		okw("[a-z].txt");
	}

	public void testIsWildcard002() {
		ngw("\\*");
		ngw("aaa");
		ngw("\"*\"");
		ngw("'*'");
	}

	public void testArithmetic001() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqa(env, "$((1 + 2))", "3");
	}

	public void testArithmetic002() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqa(env, "1$((1 + 1))3", "123");
	}

	public void testArithmetic003() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqa(env, "\"1$((1 + 1))3\"", "\"123\"");
	}

	public void testArithmetic004() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqa(env, "'1$((1 + 1))3'", "'1$((1 + 1))3'");
	}

	public void testArithmetic005() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		eqa(env, "1\\$((1 + 1))3", "1\\$((1 + 1))3");
	}

	public void testArithmetic006() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("var", "72");
		eqa(env, "$((var + 1))", "73");
	}

	public void testTilde001() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("HOME", "/home/haruka");
		eqt(env, "~", "/home/haruka");
	}

	public void testTilde002() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("HOME", "/home/haruka");
		eqt(env, "~/home", "/home/haruka/home");
	}

	public void testTilde003() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("HOME", "/home/haruka");
		eqt(env, "~/", "/home/haruka/");
	}

	public void testTilde004() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("HOME", "/home/haruka");
		eqt(env, "/~", "/~");
	}

	public void testTilde005() {
		ShEnvironment env = new ShRootShellEnvironment(new ShRootEnvStub());

		env.bind("HOME", "/home/haruka");
		eqt(env, "/~/", "/~/");
	}

}
