/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.unix.charset;

import java.util.Collection;

/**
 * int型の範囲を表すインターフェースです。
 * 
 *
 * @author MORIGUCHI, Yuichiro 2011/10/09
 */
public interface IntRange extends Iterable<Integer> {

	/**
	 * 空集合です。
	 */
	public static final IntRange O = new IntInterval(false);

	/**
	 * 全体集合です。
	 */
	public static final IntRange UINT =
		new IntInterval(Integer.MIN_VALUE, Integer.MAX_VALUE);

	/**
	 * 値がこの範囲に含まれるときtrueを得ます。
	 * 
	 * @param x 調べたい値
	 * @return 含まれるときtrue
	 */
	public boolean contains(int x);

	/**
	 * この範囲が範囲に含まれるときtrueを得ます。
	 * 
	 * @param r 調べたい範囲
	 * @return 含まれるときtrue
	 */
	public boolean isContained(IntRange r);

	/**
	 * 与えられた範囲がこの範囲と独立のときtrueを得ます。
	 * 
	 * @param r 調べたい範囲
	 * @return 独立のときtrue
	 */
	public boolean isIndependent(IntRange r);

	/**
	 * 与えられた範囲がこの範囲と重なるときtrueを得ます。
	 * 
	 * @param r 調べたい範囲
	 * @return 重なるときtrue
	 */
	public boolean isOverlapped(IntRange r);

	/**
	 * 範囲がこの範囲に含まれるときtrueを得ます。
	 * 
	 * @param r 調べたい範囲
	 * @return 含まれるときtrue
	 */
	public boolean containsAll(IntRange r);

	/**
	 * 範囲を構成する区間の1つがこの範囲を構成する区間の1つに
	 * 含まれるときtrueを得ます。
	 * 
	 * @param r 調べたい範囲
	 * @return 含まれるときtrue
	 */
	public boolean containsAny(IntRange r);

	/**
	 * 和集合(結び)を求めます。
	 * 
	 * @param r 区間
	 * @return 和集合
	 */
	public IntRange join(IntRange r);

	/**
	 * 共通部分(交わり)を求めます。
	 * 
	 * @param r 区間
	 * @return 共通部分
	 */
	public IntRange meet(IntRange r);

	/**
	 * 差集合を求めます。
	 * 
	 * @param r 区間
	 * @return 差集合
	 */
	public IntRange difference(IntRange r);

	/**
	 * XOR集合を求めます。
	 * 
	 * @param r 区間
	 * @return XOR集合
	 */
	public IntRange xor(IntRange r);

	/**
	 * 補集合を求めます。
	 * 
	 * @return 補集合
	 */
	public IntRange complement();

	/**
	 * この範囲の最小値を求めます。
	 * 
	 * @return 最小値
	 */
	public int minimum();

	/**
	 * この範囲の最大値を求めます。
	 * 
	 * @return 最大値
	 */
	public int maximum();

	/**
	 * この範囲を構成する区間の集合を得ます。
	 * 
	 * @return 区間の集合
	 */
	public Collection<IntInterval> intervals();

	/**
	 * この範囲の要素数を求めます。
	 * 
	 * @return 要素数
	 */
	public long cardinality();

}
