/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.value;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.regex.Pattern;

import net.morilib.awk.namespace.AwkNamespace;

public class AwkFloat extends AwkValue {

	/**
	 * 
	 */
	public static final AwkFloat NaN = new AwkFloat(Double.NaN);

	//
	private double value;

	private AwkFloat(double d) {
		value = d;
	}

	public static AwkFloat valueOf(double d) {
		return new AwkFloat(d);
	}

	public double toFloat() {
		return value;
	}

	public BigInteger toInteger() {
		return new BigDecimal(value).toBigInteger();
	}

	public boolean toBoolean() {
		return value != 0.0;
	}

	public Number toReal() {
		return new Double(value);
	}

	public Pattern toRegex() {
		return Pattern.compile(Double.toString(value));
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.value.AwkValue#toObject()
	 */
	public Object toObject() {
		return new Double(value);
	}

	public boolean isFloat() {
		return true;
	}

	public boolean isInteger() {
		return false;
	}

	public boolean isReal() {
		return true;
	}

	public boolean isFloatValue() {
		return true;
	}

	public boolean isIntegerValue() {
		try {
			new BigDecimal(value).toBigIntegerExact();
			return true;
		} catch(ArithmeticException e) {
			return false;
		}
	}

	public boolean isRealValue() {
		return true;
	}

	@Override
	public boolean isNegativeValue() {
		return value < 0;
	}

	@Override
	public boolean isPositiveValue() {
		return value > 0;
	}

	@Override
	public boolean isZeroValue() {
		return value == 0;
	}

	public boolean isString() {
		return false;
	}

	public boolean isRegex() {
		return false;
	}

	public boolean isFunction() {
		return false;
	}

	@Override
	public boolean isEmpty() {
		return value == 0.0;
	}

	public int hashCode() {
		return (int)Double.doubleToRawLongBits(value);
	}

	public boolean equals(Object o) {
		if(o instanceof AwkFloat) {
			return value == ((AwkFloat)o).value;
		}
		return false;
	}

	public String toString(AwkNamespace ns) {
		AwkValue v;

		if(ns == null || (v = ns.getRoot().find("OFMT")) == null) {
			return Double.toString(value);
		} else {
			return String.format(v.toString(), value);
		}
	}

	public int compareTo(AwkValue o, AwkNamespace ns) {
		if(o instanceof AwkUndefined) {
			return 1;
		} else if(o.isFloat() || o.isInteger()) {
			return value < o.toFloat() ?
					-1 : value > o.toFloat() ? 1 : 0;
		} else {
			return toString(ns).compareTo(o.toString(ns));
		}
	}

}
