﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Text.RegularExpressions;
using System.Net;
using System.IO;
using Codeplex.Data;

namespace AssistRB
{
    /// <summary>
    /// リネーム候補のリストのアイテム
    /// </summary>
    public class RenameItem
    {
        public string Id { get; set; }
        public string OldName { get; set; }
        public string NewName { get; set; }
    }

    /// <summary>
    /// フォルダ自動振り分けのリストのアイテム
    /// </summary>
    public class DispatchItem
    {
        public string Id { get; set; }
        public string Title { get; set; }
        public string Parent { get; set; }
        public string Series { get; set; }
    }

    /// <summary>
    /// 特定フォルダへの振り分けのリストのアイテム
    /// </summary>
    public class DustboxItem
    {
        public string Id { get; set; }
        public string Title { get; set; }
        public string Parent { get; set; }
    }

    /// <summary>
    /// RECBOXとの主な通信を行う
    /// </summary>
    class RecboxController
    {
        // スレッド停止のためのフラグ
        private volatile bool stop = false;

        // 要求されている処理の種類
        private enum RequestType
        {
            REQUEST_NONE = 0,
            REQUEST_UPDATE_LIST,
            REQUEST_RENAME_CHECK,
            REQUEST_RENAME_EXEC,
            REQUEST_DISPATCH_CHECK,
            REQUEST_DISPATCH_EXEC,
            REQUEST_DUSTBOX_CHECK,
            REQUEST_DUSTBOX_EXEC,
            REQUEST_ALLTRANSFER_EXEC,
            REQUEST_ALLTRANSFER_GETLIST
        }
        private RequestType currentRequest;
        private Object requestTypeLocker = new object();
        private RequestType CurrentRequest
        {
            get
            {
                lock (requestTypeLocker)
                {
                    return this.currentRequest;
                }
            }

            set
            {
                lock (requestTypeLocker)
                {
                    if (currentRequest != value)
                    {
                        currentRequest = value;
                    }
                }
            }
        }

        // 各機能のパラメータ
        private string host_url = "";
        private string renaempattern = "";
        private bool renameIncludeDir = false;
        private string dispatchpattern = "";
        private string dustboxpattern = "";
        private bool dustboxIncludeDir = false;
        private string dustboxfolder = "";
        private string udn = "";
        private string storage = "";

        // 各種イベント
        public delegate void RecboxEventHandler();
        public event RecboxEventHandler RenameChanged;
        public event RecboxEventHandler AllItemListChanged;
        public event RecboxEventHandler DispatchChanged;
        public event RecboxEventHandler DustboxChanged;
        public event RecboxEventHandler ProcessDone;
        public event RecboxEventHandler ServerListChanged;

        // 生成されるフィルタリングされたリスト
        public List<RenameItem> RenameList = new List<RenameItem>();
        public List<DispatchItem> DispatchList = new List<DispatchItem>();
        public List<DustboxItem> DustboxList = new List<DustboxItem>();
        public List<WebLowlayer.ServerList> ServerList = new List<WebLowlayer.ServerList>();

        public RecboxController()
        {
        }

        /// <summary>
        /// スレッドの停止を要求する
        /// </summary>
        public void Stop()
        {
            this.stop = true;
        }

        // 処理状況の監視スレッドから更新されます
        private bool transferProcessing = true;
        public void ChangeTransferProcessing(bool value)
        {
            transferProcessing = value;
        }

        public void RequestUpdateList(string url)
        {
            this.host_url = url;
            this.CurrentRequest = RequestType.REQUEST_UPDATE_LIST;
        }

        public void RequestRenameCheck(string pattern, bool includeDir)
        {
            this.renaempattern = pattern;
            this.renameIncludeDir = includeDir;
            this.CurrentRequest = RequestType.REQUEST_RENAME_CHECK;
        }

        public void RequestRename(string url)
        {
            this.host_url = url;
            this.CurrentRequest = RequestType.REQUEST_RENAME_EXEC;
        }

        public void RequestDispatchCheck(string pattern)
        {
            this.dispatchpattern = pattern;
            this.CurrentRequest = RequestType.REQUEST_DISPATCH_CHECK;
        }

        public void RequestDispatch(string url)
        {
            this.host_url = url;
            this.CurrentRequest = RequestType.REQUEST_DISPATCH_EXEC;
        }

        public void RequestDustboxCheck(string pattern, bool includeDir)
        {
            this.dustboxpattern = pattern;
            this.dustboxIncludeDir = includeDir;
            this.CurrentRequest = RequestType.REQUEST_DUSTBOX_CHECK;
        }

        public void RequestDustbox(string url, string folder)
        {
            this.host_url = url;
            this.dustboxfolder = folder;
            this.CurrentRequest = RequestType.REQUEST_DUSTBOX_EXEC;
        }

        public void RequestAllTransfer(string url, string udn, string storage)
        {
            this.host_url = url;
            this.udn = udn;
            this.storage = storage;
            this.CurrentRequest = RequestType.REQUEST_ALLTRANSFER_EXEC;
        }

        public void RequestAllTransferGetList(string url)
        {
            this.host_url = url;
            this.CurrentRequest = RequestType.REQUEST_ALLTRANSFER_GETLIST;
        }

        public void worker()
        {
            while (!this.stop)
            {
                if (this.host_url.Length == 0)
                {
                    this.CurrentRequest = RequestType.REQUEST_NONE;
                    continue;
                }

                TransferStatusChecker.GetInstance().Start();

                if (this.transferProcessing)
                {
                    Thread.Sleep(10);
                    continue;
                }

                RequestType request = CurrentRequest;
                switch (request)
                {
                    case RequestType.REQUEST_UPDATE_LIST:
                        {
                            this.UpdateAllItemList();

                            if (this.AllItemListChanged != null)
                            {
                                this.AllItemListChanged();
                            }
                            TransferStatusChecker.GetInstance().Stop();
                        }
                        break;

                    case RequestType.REQUEST_RENAME_CHECK:
                        {
                            RenameList.Clear();

                            CreateFilteredList(
                                this.renaempattern,
                                this.renameIncludeDir,
                                delegate(DlnaItem item, string matchTitle)
                                {
                                    if (matchTitle != item.Title)
                                    {
                                        RenameItem ren = new RenameItem();
                                        ren.Id = item.Id;
                                        ren.OldName = item.Title;
                                        ren.NewName = matchTitle;
                                        RenameList.Add(ren);
                                    }
                                });

                            ToolLog.GetInstance().Add("リネーム予定リストを更新しました");

                            if (this.RenameChanged != null)
                            {
                                this.RenameChanged();
                            }
                        }
                        break;

                    case RequestType.REQUEST_RENAME_EXEC:
                        {
                            Rename();
                        }
                        break;

                    case RequestType.REQUEST_DISPATCH_CHECK:
                        {
                            DispatchList.Clear();

                            CreateFilteredList(
                            this.dispatchpattern,
                            false,
                            delegate(DlnaItem item, string matchTitle)
                            {
                                string series = matchTitle;
                                series = series.Replace("*", "＊");
                                series = series.Replace(":", "：");
                                series = series.Replace("\\", "￥");
                                series = series.Replace("\"", "”");
                                series = series.Replace("<", "＜");
                                series = series.Replace(">", "＞");
                                series = series.Replace("?", "？");
                                series = series.Replace("|", "｜");
                                series = series.Replace("/", "／");

                                Dictionary<string, string> folderList = DlnaItemList.GetInstance().GetFolderList();
                                if (folderList.ContainsKey(item.Parent)
                                    && folderList[item.Parent] != series)
                                {
                                    DispatchItem ren = new DispatchItem();
                                    ren.Id = item.Id;
                                    ren.Parent = item.Parent;
                                    ren.Series = series;
                                    ren.Title = item.Title;
                                    DispatchList.Add(ren);
                                }
                            });

                            ToolLog.GetInstance().Add("移動予定リストを更新しました");

                            if (this.DispatchChanged != null)
                            {
                                this.DispatchChanged();
                            }
                        }
                        break;

                    case RequestType.REQUEST_DISPATCH_EXEC:
                        {
                            DispatchFolder();
                        }
                        break;

                    case RequestType.REQUEST_DUSTBOX_CHECK:
                        {
                            DustboxList.Clear();
                            CreateFilteredList(
                                this.dustboxpattern,
                                this.dustboxIncludeDir,
                                delegate(DlnaItem item, string matchTitle)
                                {
                                    if (matchTitle != item.Title)
                                    {
                                        DustboxItem ren = new DustboxItem();
                                        ren.Id = item.Id;
                                        ren.Parent = item.Parent;
                                        ren.Title = item.Title;
                                        DustboxList.Add(ren);
                                    }
                                });

                            ToolLog.GetInstance().Add("移動予定リストを更新しました");

                            if (this.DustboxChanged != null)
                            {
                                this.DustboxChanged();
                            }
                        }
                        break;

                    case RequestType.REQUEST_DUSTBOX_EXEC:
                        {
                            DustboxFolder();
                        }
                        break;

                    case RequestType.REQUEST_ALLTRANSFER_EXEC:
                        {
                            RenameList.Clear();
                            foreach (DlnaItem dlnaitem in DlnaItemList.GetInstance().GetList().FindAll(i => i.ClassType != "container"))
                            {
                                RenameItem item = new RenameItem();
                                item.Id = dlnaitem.Id;
                                item.OldName = dlnaitem.Title;
                                RenameList.Add(item);
                            }
                            AllTransfer();
                        }
                        break;
                    case RequestType.REQUEST_ALLTRANSFER_GETLIST:
                        {
                            ServerList = WebLowlayer.GetUdnList(this.host_url);
                            if (ServerListChanged != null)
                            {
                                ServerListChanged();
                            }
                        }
                        break;
                }

                TransferStatusChecker.GetInstance().Stop();

                // 終了のイベントを発行
                if (request != RequestType.REQUEST_NONE)
                {
                    this.CurrentRequest = RequestType.REQUEST_NONE;

                    if (this.ProcessDone != null)
                    {
                        this.ProcessDone();
                    }
                }

                Thread.Sleep(10);
            }
        }

        /* *************************************************************************
         * ディスパッチャから呼ばれる高レベル処理
         */
        private delegate void CreateFilteredListAddItemDelegate(DlnaItem item, string matchTitle);
        /// <summary>
        /// 全件検索の結果をフィルタリングして処理用のリストを作ります
        /// </summary>
        /// <param name="pattern"></param>
        /// <param name="includeDir"></param>
        /// <param name="addItem"></param>
        /// <returns></returns>
        private bool CreateFilteredList(string pattern, bool includeDir, CreateFilteredListAddItemDelegate addItem)
        {
            Regex regex;
            try
            {
                regex = new Regex(pattern);
            }
            catch (Exception)
            {
                ToolLog.GetInstance().Add("正規表現の表記が不正です");
                return false;
            }

            foreach (DlnaItem item in DlnaItemList.GetInstance().GetList())
            {
                if (item.ClassType != "item" && item.ClassType != "container")
                {
                    continue;
                }

                if (!includeDir && item.ClassType == "container")
                {
                    continue;
                }

                Match match = regex.Match(item.Title);
                if (match.Success)
                {
                    addItem(item, match.Groups["title"].Value);
                }
            }

            return true;
        }

        private void Rename()
        {
            foreach (RenameItem item in RenameList)
            {
                ToolLog.GetInstance().Add(string.Format(
                    "{0}を{1}にリネームします", item.OldName, item.NewName));

                WebLowlayer.Rename(host_url, item.Id, item.NewName);
            }

            ToolLog.GetInstance().Add("リネームが完了しました");
        }

        private void AllTransfer()
        {
            foreach (RenameItem item in RenameList)
            {
                while (this.transferProcessing)
                {
                    if (this.stop)
                    {
                        return;
                    }

                    Thread.Sleep(10);
                }

                ToolLog.GetInstance().Add(string.Format(
                    "{0}を転送します", item.OldName));

                string[] transferitems = { item.Id };

                WebLowlayer.Transfer(
                    host_url,
                    udn,
                    storage,
                    transferitems);

                UpdateAllItemList();
            }

            ToolLog.GetInstance().Add("転送が完了しました");
        }

        private void DispatchFolder()
        {
            Dictionary<string, string> folderList = DlnaItemList.GetInstance().GetFolderList();

            while (DispatchList.Count > 0)
            {
                string firstparent = "";
                string series = "";
                List<string> ids = new List<string>();

                foreach (DispatchItem item in DispatchList)
                {
                    if (this.stop)
                    {
                        return;
                    }

                    if (item.Series.Length == 0)
                    {
                        continue;
                    }

                    // 集めている以外のシリーズは次回以降
                    if (series.Length > 0 && (series != item.Series || firstparent != item.Parent))
                    {
                        continue;
                    }

                    string folderName = folderList[item.Parent];
                    if (folderName != item.Series)
                    {
                        if (ids.Count == 0)
                        {
                            series = item.Series;
                            firstparent = item.Parent;
                        }

                        ids.Add(item.Id);
                    }

                    if (ids.Count >= 20)
                    {
                        break;
                    }
                }

                if (ids.Count > 0)
                {
                    string[] idsArray = ids.ToArray();

                    ToolLog.GetInstance().Add(string.Format("{0}を{1}件、移動します", series, idsArray.Count()));

                    string parent = "";
                    if (folderList.ContainsValue(series))
                    {
                        parent = folderList.First(a => a.Value == series).Key;
                    }
                    else
                    {
                        // フォルダがないので作成する
                        ToolLog.GetInstance().Add(string.Format("フォルダを作成します"));
                        if (!WebLowlayer.MkDir(host_url, firstparent, series))
                        {
                            ToolLog.GetInstance().Add(string.Format("フォルダの作成に失敗しました({0})", series));
                        }
                        else
                        {
                            ToolLog.GetInstance().Add(string.Format("フォルダを作成しました"));

                            DlnaItemList.GetInstance().Invalidate(firstparent);
                            UpdateAllItemList(true);

                            parent = folderList.First(a => a.Value == series).Key;
                        }
                    }

                    if (parent.Length > 0)
                    {
                        WebLowlayer.Move(host_url, parent, idsArray);

                        DlnaItemList.GetInstance().Invalidate(parent);
                        DlnaItemList.GetInstance().Invalidate(firstparent);
                        UpdateAllItemList(true);

                        ToolLog.GetInstance().Add("移動が完了しました");
                    }

                    // 失敗しても取り除く
                    foreach (string id in ids)
                    {
                        DispatchList.RemoveAll(a => a.Id == id);
                    }
                }
            }
        }

        private void DustboxFolder()
        {
            Dictionary<string, string> folderList = DlnaItemList.GetInstance().GetFolderList();

            while (DustboxList.Count > 0)
            {
                string firstparent = "";
                List<string> ids = new List<string>();

                foreach (DustboxItem item in DustboxList)
                {
                    if (this.stop)
                    {
                        return;
                    }

                    // 同じ親以外は次回以降に
                    if (ids.Count != 0 && firstparent != item.Parent)
                    {
                        continue;
                    }

                    string folderName = folderList[item.Parent];
                    if (ids.Count == 0)
                    {
                        firstparent = item.Parent;
                    }

                    ids.Add(item.Id);

                    if (ids.Count >= 20)
                    {
                        break;
                    }
                }

                if (ids.Count > 0)
                {
                    string[] idsArray = ids.ToArray();

                    ToolLog.GetInstance().Add(string.Format("{0}件、移動します", idsArray.Count()));

                    string parent = "";
                    if (folderList.ContainsValue(dustboxfolder))
                    {
                        parent = folderList.First(a => a.Value == dustboxfolder).Key;
                    }
                    else
                    {
                        ToolLog.GetInstance().Add(string.Format("フォルダを作成します"));
                        if (!WebLowlayer.MkDir(host_url, firstparent, dustboxfolder))
                        {
                            ToolLog.GetInstance().Add(string.Format("フォルダの作成に失敗しました({0})", dustboxfolder));
                        }
                        else
                        {
                            ToolLog.GetInstance().Add(string.Format("フォルダを作成しました"));

                            DlnaItemList.GetInstance().Invalidate(firstparent);
                            UpdateAllItemList();

                            parent = folderList.First(a => a.Value == dustboxfolder).Key;
                        }
                    }

                    if (parent.Length > 0)
                    {
                        WebLowlayer.Move(host_url, parent, idsArray);

                        DlnaItemList.GetInstance().Invalidate(parent);
                        DlnaItemList.GetInstance().Invalidate(firstparent);
                        UpdateAllItemList();

                        ToolLog.GetInstance().Add("移動が完了しました");
                    }

                    // 失敗しても取り除く
                    foreach (string id in ids)
                    {
                        DustboxList.RemoveAll(a => a.Id == id);
                    }
                }
            }
        }

        private void UpdateAllItemList(bool fastFlag = false)
        {
            Thread.Sleep(3000);
            while (this.transferProcessing)
            {
                if (this.stop)
                {
                    return;
                }

                Thread.Sleep(10);
            }

            DlnaItemList dlist = DlnaItemList.GetInstance();
            List<DlnaItem> newList =
                this.UpdateListSub(
                dlist.GetFolderList(),
                dlist.GetInvalidList(),
                dlist.GetList(),
                fastFlag);
            DlnaItemList.GetInstance().UpdateList(newList);
        }

        private List<DlnaItem> UpdateListSub(
            Dictionary<string, string> afolderDictionary,
            List<string> ainvalidFolder,
            List<DlnaItem> prevList,
            bool fastFlag = false)
        {
            List<DlnaItem> resultList = new List<DlnaItem>(prevList);
            Dictionary<string, string> folderDictionary = new Dictionary<string, string>(afolderDictionary);
            List<string> invalidFolder = new List<string>(ainvalidFolder);

            ToolLog.GetInstance().Add("リストを更新します");

            // フォルダ辞書にROOTは追加しておく
            if (!folderDictionary.ContainsKey("FS-Folder"))
            {
                folderDictionary.Add("FS-Folder", "ROOT");
            }

            for (int i = 0; i < invalidFolder.Count; ++i)
            {
                // 停止要求がある場合は処理をやめる
                if (this.stop)
                {
                    break;
                }

                string target = invalidFolder[i];

                // これから探そうとしている階層の前回の結果を破棄する
                resultList.RemoveAll(a => a.Parent == target);

                ToolLog.GetInstance().Add(string.Format(
                    "フォルダ {0}/{1} {2} を検索中です",
                    i + 1, invalidFolder.Count, folderDictionary[target]));

                List<DlnaItem> newitems = WebLowlayer.UpdateFolder(host_url, target);
                foreach (DlnaItem item in newitems)
                {
                    if (item.ClassType == "container")
                    {
                        // infoは操作できるものではないので除外する
                        if (item.Parent == "FS-Folder" && item.Title == "info")
                        {
                            continue;
                        }

                        // フォルダ辞書にある場合は破棄して登録しなおす
                        if (folderDictionary.ContainsKey(item.Id))
                        {
                            folderDictionary.Remove(item.Id);
                        }
                        folderDictionary.Add(item.Id, item.Title); // 表示のため仮登録

                        // 見つけたフォルダ以下も取得しなおす
                        if (!invalidFolder.Contains(item.Id))
                        {
                            if (!fastFlag)
                            {
                                invalidFolder.Add(item.Id);
                            }
                        }
                    }

                    resultList.Add(item);
                }
            }

            ToolLog.GetInstance().Add("リストを更新しました");

            return resultList;
        }

    }
}
