﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.IO;
using System.Diagnostics;
using System.Threading;

namespace AssistRB
{
    /// <summary>
    /// オンラインのRECBOXを探します
    /// </summary>
    public class FindRecbox
    {
        /// <summary>
        /// 発見したRECBOXを表します
        /// </summary>
        public class RecboxInfo : IEquatable<RecboxInfo>
        {
            private string mName;
            private byte[] mMac;
            private IPAddress mAddress;
            private IPAddress mMask;
            private IPAddress mRoute;
            private UInt16 mPort;

            public string Name
            {
                get { return mName; }
            }
            public byte[] Mac
            {
                get { return mMac; }
            }
            public IPAddress Address
            {
                get { return mAddress; }
            }
            public IPAddress Mask
            {
                get { return mMask; }
            }
            public IPAddress Route
            {
                get { return mRoute; }
            }
            public UInt16 Port
            {
                get { return mPort; }
            }

            public RecboxInfo(
                string name,
                byte[] mac,
                byte[] address,
                byte[] mask,
                byte[] route,
                UInt16 port)
            {
                Debug.Assert(mac.Length == 6);
                Debug.Assert(address.Length == 4);
                Debug.Assert(mask.Length == 4);
                Debug.Assert(route.Length == 4);

                this.mName = name;
                this.mMac = (byte[])(mac.Clone());
                this.mAddress = new IPAddress(address);
                this.mMask = new IPAddress(mask);
                this.mRoute = new IPAddress(route);
                this.mPort = port;

            }

            public bool Equals(RecboxInfo other)
            {
                if (this.mName != other.mName) return false;
                if (this.mMac.Length != other.mMac.Length) return false;
                for (int i = 0; i < this.mMac.Length; ++i)
                {
                    if (this.mMac[i] != other.mMac[i]) return false;
                }
                if (!this.mAddress.Equals(other.mAddress)) return false;
                if (!this.mMask.Equals(other.mMask)) return false;
                if (!this.mRoute.Equals(other.mRoute)) return false;
                if (this.mPort != other.mPort) return false;
                return true;
            }
        };

        private List<RecboxInfo> mRecboxList = new List<RecboxInfo>();

        private UInt32 requestSeq { get; set; }
        private byte[] request = {
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x46, 0x52, 0x4f, 0x4d, 0x2d, 0x50,
                    0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x49, 0x4f, 0x2d, 0x44, 0x41,
                    0x54, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00
        };

        /// <summary>
        /// コンストラクタです
        /// </summary>
        public FindRecbox()
        {
            Random rand = new Random();
            requestSeq = (UInt32)rand.Next(0x7fffffff);
        }

        /// <summary>
        /// 発見したRECBOXのリストを返します
        /// </summary>
        /// <returns></returns>
        public List<RecboxInfo> GetRecboxList()
        {
            return mRecboxList;
        }

        /// <summary>
        /// 実際に検索します
        /// </summary>
        /// <returns>発見した数を返します。エラーが発生した場合は負数を返します。</returns>
        public int Search()
        {
            int count = 0;
            const double SearchTimeout = 5; // seconds

            mRecboxList.Clear();

            try
            {
                UdpClient udpclient = new UdpClient();

                IPEndPoint broadcast = new IPEndPoint(IPAddress.Broadcast, 65);
                udpclient.Connect(broadcast);

                request[1] = (byte)((requestSeq / 0x1000000) % 0x100);
                request[2] = (byte)((requestSeq / 0x10000) % 0x100);
                request[3] = (byte)((requestSeq / 0x100) % 0x100);
                request[4] = (byte)(requestSeq % 0x100);
                ++requestSeq;

                udpclient.Send(request, request.Length);
                udpclient.Send(request, request.Length);
                udpclient.Send(request, request.Length);

                DateTime limitTime = DateTime.Now.AddSeconds(SearchTimeout);

                State state = new State();
                state.e = new IPEndPoint(IPAddress.Any, 65);
                state.u = new UdpClient(state.e);

                while (limitTime > DateTime.Now)
                {
                    recieved = false;
                    state.u.BeginReceive(new AsyncCallback(processUdpResponse), state);

                    while (limitTime > DateTime.Now && !recieved)
                    {
                        Thread.Sleep(100);
                    }

                }

                udpclient.Close();

                count = mRecboxList.Count;
            }
            catch (Exception)
            {
                count = -1;
            }

            return count;
        }

        /// <summary>
        /// 非同期処理用
        /// </summary>
        private struct State
        {
            public IPEndPoint e;
            public UdpClient u;
        };

        /// <summary>
        /// 非同期処理用
        /// </summary>
        private bool recieved;

        /// <summary>
        /// 非同期処理用
        /// </summary>
        private void processUdpResponse(IAsyncResult result)
        {
            UdpClient u = (UdpClient)((State)(result.AsyncState)).u;
            IPEndPoint e = (IPEndPoint)((State)(result.AsyncState)).e;

            byte[] recieve = u.EndReceive(result, ref e);

            if (recieve.Length == 132)
            {
                const int IPAddressOffset = 14;
                const int NetMaskOffset = 18;
                const int DefaultOffset = 22;
                const int PortOffset = 26;
                const int MACOffset = 28;
                const int NameOffset = 34;
                const int NameMaxLength = 80;
                const int VendorOffset = 115;
                const int VendorMaxLength = 16;

                string name = Encoding.ASCII.GetString(recieve, NameOffset, NameMaxLength);
                name = name.TrimEnd('\0');
                string vendor = Encoding.ASCII.GetString(recieve, VendorOffset, VendorMaxLength);
                vendor = vendor.TrimEnd('\0');

                byte[] mac = new byte[6];
                for (int i = 0; i < 6; ++i)
                {
                    mac[i] = recieve[MACOffset + i];
                }

                byte[] address = new byte[4];
                for (int i = 0; i < 4; ++i)
                {
                    address[i] = recieve[IPAddressOffset + i];
                }

                byte[] mask = new byte[4];
                for (int i = 0; i < 4; ++i)
                {
                    mask[i] = recieve[NetMaskOffset + i];
                }

                byte[] route = new byte[4];
                for (int i = 0; i < 4; ++i)
                {
                    route[i] = recieve[DefaultOffset + i];
                }

                UInt16 port = (UInt16)(recieve[PortOffset] * 0x100 + recieve[PortOffset + 1]);

                RecboxInfo recboxInfo = new RecboxInfo(
                    name, mac, address, mask, route, port);

                if (!mRecboxList.Contains(recboxInfo))
                {
                    mRecboxList.Add(recboxInfo);
                }
            }

            recieved = true;
        }
    }
}
