/*
 * Copyright (c) 2000 Blue Mug, Inc.  All Rights Reserved.
 */

#include <target/assert.h>
#include <target/crc.h>
#include <target/command.h>
#include <target/herrno.h>
#include <target/io.h>
#include <ep7211/ioregs.h>
#include "eth.h"

typedef enum medium_type {
	MEDIUM_SERIAL = 0,
	MEDIUM_ETHERNET
} medium_t;
static medium_t current_medium;

/* buffer used to read commands from the Ethernet interface */
static unsigned char ethdata [128];
static int ethdata_count, ethdata_pos;

int hgetchar(void)
{
	if (current_medium == MEDIUM_SERIAL) {
		while (IO_SYSFLG2 & URXFE2);
		return IO_UARTDR2 & 0xff;
	}
	/* ethernet commands */
	if (ethdata_pos == ethdata_count) {
		/* read a packet from the Ethernet */
		ethdata_count = ether_read(ethdata, sizeof ethdata);
		assert(ethdata_count > 0);
		assert(ethdata_count <= sizeof ethdata);
		ethdata_pos = 0;
	}
	return ethdata[ethdata_pos++];
}

int hputchar(int c)
{
	while (IO_SYSFLG2 & UTXFF2);
	IO_UARTDR2 = (unsigned char) c;
	return 0;
}

int hgetblock(unsigned char *dst, int count)
{
	int remain = count;
	if (current_medium == MEDIUM_SERIAL) {
		while (remain--)
			*dst++ = hgetchar();
	} else {
		assert(current_medium == MEDIUM_ETHERNET);
		while (remain > 0) {
			int nread = ether_read(dst, remain);
			assert(nread <= remain);
			dst += nread;
			remain -= nread;
		}
	}
	return count;
}

/*
 * This works for now, since medium only affects downstream data, not
 * replies from the target to the host.  Once we add upstream media
 * switching, we'll have to implement a delayed media switch that
 * allows the successful command return value to go out along the old
 * medium.
 */
static int medium_cmdfunc(int argc, char *argv[])
{
	medium_t medium;
	
	if (argc != 2)
		return -H_EUSAGE;
	++argv;
	if (!strcmp(*argv, "ethernet"))
		medium = MEDIUM_ETHERNET;
	else if (!strcmp(*argv, "serial"))
		medium = MEDIUM_SERIAL;
	else
		return -H_EINVAL;
	if (medium == current_medium)
		return 0;
	if (medium == MEDIUM_ETHERNET) {
		if (ether_init() < 0)
			return -H_EIO;
		IO_LEDFLSH = 0x5c;	/* LED 50% duty cycle */
	} else if (medium == MEDIUM_SERIAL) {
		IO_LEDFLSH = 0x7c;	/* LED continuous on */
	}
	current_medium = medium;
	return 0;
}

const command_t medium_command =
	{ "medium", "<medium name>",
	  "select communication medium",
	  &medium_cmdfunc };

