/*
 * Copyright (c) 2000 Blue Mug, Inc.  All Rights Reserved.
 */

#include <target/assert.h>
#include <target/io.h>
#include <ep7211/memcpy.h>

#include "gunzip.h"
#include "lcd.h"
#include "memmap.h"
#include "memzero.h"

/* leave 64K reserved for the boot loader (generous) */
#define LINUX_FLASH_ADDRESS	(64 * 1024)
#define LINUX_PARAM_ADDRESS	(DRAM1_START + 0x00020000)
#define LINUX_LOAD_ADDRESS	(DRAM1_START + 0x00028000)

/* declarations for Linux kernel gzip */
#define NULL ((void*) 0)
#define OF(args) args
#define STATIC static

typedef unsigned char  uch;
typedef unsigned short ush;
typedef unsigned long  ulg;

/* sliding window size; must be at least 32K, and a power of two */
#define WSIZE 0x8000

/* diagnostic functions; not used here */
#ifdef DEBUG_GUNZIP
#  define Assert(cond,msg) {if(!(cond)) error(msg);}
#  define Trace(x) fprintf x
#  define Tracev(x) {if (verbose) fprintf x ;}
#  define Tracevv(x) {if (verbose>1) fprintf x ;}
#  define Tracec(c,x) {if (verbose && (c)) fprintf x ;}
#  define Tracecv(c,x) {if (verbose>1 && (c)) fprintf x ;}
#else
#  define Assert(cond,msg)
#  define Trace(x)
#  define Tracev(x)
#  define Tracevv(x)
#  define Tracec(c,x)
#  define Tracecv(c,x)
#endif

/* variables accessed/referenced by inflate.c code */
static uch *window;		/* sliding window buffer */
static unsigned outcnt;		/* bytes in output buffer */
static ulg bytes_out;		/* total bytes output */

/* support functions for inflate.c code */
static void flush_window(void);
static void free(void *where);
#define STRING_ERRORS
#ifdef STRING_ERRORS
static void error(const char *message);
#else
#define error(m)
#endif
static void gzip_mark(void **);
static void gzip_release(void **);
static void *malloc(int size);

static uch *hermit_input_ptr;	/* address of next input byte */
extern __inline__ uch get_byte(void) { return *hermit_input_ptr++; }
extern __inline__ void put_byte(void) { --hermit_input_ptr; }

/* include gunzip implementation from Linux kernel */
#include <target/inflate.c>

/* variables not used by inflate.c */
#define HERMIT_HEAP_SIZE 0x2000		/* size of heap for mini-allocator */
static ulg hermit_free_mem_ptr;		/* used by mini-allocator below */
static ulg hermit_free_mem_ptr_end;	/* likewise */
static uch *hermit_output_ptr;		/* decompress destination */
static int hermit_progress;		/* < 0 for no progress output;
					   otherwise keeps track of # of
					   times the decompression window
					   has been flushed, for output on
					   the LCD. */

/* for progress boxes on LCD */
#ifdef HAVE_LCD
static void progress(void);
#else
extern __inline__ void progress(void) { }
#endif

#ifdef STRING_ERRORS
static void error(const char *message)
{
	hprintf("\n\nERROR: %s\n\n -- System halted\n", message);
	while (1);
}
#endif

/* write the contents of the decompression window to output */
static void flush_window(void)
{
	uch *in = window;
	for (bytes_out += outcnt; outcnt; --outcnt) {
		uch ch = *hermit_output_ptr++ = *in++;
		crc = crc_32_tab[((int)crc ^ ch) & 0xff] ^ (crc >> 8);
	}
	if (hermit_progress >= 0) {
		hprintf(".");
		progress();
	}
}

static void *malloc(int size)
{
	void *p;

	assert(size >= 0 || hermit_free_mem_ptr > 0);
	hermit_free_mem_ptr = (hermit_free_mem_ptr + 3) & ~3;	/* align */
	assert(hermit_free_mem_ptr + size <= hermit_free_mem_ptr_end);

	p = (void*) hermit_free_mem_ptr;
	hermit_free_mem_ptr += size;
	return p;
}

static void free(void *where)
{
	/* gzip_mark and gzip_release do the free */
}

static void gzip_mark(void **ptr)
{
	*ptr = (void*) hermit_free_mem_ptr;
}

static void gzip_release(void **ptr)
{
	hermit_free_mem_ptr = (long) *ptr;
}

/* common initialization for the gunzip code */
static void hermit_gunzip_init(void)
{
	window = (uch*) (VIDEO_MEM_BASE + VIDEO_MEM_SIZE);
	bytes_out = 0;
	makecrc();
}

void gunzip_kernel(void)
{
	/* initialize variables for decompression */
	hermit_gunzip_init();
	hermit_input_ptr = (uch*) LINUX_FLASH_ADDRESS;
	hermit_free_mem_ptr = WSIZE + (ulg) window;
	hermit_free_mem_ptr_end = hermit_free_mem_ptr + HERMIT_HEAP_SIZE;
	hermit_output_ptr = (uch*) LINUX_LOAD_ADDRESS;

	/* decompress kernel image to DRAM */
	hprintf("Uncompressing Linux");
	gunzip();
	hprintf("done.\n");
}

#ifdef HAVE_LCD
extern unsigned char __splash_start;

void gunzip_splash(void)
{
	/* initialize variables for decompression */
	hermit_gunzip_init();
	hermit_input_ptr = &__splash_start;
	hermit_free_mem_ptr = WSIZE + (ulg) window;
	hermit_free_mem_ptr_end = hermit_free_mem_ptr + HERMIT_HEAP_SIZE;
	hermit_output_ptr = (uch*) VIDEO_MEM_BASE;

	/* initialize display */
	lcd_init();

	/* decompress splash screen to video RAM */
	hermit_progress = -1;
	gunzip();
	hermit_progress = 0;
}

#define WORDS_PER_ROW	(SCREENWIDTH / (PIXELS_PER_BYTE * 4))
#define TOPHEIGHT 9
#define BOXHEIGHT 3
#define GAPHEIGHT 4

void progress(void)
{
	unsigned *dst = (unsigned*) VIDEO_MEM_BASE;
	unsigned i, x, y;

	x = ((hermit_progress >> 5) << 1) + 1;
	y = hermit_progress & 31;

	dst += (TOPHEIGHT * WORDS_PER_ROW) + (y * WORDS_PER_ROW * BOXHEIGHT) +
		(y * WORDS_PER_ROW * GAPHEIGHT) + x;
	for (i=0; i<BOXHEIGHT; ++i) {
		*dst = 0;
		dst += WORDS_PER_ROW;
	}
	++hermit_progress;
}

#endif	/* HAVE_LCD */

