/*
 * Copyright (c) 2006 Atmark Techno, Inc.  All Rights Reserved.
 */

#include <ep93xx/ioregs.h>
#include <target/io.h>
#include "memmap.h"
#include "board.h"
#include <target/flash.h>

#define SECTOR_TABLE_SIZE_4MB 78
static const unsigned short flash_sector_table_4MB[] = {
  /* [ 0] */ 8,8,8,8,8,8,8,8,
  /* [ 8] */ 64,64,64,64,64,64,64,64,
  /* [16] */ 64,64,64,64,64,64,64,64,
  /* [24] */ 64,64,64,64,64,64,64,64,
  /* [32] */ 64,64,64,64,64,64,64,64,
  /* [40] */ 64,64,64,64,64,64,64,64,
  /* [48] */ 64,64,64,64,64,64,64,64,
  /* [56] */ 64,64,64,64,64,64,64,64,
  /* [64] */ 64,64,64,64,64,64,
  /* [70] */ 8,8,8,8,8,8,8,8,
  /* [78] */ 0
};

#define SECTOR_TABLE_SIZE_8MB 142
static const unsigned short flash_sector_table_8MB[] = {
  /* [  0] */ 8,8,8,8,8,8,8,8,
  /* [  8] */ 64,64,64,64,64,64,64,64,
  /* [ 16] */ 64,64,64,64,64,64,64,64,
  /* [ 24] */ 64,64,64,64,64,64,64,64,
  /* [ 32] */ 64,64,64,64,64,64,64,64,
  /* [ 40] */ 64,64,64,64,64,64,64,64,
  /* [ 48] */ 64,64,64,64,64,64,64,64,
  /* [ 56] */ 64,64,64,64,64,64,64,64,
  /* [ 64] */ 64,64,64,64,64,64,64,64,
  /* [ 72] */ 64,64,64,64,64,64,64,64,
  /* [ 80] */ 64,64,64,64,64,64,64,64,
  /* [ 88] */ 64,64,64,64,64,64,64,64,
  /* [ 96] */ 64,64,64,64,64,64,64,64,
  /* [104] */ 64,64,64,64,64,64,64,64,
  /* [112] */ 64,64,64,64,64,64,64,64,
  /* [120] */ 64,64,64,64,64,64,64,64,
  /* [128] */ 64,64,64,64,64,64,
  /* [134] */ 8,8,8,8,8,8,8,8,
  /* [142] */ 0
};

#define SECTOR_TABLE_SIZE  (board_info.flash == FLASH_8MiB ? \
                            SECTOR_TABLE_SIZE_8MB : SECTOR_TABLE_SIZE_4MB)
#define flash_sector_table (board_info.flash == FLASH_8MiB ? \
                            flash_sector_table_8MB : flash_sector_table_4MB)

unsigned int get_flash_sector_size(int sector){
	if(0 <= sector && sector <= SECTOR_TABLE_SIZE){
		return (unsigned int)(flash_sector_table[sector] * 1024);
	}
	return 0;
}

/*
 * The only vaguely tricky things about this memory map are:
 *
 * (1) It includes remapped memory addresses for the inverted mapping
 *     used by the CSC in boot mode, if we're compiling for that
 *     target.
 *
 * (2) Depicts flash block sizes as they are seen from outside the
 *     boot loader.  The flash parts are 16 bits wide (each spans 16
 *     bits of the data bus), so the blocks as seen by a flash
 *     programmer are twice data sheet size (of each 32-bit write, 16
 *     bits goes to each flash chip).
 */
static int memmap_cmdfunc(int argc, char *argv[])
{
	hprintf("%p:%p FLA %s bf:8K bl:%s\n"
		"%p:%p FLA %s bf:8K bl:%s\n"
		"%p:%p FLA %s bf:8K bl:%s\n"
		"%p:%p FLA %s bf:8K bl:%s\n"
		"%p:%p FLA %s bf:8K bl:%s\n"
		"%p:%p RAM onchip-sram\n"
		"%p:%p ROM boot-rom\n",

		FLASH_ALL_START, FLASH_ALL_START + FLASH_ALL_SIZE - 1,
		FLASH_ALL_NAME, FLASH_ALL_BLOCK,

		FLASH_BOOTLOADER_START,
		FLASH_BOOTLOADER_START + FLASH_BOOTLOADER_SIZE - 1,
		FLASH_BOOTLOADER_NAME, FLASH_BOOTLOADER_BLOCK,

		FLASH_KERNEL_START, FLASH_KERNEL_START + FLASH_KERNEL_SIZE - 1,
		FLASH_KERNEL_NAME, FLASH_KERNEL_BLOCK,

		FLASH_USERLAND_START,
		(board_info.flash != FLASH_8MiB) ?
		(FLASH_USERLAND_START + FLASH_USERLAND_SIZE - 1) :
		(FLASH_USERLAND_START + FLASH_USERLAND_SIZE8MB - 1),
		FLASH_USERLAND_NAME,
		(board_info.flash != FLASH_8MiB) ?
		FLASH_USERLAND_BLOCK : FLASH_USERLAND_BLOCK8MB,

		(board_info.flash != FLASH_8MiB) ?
		FLASH_CONFIG_START : FLASH_CONFIG_START8MB,

		(board_info.flash != FLASH_8MiB) ?
		(FLASH_CONFIG_START + FLASH_CONFIG_SIZE - 1) :
		(FLASH_CONFIG_START8MB + FLASH_CONFIG_SIZE - 1),
		FLASH_CONFIG_NAME, FLASH_CONFIG_BLOCK,

		SRAM_START, SRAM_START + SRAM_SIZE - 1,
		BOOTROM_START, BOOTROM_START + BOOTROM_SIZE - 1);

	if(board_info.sdram == SDRAM_32Bit) {
	  hprintf("%p:%p RAM dram-1\n"
		  "%p:%p RAM dram-2\n",
		  DRAM1_START, DRAM1_START + DRAM1_SIZE - 1,
		  DRAM2_START, DRAM2_START + DRAM2_SIZE - 1);
	}
	else {
	  hprintf("%p:%p RAM dram-1\n"
		  "%p:%p RAM dram-2\n"
		  "%p:%p RAM dram-3\n",
		  DRAM1_START16, DRAM1_START16 + DRAM1_SIZE16 - 1,
		  DRAM2_START16, DRAM2_START16 + DRAM2_SIZE16 - 1,
		  DRAM3_START16, DRAM3_START16 + DRAM3_SIZE16 - 1);
	}

	return 0;
}

const command_t memmap_command =
	{ "memmap", 0, "output memory map", &memmap_cmdfunc };
