﻿using System;
using System.Diagnostics;
using System.Net;
using System.Net.Mime;
using System.IO;
using System.Threading;
using NaGet.Tasks;

namespace NaGet.SubCommands.SubTask
{
	/// <summary>
	/// ダウンロードタスク
	/// </summary>
	public class DownloadSubTask : NaGetSubTask
	{
		/// <summary>
		/// アクセスURL
		/// </summary>
		protected Uri url;
		
		/// <summary>
		/// プロキシ
		/// </summary>
		protected IWebProxy proxy;
		
		/// <summary>
		/// 保存先
		/// </summary>
		protected string filepath;
		
		/// <summary>
		/// リクエストオブジェクト
		/// </summary>
		protected WebRequest request;
		
		/// <summary>
		/// レスポンスオブジェクト。応答がくるまではnullである。
		/// </summary>
		protected WebResponse response;
		
		/// <summary>
		/// ダウンロード時にHTTPヘッダなどから取得した本来のファイル名
		/// </summary>
		private string downloadedFileName = null;
		
		/// <summary>
		/// ダウンロード要求時のキャッシュレベル。デフォルトではキャッシュ無視
		/// </summary>
		public System.Net.Cache.RequestCacheLevel CacheLevel = System.Net.Cache.RequestCacheLevel.NoCacheNoStore;
		
		/// <summary>
		/// ダウンロード時に downloadedFileName に改名するか否か。
		/// </summary>
		protected bool enableChangeFileName = false;
		
		/// <summary>
		/// キャンセルが呼ばれたか否か。
		/// </summary>
		private bool cancelCalled = false;
		
		/// <summary>
		/// ダウンロード中のファイル名につける接尾辞
		/// </summary>
		private static readonly string PartialFilePostfix = ".part";
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="url">ダウンロード先URL</param>
		/// <param name="filepath">保存ファイルパス</param>
		/// <param name="proxy">プロキシ</param>
		public DownloadSubTask(Uri url, string filepath, IWebProxy proxy)
		{
			this.url = url;
			this.filepath = filepath;
			this.proxy = proxy;
			
			this.request = null;
			this.response = null;
			this.downloadedFileName = null;
		}
		
		public DownloadSubTask(string url, string filepath, IWebProxy proxy)
			: this(new Uri(url), filepath, proxy)
		{
		}
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="url">ダウンロード先URL</param>
		/// <param name="filepath">保存ファイルパス</param>
		public DownloadSubTask(Uri url, string filepath)
			: this(url, filepath, NaGet.Env.WebProxy)
		{
		}
		
		public DownloadSubTask(string url, string filepath)
			: this(new Uri(url), filepath, NaGet.Env.WebProxy)
		{
		}
	
		/// <summary>
		/// ダウンロード時にHTTPヘッダなどから取得した本来のファイル名
		/// </summary>
		public string DownloadedFileName {
			get { return downloadedFileName; }
		}
		
		/// <summary>
		/// 保存先ファイル名を本来のファイル名に変えるか。
		/// </summary>
		public bool EnableChangeFileName {
			get { return enableChangeFileName; }
			set { enableChangeFileName = value; }
		}
		
		/// <summary>
		/// 保存ファイル。
		/// </summary>
		public string Filepath {
			get { return filepath; }
		}
		
		/// <summary>
		/// キャンセル可能
		/// </summary>
		public override bool Cancelable {
			get { return !this.cancelCalled; }
		}
		
		/// <summary>
		/// ダウンロード処理をキャンセルする
		/// </summary>
		/// <returns>キャンセルに成功したときtrue</returns>
		public override bool Cancel()
		{
			if (! this.cancelCalled && ! this.Done) {
				this.cancelCalled = true;
				if (request != null) {
					try {
						request.Abort();
					} catch (WebException) {
					}
				}
				return true;
			} else {
				return false;
			}
		}
		
		public override void Run()
		{
			NotifyStarted();
			RaiseTaskSetEvent(TaskEventType.STARTED, string.Format("ダウンロード:{0}", this.url), 0);
			
			try {
				runBuildRequest();
				runHandleCancelTrigger();
				
				RaiseTaskSetEvent(TaskEventType.PING, string.Format("接続中...{0}", this.url.Host), -1);
				
				runAcquireResponse();
				runHandleCancelTrigger();
				
				runPrepareFile();
				runDownloadToFile();
				runHandleCancelTrigger();
				
				runPostprocess();
				
				RaiseTaskSetEvent(TaskEventType.COMPLETED, "ダウンロード終了", 100);
			} catch (System.Net.WebException e) {
				if (System.Net.NetworkInformation.NetworkInterface.GetIsNetworkAvailable()) {
					RaiseTaskSetEvent(TaskEventType.WARNING, "ネットワークに接続されていません。", -1);
				} else {
					RaiseTaskSetEvent(TaskEventType.WARNING, "ネットワークに接続できませんでした。ネットワークが切断されているか、ファイアウォールによって遮断された可能性があります。", -1);
				}
				throw new System.Net.WebException(e.Message, e);
			} finally {
				runReleaseResponse();
				
				if (cancelCalled) {
					NotifyCancelled();
				} else {
					NotifyCompleted();
				}
			}
		}
		
		/// <summary>
		/// キャンセルされたかを確認して、キャンセル要求があるのならば TaskCanceledException を投げる
		/// </summary>
		private void runHandleCancelTrigger()
		{
			if (this.cancelCalled) {
				throw new TaskCanceledException(string.Empty);
			}
		}
		
		/// <summary>
		/// requestの構築。
		/// </summary>
		private void runBuildRequest()
		{
			request = WebRequest.Create(url);
			request.Proxy = this.proxy;
			request.CachePolicy = new System.Net.Cache.RequestCachePolicy(CacheLevel);
			
			HttpWebRequest httpRequest = request as HttpWebRequest;
			if (httpRequest != null) {
				httpRequest.AutomaticDecompression = DecompressionMethods.Deflate | DecompressionMethods.GZip;
				httpRequest.UserAgent = NaGet.Env.UserAgentString;
			}
		}
		
		/// <summary>
		/// Responseを得る
		/// </summary>
		private void runAcquireResponse()
		{
			try {
				response = request.GetResponse();
			} catch (WebException e) {
				if (cancelCalled) { // キャンセル時
					throw new TaskCanceledException(string.Empty, e);
				} else {
					throw new WebException(e.Message, e);
				}
			}
		}
		
		/// <summary>
		/// ダウンロード先ファイル名の決定
		/// </summary>
		private void runPrepareFile()
		{
			try {
				downloadedFileName = getFileNameFromWebResponse(response);
			} catch (Exception) {
			}
			
			// パス名を変えるときは、HTTPヘッダから取得したファイル名に変更する。
			if (enableChangeFileName && (!string.IsNullOrEmpty(downloadedFileName))) {
				filepath = Path.Combine(Path.GetDirectoryName(filepath), downloadedFileName);
			}
			
			// ファイルが存在するとき削除
			if (File.Exists(filepath)) {
				File.Delete(filepath);
			}
			// 部分ファイルが存在するときも削除 TODO レジューム処理
			if (File.Exists(filepath + PartialFilePostfix)) {
				File.Delete(filepath + PartialFilePostfix);
			}
		}
		
		private void runDownloadToFile()
		{
			Stopwatch stopwatch = new Stopwatch();
			string partialFilepath = filepath + PartialFilePostfix;
			
			using (Stream stream = response.GetResponseStream() )
			using (FileStream fs = new FileStream(partialFilepath,
				                    FileMode.Create,
				                    FileAccess.Write) ) {
				
				try {
					File.SetAttributes(partialFilepath, FileAttributes.Hidden);
					long contentLength = response.ContentLength;
	
					stopwatch.Start();
					RaiseDownloadProgressTaskSetEvent(0, contentLength, 0);
	
					Timer timer = new Timer(new TimerCallback(
						delegate(object obj) {
							try {
								RaiseDownloadProgressTaskSetEvent(fs.Position, contentLength, stopwatch.ElapsedMilliseconds);
							} catch (ObjectDisposedException) {
							}
						}), null, 0, 1000);
					
					try {
						byte[] data = new byte[4096];
						int size = 0;
						while ((size = stream.Read(data,0,data.Length)) > 0) {
							fs.Write(data, 0, size);
							
							if (cancelCalled) {
								throw new TaskCanceledException(string.Empty);
							}
						}
					} finally {
						timer.Dispose();
					}
				} catch (IOException ex) {
					if (cancelCalled) {
						throw new TaskCanceledException(string.Empty);
					} else {
						throw new IOException(ex.Message, ex);
					}
				} finally {
					if (stopwatch != null) {
						stopwatch.Stop();
						stopwatch = null;
					}
				}
			}
			
			if (File.Exists(partialFilepath)) {
				File.Move(partialFilepath, filepath);
				File.SetAttributes(filepath, FileAttributes.Normal);
			}
		}
		
		private void runPostprocess()
		{
			// 更新日を補完
			if (File.Exists(filepath)) {
				HttpWebResponse	httpResponse = response as HttpWebResponse;
				FtpWebResponse	ftpResponse	 = response as FtpWebResponse;
				
				if (httpResponse != null) {
					File.SetLastWriteTime(filepath, httpResponse.LastModified);
				} else if (ftpResponse != null) {
					File.SetLastWriteTime(filepath, ftpResponse.LastModified);
				}
			}
		}
		
		/// <summary>
		/// responseの開放
		/// </summary>
		private void runReleaseResponse()
		{
			if (response != null) {
				response.Close();
			}
		}
		
		/// <summary>
		/// Webレスポンスからダウンロードしたファイルの名前を取得
		/// </summary>
		/// <remarks>Content-Dispositionヘッダから取得あるいはURLの末尾から推定します</remarks>
		/// <param name="response">レスポンスオブジェクト</param>
		/// <returns>取得したファイル名</returns>
		private static string getFileNameFromWebResponse(WebResponse response)
		{
			HttpWebResponse httpresp = response as HttpWebResponse;
			if (httpresp != null) {
				string contentDisposition = httpresp.Headers["Content-Disposition"];
				
				if (! string.IsNullOrEmpty(contentDisposition)) {
					try {
						ContentDisposition parser = new ContentDisposition(contentDisposition);
						if (! string.IsNullOrEmpty(parser.FileName)) {
							return parser.FileName;
						}
					} catch (FormatException) {
					}
				}
			}
			
			return NaGet.Utils.Url2filename(response.ResponseUri);
		}
		
		/// <summary>
		/// ダウンロード進捗メッセージを生成
		/// </summary>
		/// <param name="downloadsize">現在ダウンロード済みのサイズ</param>
		/// <param name="filesize">全体のファイルサイズ。不明なときはゼロを指定。</param>
		/// <param name="elapsedms">ダウンロード開始からの時間をms単位で。</param>
		protected virtual void RaiseDownloadProgressTaskSetEvent(long downloadsize, long filesize, long elapsedms)
		{
			float percent = -1;
			TimeSpan eta = new TimeSpan(0);
			long byteps = 0;
			
			// 進捗率の算出
			if (filesize > 0) {
				percent = 100 * ((float) downloadsize) / ((float) filesize);
			}
			
			// スループット・残り時間の算出
			if (elapsedms > 0) {
				byteps = 1000 * downloadsize / elapsedms;
				if (filesize > 0 && byteps > 0) {
					eta = TimeSpan.FromSeconds((filesize - downloadsize) / byteps);
				}
			}
			
			System.Text.StringBuilder msgbuilder = new System.Text.StringBuilder();
			msgbuilder.AppendFormat("{0} bytes", downloadsize);
			if (percent > 0) {
				msgbuilder.AppendFormat(" ({0:F2}%)", percent);
			}
			if (eta.Ticks > 0) {
				msgbuilder.AppendFormat(" ETA {0}", eta);
			}
			if (byteps > 0) {
				msgbuilder.AppendFormat(" ({0}/s)", NaGet.Utils.FormatSize(byteps));
			}
			
			RaiseTaskSetEvent(TaskEventType.PING, msgbuilder.ToString(), percent);
		}
	}
}
