! Copyright (c) 2012-2013 Manuel Hasert <m.hasert@grs-sim.de>
! Copyright (c) 2012-2020 Kannan Masilamani <kannan.masilamani@uni-siegen.de>
! Copyright (c) 2012-2013 Simon Zimny <s.zimny@grs-sim.de>
! Copyright (c) 2013-2016 Jiaxing Qi <jiaxing.qi@uni-siegen.de>
! Copyright (c) 2016 Tobias Schneider <tobias1.schneider@student.uni-siegen.de>
! Copyright (c) 2018 Raphael Haupt <raphael.haupt@uni-siegen.de>
! Copyright (c) 2019-2020 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2019 Jana Gericke <jana.gericke@uni-siegen.de>
!
! Redistribution and use in source and binary forms, with or without
! modification, are permitted provided that the following conditions are met:
!
! 1. Redistributions of source code must retain the above copyright notice,
! this list of conditions and the following disclaimer.
!
! 2. Redistributions in binary form must reproduce the above copyright notice,
! this list of conditions and the following disclaimer in the documentation
! and/or other materials provided with the distribution.
!
! THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY OF SIEGEN “AS IS” AND ANY EXPRESS
! OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
! OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
! IN NO EVENT SHALL UNIVERSITY OF SIEGEN OR CONTRIBUTORS BE LIABLE FOR ANY
! DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
! (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
! LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
! ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
! (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
! SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
! ************************************************************************** !
!> author: Jiaxing Qi
!! author: Manuel Hasert
!! Linear Interpolation of flow quantities between different grid levels
!!
!! # Interpolation
!!
!! The routines defined here, fill up the ghost elements with valid data.
!! Ghost elements are employed at grid level interfaces to provide valid
!! pdf values to the neighboring fluid elements. This way, the solvers can
!! act on elements of the same size only, treating the levels successively.
!! Target elements are the ghost elements, which have to be filled with
!! valid values.
!! Source elements are the fluid elements from other levels, from where to
!! take the input values for the interpolation.
!! The target ghost elements on the target level have corresponding source
!! fluid elements on the source level.
!!
!! [[tem_topology_module]] For a detailed description of the grid
!!
!! # Workflow
!!
!! Each interpolation routine acts on a list of ghost elements.
!! This list contains pointers to the position in the total list.
!! For each of these ghost elements, the source elements are identified.
!! Before that, the sourceLevel is identified. However, the code is restricted
!! to work with a level jump of only one level, so the sourceLevel is:\n
!! for fillMyGhostsFromFiner: sourceLevel = targetLevel + 1
!! for fillFinerGhostsFromMe: sourceLevel = targetLevel - 1
!!
!! For an overview over implemented interpolation methods, see
!! [Interpolation methods](../page/features/intp_methods.html)
!!
?? include 'header/lbm_macros.inc'
?? include 'treelm/source/logMacros.inc'
module mus_interpolate_linear_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use env_module,            only: rk
  use tem_aux_module,          only: tem_abort
  use tem_element_module,    only: eT_GhostFromCoarser
  use tem_param_module,      only: cs2inv, cs2, PI, div1_2, div1_9, div4_9,&
    &                              div1_36, div2_3, c_x, c_y, c_z,&
    &                              rho0, rho0Inv
  use tem_comm_env_module,   only: tem_comm_env_type
  use tem_debug_module,      only: dbgUnit
  use tem_construction_module, only: tem_levelDesc_type
  use tem_stencil_module,      only: tem_stencilHeader_type
  use tem_logging_module,      only: logUnit
  use tem_matrix_module,       only: tem_matrix_type
  use tem_varSys_module,       only: tem_varSys_type
  use tem_time_module,         only: tem_time_type

?? IF (loglvl >= 1) THEN
  use env_module,            only: long_k
  use tem_topology_module,   only: tem_coordOfID
  use tem_param_module,      only: PI
?? ENDIF

  ! include musubi modules
?? IF( .not. SOA) then
  use mus_interpolate_debug_module,  only: TGV_2D
?? ENDIF
  use mus_scheme_layout_module,      only: mus_scheme_layout_type
  use mus_derVarPos_module,          only: mus_derVarPos_type
  use mus_interpolate_header_module, only: mus_interpolation_method_type
  use mus_physics_module,            only: mus_physics_type
  use mus_interpolate_tools_module,  only: mus_intp_convertMomToPDF3D,        &
    &                                      mus_intp_convertMomToPDF3D_incomp, &
    &                                      mus_intp_convertMomToPDF2D,        &
    &                                      mus_intp_convertMomToPDF2D_incomp, &
    &                                      mus_intp_getMoments
  use mus_field_prop_module,         only: mus_field_prop_type
  use mus_fluid_module,              only: mus_fluid_type
  use mus_relaxationParam_module,    only: mus_calcOmegaFromVisc

  implicit none

  private

  public :: fillArbiFinerGhostsFromMe_linear
  public :: fillArbiFinerGhostsFromMe_linear2D
  public :: fillFinerGhostsFromMe_linear
  public :: fillFinerGhostsFromMe_linearLES
  public :: fillFinerGhostsFromMe_linearIncomp
  public :: fillFinerGhostsFromMe_linearIncompLES
  public :: fillFinerGhostsFromMe_linear2D
  public :: fillFinerGhostsFromMe_linear2DIncomp

contains

! **************************************************************************** !
  !> Interpolate auxiliary field from coarse source to fine target
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine_arbitraryVal]] in intp/[[mus_interpolate_header_module]].f90
  !! in order to be callable via
  !! [[mus_interpolation_method_type:do_intpArbiVal]] function pointer.
  subroutine fillArbiFinerGhostsFromMe_linear( method, tLevelDesc, level,    &
    &                                          stencil, sVal, snSize, tVal,  &
    &                                          tnSize, nTargets, targetList, &
    &                                          nScalars                      )
    ! ------------------------------------------------------------------ !
    class(mus_interpolation_method_type), intent(in) :: method

    !> my refinement level
    integer, intent(in) :: level

    !> stencil header
    type(tem_stencilHeader_type), intent(in) :: stencil

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sVal(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tVal(:)
    integer, intent(in) :: tnSize

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> number of scalars to interpolate
    integer, intent(in) :: nScalars
    ! ------------------------------------------------------------------ !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    integer :: posInIntpMatLSF
    real(kind=rk) :: tArbi(nScalars)
    real(kind=rk) :: sArbi(nScalars, stencil%QQ)  ! temp source ArbiField
    ! --------------------------------------------------------------------------
    sourceLevel = level
    targetLevel = level + 1

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get souce auxilary variables
        sArbi(:, iSourceElem) = sVal( (sourceElem-1)*nScalars+1 &
          &                           : sourceElem*nScalars       )

      end do  ! iSourceElem

      ! interpolate all auxiliary variables by linear interpolation
      tArbi(1:nScalars) = mus_interpolate_linear3D_leastSq(       &
        &   srcMom      = sArbi(1:nScalars, 1:nSourceElems),      &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = nScalars                               )

      ! write interpolated value
      tVal((targetElem-1)*nScalars+1 : targetElem*nScalars) &
          & = tArbi
    enddo

  end subroutine fillArbiFinerGhostsFromMe_linear
! **************************************************************************** !


! **************************************************************************** !
  !> Interpolate auxiliary field from coarse source to fine target
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine_arbitraryVal]] in intp/[[mus_interpolate_header_module]].f90
  !! in order to be callable via
  !! [[mus_interpolation_method_type:do_intpArbiVal]] function pointer.
  subroutine fillArbiFinerGhostsFromMe_linear2D( method, tLevelDesc, level,    &
    &                                            stencil, sVal, snSize, tVal,  &
    &                                            tnSize, nTargets, targetList, &
    &                                            nScalars                      )
    ! ------------------------------------------------------------------ !
    class(mus_interpolation_method_type), intent(in) :: method

    !> my refinement level
    integer, intent(in) :: level

    !> stencil header
    type(tem_stencilHeader_type), intent(in) :: stencil

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sVal(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tVal(:)
    integer, intent(in) :: tnSize

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> number of scalars to interpolate
    integer, intent(in) :: nScalars
    ! ------------------------------------------------------------------ !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    integer :: posInIntpMatLSF
    real(kind=rk) :: tArbi(nScalars)
    real(kind=rk) :: sArbi(nScalars, stencil%QQ)  ! temp source ArbiField
    ! --------------------------------------------------------------------------
    sourceLevel = level
    targetLevel = level + 1

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get souce auxilary variables
        sArbi(:, iSourceElem) = sVal( (sourceElem-1)*nScalars+1 &
          &                           : sourceElem*nScalars       )

      end do  ! iSourceElem

      ! interpolate all auxiliary variables by linear interpolation
      tArbi(1:nScalars) = mus_interpolate_linear2D_leastSq(       &
        &   srcMom      = sArbi(1:nScalars, 1:nSourceElems),      &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = nScalars                               )

      ! write interpolated value
      tVal((targetElem-1)*nScalars+1 : targetElem*nScalars) &
          & = tArbi
    enddo

  end subroutine fillArbiFinerGhostsFromMe_linear2D
! **************************************************************************** !


  ! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear least square fit interpolation.
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (den, vel, tau)
  !!    (10 moments for 3D and 6 moments for 2D)
  !! 3. calculate fEq and use it to calculate high order moments
  !! 4. convert moments to PDF
  !! This routine is used by 3D acoustic linear interpolation.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linear( method, fieldProp, tLevelDesc, &
    &                                      level, sState, snSize, tState, &
    &                                      tnSize, tAuxField, layout,     &
    &                                      nTargets, targetList, physics, &
    &                                      time, varSys, derVarPos        )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc, invRho
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do

      ! interpolate moments
      tMom(1:QQ) = mus_interpolate_linear3D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ )

      ! store interpolated target auxField
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! total viscosity on source element
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(0.5_rk * tVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linear
! ****************************************************************************** !


  ! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear least square fit interpolation.
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (den, vel, tau)
  !!    (10 moments for 3D and 6 moments for 2D)
  !! 3. calculate fEq and use it to calculate high order moments
  !! 4. convert moments to PDF
  !! This routine is used by 3D acoustic linear interpolation.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linearLES( method, fieldProp, tLevelDesc, &
    &                                         level, sState, snSize, tState, &
    &                                         tnSize, tAuxField, layout,     &
    &                                         nTargets, targetList, physics, &
    &                                         time, varSys, derVarPos        )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, sVisc, tVisc, invRho
    real(kind=rk) :: sTurbVisc(layout%fStencil%QQ), tTurbVisc(1)
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

        ! get turbulent viscosity
        sTurbVisc(iSourceElem) = fluid%turbulence%dataOnLvl(sourceLevel) &
          &                                       %visc(sourceElem)
      end do

      ! interpolate moments
      tMom(1:QQ) = mus_interpolate_linear3D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ )

      ! store interpolated target auxField
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho

      !interpolate turbulent viscosity to target element
      tTurbVisc = mus_interpolate_linear3D_leastSq(                   &
        &   srcMom      = sTurbVisc(1:nSourceElems),                &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = 1                                         )

      ! scale interpolated turbulent viscosity to target element
      fluid%turbulence%dataOnLvl(targetLevel)%visc(targetElem) &
        & = fluid%turbulence%fac_c2f*tTurbVisc(1)

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! total viscosity on source element
      sVisc = 0.5_rk * tVisc + tTurbVisc(1)
      ! total viscosity on target element
      tVisc = tVisc + fluid%turbulence%fac_c2f*tTurbVisc(1)

      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(sVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linearLES
! ****************************************************************************** !


  ! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear interpolation:
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (10 moms for 3D; 6 moms for 2D)
  !! 3. Store target auxilary field
  !! 4. Get viscosity on target element and compute source and target omega
  !! 5. Get nonEq scaling factor depeding on scheme layout and relaxation
  !! 6. Convert moments to PDF
  !! 7. Calculate target: Eq + Scale * nonEquilibrium
  !! This routine is used by 3D acoustic linear interpolation for
  !! incompressible model.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linearIncomp( method, fieldProp,         &
    & tLevelDesc, level, sState, snSize, tState, tnSize, tAuxField, layout, &
    & nTargets, targetList, physics, time, varSys, derVarPos                )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do

      !interpolate moments to target element
      tMom(1:QQ) = mus_interpolate_linear3D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ )

      ! store interpolated target auxField
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*rho0Inv
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*rho0Inv
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*rho0Inv

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(0.5_rk * tVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D_incomp(                              &
        &                               moments          = tMom,             &
        &                               nonEqScalingFacs = nonEqScalingFacs, &
        &                               layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linearIncomp
! ****************************************************************************** !


  ! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear interpolation.
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (10 moms for 3D; 6 moms for 2D)
  !! 3. Store target auxilary field
  !! 4. Compute viscosity on target element and compute source and target omega
  !! 5. Get nonEq scaling factor depeding on scheme layout and relaxation
  !! 6. Calculate Equilibrium and nonEquilibrium
  !! 7. calculate target: Eq + Scale * nonEquilibrium
  !! This routine is used by 3D acoustic linear interpolation for
  !! incompressible LES model.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linearIncompLES( method, fieldProp,      &
    & tLevelDesc, level, sState, snSize, tState, tnSize, tAuxField, layout, &
    & nTargets, targetList, physics, time, varSys, derVarPos                )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, sVisc, tVisc
    real(kind=rk) :: sTurbVisc(layout%fStencil%QQ), tTurbVisc(1)
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

        ! get turbulent viscosity
        sTurbVisc(iSourceElem) = fluid%turbulence%dataOnLvl(sourceLevel) &
          &                                       %visc(sourceElem)
      end do


      !interpolate moments to target element
      tMom(1:QQ) = mus_interpolate_linear3D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ )

      ! store interpolated target auxField
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*rho0Inv
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*rho0Inv
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*rho0Inv

      !interpolate turbulent viscosity to target element
      tTurbVisc = mus_interpolate_linear3D_leastSq(                   &
        &   srcMom      = sTurbVisc(1:nSourceElems),                &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = 1                                         )

      ! scale interpolated turbulent viscosity to target element
      fluid%turbulence%dataOnLvl(targetLevel)%visc(targetElem) &
        & = fluid%turbulence%fac_c2f*tTurbVisc(1)

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! total viscosity on source element
      sVisc = 0.5_rk * tVisc + tTurbVisc(1)
      ! total viscosity on target element
      tVisc = tVisc + fluid%turbulence%fac_c2f*tTurbVisc(1)

      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(sVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D_incomp(                              &
        &                               moments          = tMom,             &
        &                               nonEqScalingFacs = nonEqScalingFacs, &
        &                               layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linearIncompLES
! ****************************************************************************** !


! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear interpolation.
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (den, vel, tau)
  !!    (10 moments for 3D and 6 moments for 2D)
  !! 3. calculate fEq and use it to calculate high order moments
  !! 4. convert moments to PDF
  !! This routine is used by 3D acoustic linear interpolation.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linear2D( method, fieldProp, tLevelDesc, &
    &                                        level, sState, snSize, tState, &
    &                                        tnSize, tAuxField, layout,     &
    &                                        nTargets, targetList, physics, &
    &                                        time, varSys, derVarPos        )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc, invRho
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1


    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do

      tMom(1:QQ) = mus_interpolate_linear2D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ                                        )

      ! store interpolated target auxField
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho
      tAuxField(tOffset+vel_pos(3)) =  0.0_rk

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! total viscosity on source element
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(0.5_rk * tVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linear2D
! ****************************************************************************** !


  ! ************************************************************************** !
  !> Fill fine ghost from coarse fluid by linear interpolation.
  !! 1. Compute moments for all source elements, save in momBuf
  !! 2. For each target, interpolate moments (den, vel, tau)
  !!    (10 moments for 3D and 6 moments for 2D)
  !! 3. calculate fEq and use it to calculate high order moments
  !! 4. convert moments to PDF
  !! This routine is used by 3D acoustic linear interpolation for
  !! incompressible model.
  !!
  !! This subroutine's interface must match the abstract interface definition
  !! [[intpRoutine]] in intp/[[mus_interpolate_header_module]].f90 in order to
  !! be callable via [[mus_interpolation_method_type:do_intp]] function pointer.
  subroutine fillFinerGhostsFromMe_linear2DIncomp( method, fieldProp,       &
    & tLevelDesc, level, sState, snSize, tState, tnSize, tAuxField, layout, &
    & nTargets, targetList, physics, time, varSys, derVarPos                )
    ! -------------------------------------------------------------------- !
    class(mus_interpolation_method_type), intent(in) :: method

    !> Array of field properties (fluid or species)
    type(mus_field_prop_type), target, intent(in) :: fieldProp(:)

    !> level descriptor on target level
    type( tem_levelDesc_type ), intent(in) :: tLevelDesc

    !> my refinement level
    integer, intent(in) :: level

    !> State vector of SOURCE FLUID elements
    real(kind=rk), intent(in) :: sState(:)
    ! integer, intent(in) :: sNeigh(:)
    integer, intent(in) :: snSize

    !> State vector of TARGET GHOST elements
    real(kind=rk), intent(inout) :: tState(:)
    ! integer, intent(in) :: tNeigh(:)
    integer, intent(in) :: tnSize

    !> AuxField variable to fill on target GHOST elements
    real(kind=rk), intent(inout) :: tAuxField(:)

    !> the layout used
    type( mus_scheme_layout_type ), intent(in) :: layout

    !> List of target elements ( their position in depSource list )
    integer, intent(in) :: nTargets
    integer, intent(in) :: targetList(nTargets)

    !> physics type to convert lattice to physics SI unit and vice versa
    !! @todo: This can be replaced by scale factor by level
    type( mus_physics_type ), intent(in) :: physics

    !> time required to compute viscosity on target element barycenter
    type(tem_time_type), intent(in) :: time

    !> scheme variable system
    type( tem_varSys_type ), intent(in) :: varSys

    !> position of all derive variable in varSys for all fields
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------- !
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iVal, iElem, indElem
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ ) ! pdf to reconstruct from
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: QQ
    integer :: posInIntpMatLSF
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------------
!write(dbgUnit(1),*) 'Entering fillFiner Linear'
    fluid => fieldProp(1)%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals
      posInIntpMatLSF = tLevelDesc%depFromCoarser( iElem )%posInIntpMatLSF
!write(dbgUnit(1),*) 'Target iElem ', indElem, ' ID ', tLevelDesc%total(targetElem)

      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do
!write(dbgUnit(1),*) 'sourceRho-rho0: ', srcMom(1,1:nSourceElems)

      tMom(1:QQ) = mus_interpolate_linear2D_leastSq(                &
        &   srcMom      = srcMom(1:QQ, 1:nSourceElems),             &
        &   targetCoord = tLevelDesc%depFromCoarser( iElem )%coord, &
        &   LSFmat      = method%intpMat_forLSF%matArray            &
        &                       %val(posInIntpMatLSF),              &
        &   nSources    = nSourceElems,                             &
        &   nVals       = QQ )

!write(dbgUnit(1),*) 'before scaling tMom: ', tMom(1:QQ)

      ! store interpolated target auxField
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*rho0Inv
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*rho0Inv
      tAuxField(tOffset+vel_pos(3)) = 0.0_rk

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity:
      ! v^s_f = 2 v^s_c
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(0.5_rk * tVisc)
      tOmegaKine = mus_calcOmegaFromVisc(tVisc)

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(           &
        & omegaKine_SRC = sOmegaKine,                      &
        & omegaKine_TGT = tOmegaKine,                      &
        & omegaBulk_SRC = fluid%omegaBulkLvl(sourceLevel), &
        & omegaBulk_TGT = fluid%omegaBulkLvl(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                  )


      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D_incomp(                               &
        &                                moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iVal = 1, QQ
        tState( ?IDX?( iVal, targetElem, nScalars, tnSize )) = tPDF(iVal)
      enddo
!write(dbgUnit(1),*) 'targetRho-rho0 ', rho-rho0
!write(dbgUnit(1),*) 'tMom: ', tMom(1:6)
!write(dbgUnit(1),*) 'f: ', f

    end do ! indElem

  end subroutine fillFinerGhostsFromMe_linear2DIncomp
! ****************************************************************************** !


! ****************************************************************************** !
  !> Bilinear interpolation for a vector quantity phi
  !!
  ! real(kind=rk), dimension(6,9),parameter  :: intpMaxtrix =  &
  !           reshape((/ &
  !   &  div2_9,  -div1_6,    0.0d0,    div1_6,   -div1_3,   0.0d0, &
  !   &  div2_9,    0.0d0,  -div1_6,   -div1_3,    div1_6,   0.0d0, &
  !   &  div2_9,   div1_6,    0.0d0,    div1_6,   -div1_3,   0.0d0, &
  !   &  div2_9,    0.0d0,   div1_6,   -div1_3,    div1_6,   0.0d0, &
  !   & -div1_9,  -div1_6,  -div1_6,    div1_6,    div1_6,  0.25d0, &
  !   & -div1_9,  -div1_6,   div1_6,    div1_6,    div1_6, -0.25d0, &
  !   & -div1_9,   div1_6,  -div1_6,    div1_6,    div1_6, -0.25d0, &
  !   & -div1_9,   div1_6,   div1_6,    div1_6,    div1_6,  0.25d0, &
  !   &  div5_9,    0.0d0,    0.0d0,   -div1_3,   -div1_3,   0.0d0  &
  !  /),(/6,9/))
  pure function mus_interpolate_linear2D_leastSq( srcMom, targetCoord, LSFmat, &
    &                                           nSources, nVals ) result(phi)
    ! ---------------------------------------------------------------------------
    !> number of quantities to interpolation
    integer, intent(in) :: nVals
    !> Number of source elements
    integer, intent(in) :: nSources
    !> source values of the momentum on the square corners
    real(kind=rk), intent(in) :: srcMom(nVals, nSources)
    !> interpolation location within the square
    real(kind=rk), intent(in) :: targetCoord(3)
    !> matrix for least square fit
    type(tem_matrix_type), intent(in) :: LSFmat
    !> interpolated value
    real(kind=rk) :: phi( nVals )
    ! ---------------------------------------------------------------------------
    real(kind=rk) :: a(3) ! Coefficients
    integer :: iVal
    ! ---------------------------------------------------------------------------
    ! We extract momentum information completely on the view of the source
    ! coordinate system
    ! Set the right hand side of the equation system
    ! Solve the problem, where b = rhs, x = coefficients
    ! A*x = b
    ! overdetermined, solve the least Square fit problem
    ! (A^T)A*x = (A^T)b
    ! x = ((A^T)A)^-1*(A^T)b
    ! Solve linear system of equation with inverted matrix
    do iVal = 1, nVals
      a(:) = matmul( LSFmat%A, srcMom(iVal, :) )

      ! Evaluate the bubble function with the above calculated coefficients
      ! m_pi(x) = a1+a2*xcoord+a3*ycoord+a4*xcoord*xcoord+a5*ycoord*ycoord+a6*xcoord*ycoord;
      phi( iVal ) =   a( 1) &
        &           + a( 2)*targetCoord(c_x)                  &
        &           + a( 3)*targetCoord(c_y)
    end do

  end function mus_interpolate_linear2D_leastSq
! ****************************************************************************** !


! ****************************************************************************** !
  !> Trilinear interpolation for a vector quantity phi
  !! Each phi corresponds to each moment
  ! matrixInv%A(1:10,1:19) =
  !      1     2     3     4     5     6     7     8     9    10    11    12    13    14    15    16    17    18    19
  ! [ 4/21  4/21  4/21  4/21  4/21  4/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21 -1/21   3/7]
  ! [-1/10     0     0  1/10     0     0     0     0     0     0 -1/10  1/10 -1/10  1/10 -1/10 -1/10  1/10  1/10     0]
  ! [    0 -1/10     0     0  1/10     0 -1/10 -1/10  1/10  1/10     0     0     0     0 -1/10  1/10 -1/10  1/10     0]
  ! [    0     0 -1/10     0     0  1/10 -1/10  1/10 -1/10  1/10 -1/10 -1/10  1/10  1/10     0     0     0     0     0]
  ! [ 1/42  -1/7  -1/7  1/42  -1/7  -1/7 -1/21 -1/21 -1/21 -1/21  5/42  5/42  5/42  5/42  5/42  5/42  5/42  5/42 -5/21]
  ! [ -1/7  1/42  -1/7  -1/7  1/42  -1/7  5/42  5/42  5/42  5/42 -1/21 -1/21 -1/21 -1/21  5/42  5/42  5/42  5/42 -5/21]
  ! [ -1/7  -1/7  1/42  -1/7  -1/7  1/42  5/42  5/42  5/42  5/42  5/42  5/42  5/42  5/42 -1/21 -1/21 -1/21 -1/21 -5/21]
  ! [    0     0     0     0     0     0     0     0     0     0     0     0     0     0   1/4  -1/4  -1/4   1/4     0]
  ! [    0     0     0     0     0     0   1/4  -1/4  -1/4   1/4     0     0     0     0     0     0     0     0     0]
  ! [    0     0     0     0     0     0     0     0     0     0   1/4  -1/4  -1/4   1/4     0     0     0     0     0]
  !
! ****************************************************************************** !
  pure function mus_interpolate_linear3D_leastSq( srcMom, targetCoord, LSFmat, &
    &                                           nSources, nVals ) result(phi)
    ! ---------------------------------------------------------------------------
    !> number of quantities to interpolation
    integer, intent(in) :: nVals
    !> Number of source elements
    integer, intent(in) :: nSources
    !> source values of the momentum on the square corners
    real(kind=rk), intent(in) :: srcMom(nVals, nSources)
    !> interpolation location within the square
    real(kind=rk), intent(in) :: targetCoord(3)
    !> matrix for least square fit
    type(tem_matrix_type), intent(in) :: LSFmat
    !> interpolated value
    real(kind=rk) :: phi( nVals )
    ! ---------------------------------------------------------------------------
    real(kind=rk) :: a(4) ! Coefficients
    integer :: iVal
    ! ---------------------------------------------------------------------------
    ! We extract momentum information completely on the view of the source
    ! coordinate system
    ! Set the right hand side of the equation system
    ! Solve the problem, where b = rhs, x = coefficients
    ! A*x = b
    ! overdetermined, solve the least Square fit problem
    ! (A^T)A*x = (A^T)b
    ! x = ((A^T)A)^-1*(A^T)b
    ! Solve linear system of equation with inverted matrix
    do iVal = 1, nVals
      a(:) = matmul( LSFmat%A, srcMom(iVal, :) )

      ! Evaluate the bubble function with the above calculated coefficients
      ! m_pi(x) = a1+a2*xcoord+a3*ycoord+a4*xcoord*xcoord+a5*ycoord*ycoord+a6*xcoord*ycoord;
      phi( iVal ) =   a( 1) &
        &           + a( 2)*targetCoord(c_x)                  &
        &           + a( 3)*targetCoord(c_y)                  &
        &           + a( 4)*targetCoord(c_z)
    enddo

  end function mus_interpolate_linear3D_leastSq
! ****************************************************************************** !

end module mus_interpolate_linear_module
! ****************************************************************************** !
